//
// Copyright (C) 2020 Ross Martin
//
// Versions
//
// 2018.03.21  Coding begun
// 2020.05.07  Modified for capture ram

`undef ra
`undef rb
`define w assign
`define ra always@(posedge clk_a)
`define rb always@(posedge clk_b)


(* DONT_TOUCH = "YES" *)
module capture_ram
  #(
    parameter  RAM_ITEM_WIDTH       = 72,
    parameter  RAM_ITEM_NUMBER      = 1,
    parameter  RAM_LENGTH           = 512,
    parameter  ADDR_WIDTH           = $clog2(RAM_LENGTH),
    parameter  RAM_WIDTH            = RAM_ITEM_WIDTH*RAM_ITEM_NUMBER,
    parameter  USE_ULTRA_RAM        = (RAM_LENGTH*RAM_WIDTH > 50*32768),  // More than 50 BRAMs, go to ultra ram
    parameter  USE_BRAM             = (RAM_LENGTH >= 65),
    parameter  USE_DISTRIBUTED_RAM  = !(USE_ULTRA_RAM||USE_BRAM),
    parameter  DELAY                = USE_ULTRA_RAM ? 5 : 3,
    parameter  NUM_SYNC             = 1
   )
   (
    input                       reset_write,
    input                       clk_write,
    input                       write_enable_i,
    input [ADDR_WIDTH-1:0]      write_addr_i,
    input [RAM_ITEM_WIDTH-1:0]  write_data_i [0:RAM_ITEM_NUMBER-1],
    input                       reset_read,
    input                       clk_read,
    input [ADDR_WIDTH-1:0]      read_addr_i,
    output [RAM_ITEM_WIDTH-1:0] read_data_ro [0:RAM_ITEM_NUMBER-1],
    input  [NUM_SYNC-1:0]       in_sync_i,  // The sync is delayed by the same delay as the read delay, so read timing is known
    output [NUM_SYNC-1:0]       out_sync_o
   );

   capture_ram_inner
     #(
       .RAM_ITEM_WIDTH       ( RAM_ITEM_WIDTH      ), 
       .RAM_ITEM_NUMBER      ( RAM_ITEM_NUMBER     ), 
       .RAM_LENGTH           ( RAM_LENGTH          ),
       .USE_ULTRA_RAM        ( USE_ULTRA_RAM       ),
       .USE_DISTRIBUTED_RAM  ( USE_DISTRIBUTED_RAM ),
       .USE_BRAM             ( USE_BRAM            )
       )
   bram
     (
      .clk_a             ( clk_write          ),
      .clk_b             ( clk_read           ),
      .a_write_enable_i  ( write_enable_i     ),
      .a_write_addr_i    ( write_addr_i       ),
      .a_write_data_i    ( write_data_i       ),
      .b_read_addr_i     ( read_addr_i        ),
      .b_read_data_ro    ( read_data_ro       )
      ); 
   
   delay_with_reset #(.DELAY(DELAY), .BIT_WIDTH(NUM_SYNC)) sync_delay_instance ( .reset(reset_read), .clk(clk_read), .in_i(in_sync_i), .out_o(out_sync_o));
   
endmodule // capture_ram



(* DONT_TOUCH = "YES" *)
module capture_ram_inner
  #(
    parameter  RAM_ITEM_WIDTH       = 72,
    parameter  RAM_ITEM_NUMBER      = 1,
    parameter  RAM_WIDTH            = RAM_ITEM_WIDTH*RAM_ITEM_NUMBER,
    parameter  RAM_LENGTH           = 512,
    parameter  ADDR_WIDTH           = $clog2(RAM_LENGTH),
    parameter  USE_ULTRA_RAM        = 0,
    parameter  USE_DISTRIBUTED_RAM  = 0,
    parameter  USE_BRAM             = 0
   )
   (
    input                       clk_a,
    input                       clk_b,
    input                       a_write_enable_i,
    input [ADDR_WIDTH-1:0]      a_write_addr_i,
    input [RAM_ITEM_WIDTH-1:0]  a_write_data_i [0:RAM_ITEM_NUMBER-1],
    input [ADDR_WIDTH-1:0]      b_read_addr_i,
    output [RAM_ITEM_WIDTH-1:0] b_read_data_ro [0:RAM_ITEM_NUMBER-1]
   );

   reg                         a_write_enable_r;
   reg [ADDR_WIDTH-1:0]        a_write_addr_r;
   reg [RAM_WIDTH-1:0]         a_write_data_r;
   reg                         a_write_enable_rr;
   reg [ADDR_WIDTH-1:0]        a_write_addr_rr;
   reg [RAM_WIDTH-1:0]         a_write_data_rr;
   reg                         a_write_enable_rrr;
   (* max_fanout = 5 *) reg [ADDR_WIDTH-1:0]        a_write_addr_rrr;
   reg [RAM_WIDTH-1:0]         a_write_data_rrr;
   reg [ADDR_WIDTH-1:0]        b_read_addr_r;
   reg [RAM_WIDTH-1:0]         b_read_data_r;
   reg [RAM_WIDTH-1:0]         b_read_data_rr;
                               
   `ra a_write_enable_r    <=  a_write_enable_i;
   `ra a_write_addr_r      <=  a_write_addr_i;

   `ra a_write_enable_rr   <=  a_write_enable_r;
   `ra a_write_addr_rr     <=  a_write_addr_r;
   `ra a_write_data_rr     <=  a_write_data_r;

   `ra a_write_enable_rrr  <=  a_write_enable_rr;
   `ra a_write_addr_rrr    <=  a_write_addr_rr;
   `ra a_write_data_rrr    <=  a_write_data_rr;

   `rb b_read_addr_r       <=  b_read_addr_i;


generate

   genvar                      item;

   for(item=0; item<RAM_ITEM_NUMBER; item = item + 1)
     `ra a_write_data_r[item*RAM_ITEM_WIDTH +: RAM_ITEM_WIDTH] <=  a_write_data_i[item];

   
   if ( USE_ULTRA_RAM )
     begin
	(* RAM_STYLE = "ultra" *)
	(* CASCADE_HEIGHT = 0 *)
	reg [RAM_WIDTH-1:0] 	   ram_t[0:RAM_LENGTH-1];

	`ra if(a_write_enable_rrr) ram_t[a_write_addr_rrr] <= a_write_data_rrr;

	`rb b_read_data_r <= ram_t[b_read_addr_r];

        reg [RAM_WIDTH-1:0]        b_read_data_rx;
        reg [RAM_WIDTH-1:0]        b_read_data_ry;
        
	`rb b_read_data_rx  <= b_read_data_r;
	`rb b_read_data_ry  <= b_read_data_rx;
	`rb b_read_data_rr  <= b_read_data_ry;

     end
   else if ( USE_DISTRIBUTED_RAM )
     begin
	(* RAM_STYLE = "distributed" *)
	reg [RAM_WIDTH-1:0] 	   ram_t[0:RAM_LENGTH-1];

        // Very little delay for case where memory is small and in distributed ram
	`ra if(a_write_enable_r) ram_t[a_write_addr_r] <= a_write_data_r;   

	`rb b_read_data_r <= ram_t[b_read_addr_r];
	`rb b_read_data_rr  <= b_read_data_r;

     end
   else if ( USE_BRAM )
     begin
	(* RAM_STYLE = "block" *)  (* cascade_height = 0 *)
	reg [RAM_WIDTH-1:0] 	   ram_t[0:RAM_LENGTH-1];

	`ra if(a_write_enable_rrr) ram_t[a_write_addr_rrr] <= a_write_data_rrr;

	`rb b_read_data_r <= ram_t[b_read_addr_r];
	`rb b_read_data_rr  <= b_read_data_r;
     end
   else
     begin
        //error_module error(clk);  // Need to define a RAM implementation.  DELAY must be set in outer module to match.
     end // else: !if( USE_BRAM )

   for(item=0; item<RAM_ITEM_NUMBER; item = item + 1)
     `w b_read_data_ro[item] = b_read_data_rr[item*RAM_ITEM_WIDTH +: RAM_ITEM_WIDTH];
	
endgenerate

     
endmodule // capture_ram_inner


