
//`timescale 1ns / 1ps

`define w assign
`define raxis always@(posedge IN_AXIS_aclk)
`define raxi  always@(posedge S_AXI_ACLK)

//
// Only works for power-of-2 samples per clock.  Only works if AXI_SAMPLES_PER_WORD
// evenly divides into AXIS_SAMPLES_PER_CLOCK
//
module simple_trigger
  #(
    // Capture parameters
    parameter AXIS_SAMPLES_PER_CLOCK       = 8,
    parameter AXIS_BITS_PER_SAMPLE         = 16,
    parameter AXIS_BUS_WIDTH               = AXIS_BITS_PER_SAMPLE * AXIS_SAMPLES_PER_CLOCK,
    
    // ARM interface parameters
    parameter AXI_BITS_PER_WORD            = 32,
    parameter AXI_BYTES_PER_WORD           = AXI_BITS_PER_WORD/8,
    parameter NUM_REGISTERS                = 16,  // Must be greater than 1
    parameter AXI_BYTE_ADDR_WIDTH          = $clog2( NUM_REGISTERS * AXI_BYTES_PER_WORD )
    )
   (
    // Data collection and passthrough

    (* X_INTERFACE_INFO = "xilinx.com:signal:clock:1.0 IN_AXIS_aclk CLK" *)
    (* X_INTERFACE_PARAMETER = "ASSOCIATED_BUSIF IN_0_AXIS:IN_1_AXIS:IN_2_AXIS:IN_3_AXIS:IN_4_AXIS:IN_5_AXIS:IN_6_AXIS:IN_7_AXIS:PASS_0_AXIS:PASS_1_AXIS:PASS_2_AXIS:PASS_3_AXIS:PASS_4_AXIS:PASS_5_AXIS:PASS_6_AXIS:PASS_7_AXIS, ASSOCIATED_RESET IN_AXIS_aresetn" *)
    input                                IN_AXIS_aclk,

    (* X_INTERFACE_INFO = "xilinx.com:signal:reset:1.0 IN_AXIS_aresetn RST" *)
    input                                IN_AXIS_aresetn,

    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_0_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_0_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_0_AXIS TREADY" *)  output wire                          IN_0_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_0_AXIS TVALID" *)  input                                IN_0_AXIS_tvalid,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_1_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_1_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_1_AXIS TREADY" *)  output wire                          IN_1_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_1_AXIS TVALID" *)  input                                IN_1_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_2_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_2_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_2_AXIS TREADY" *)  output wire                          IN_2_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_2_AXIS TVALID" *)  input                                IN_2_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_3_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_3_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_3_AXIS TREADY" *)  output wire                          IN_3_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_3_AXIS TVALID" *)  input                                IN_3_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_4_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_4_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_4_AXIS TREADY" *)  output wire                          IN_4_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_4_AXIS TVALID" *)  input                                IN_4_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_5_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_5_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_5_AXIS TREADY" *)  output wire                          IN_5_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_5_AXIS TVALID" *)  input                                IN_5_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_6_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_6_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_6_AXIS TREADY" *)  output wire                          IN_6_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_6_AXIS TVALID" *)  input                                IN_6_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_7_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_7_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_7_AXIS TREADY" *)  output wire                          IN_7_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_7_AXIS TVALID" *)  input                                IN_7_AXIS_tvalid,


    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_0_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_0_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_0_AXIS TREADY" *)  input                              PASS_0_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_0_AXIS TVALID" *)  output reg                         PASS_0_AXIS_tvalid,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_1_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_1_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_1_AXIS TREADY" *)  input                              PASS_1_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_1_AXIS TVALID" *)  output reg                         PASS_1_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_2_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_2_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_2_AXIS TREADY" *)  input                              PASS_2_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_2_AXIS TVALID" *)  output reg                         PASS_2_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_3_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_3_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_3_AXIS TREADY" *)  input                              PASS_3_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_3_AXIS TVALID" *)  output reg                         PASS_3_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_4_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_4_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_4_AXIS TREADY" *)  input                              PASS_4_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_4_AXIS TVALID" *)  output reg                         PASS_4_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_5_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_5_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_5_AXIS TREADY" *)  input                              PASS_5_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_5_AXIS TVALID" *)  output reg                         PASS_5_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_6_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_6_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_6_AXIS TREADY" *)  input                              PASS_6_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_6_AXIS TVALID" *)  output reg                         PASS_6_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_7_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]    PASS_7_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_7_AXIS TREADY" *)  input                              PASS_7_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_7_AXIS TVALID" *)  output reg                         PASS_7_AXIS_tvalid,


    // Data is continuously collected while collect is high
    // bringing collect low stops collection immediately
    // so the data can be read.
    //
    // Arming the trigger immediately brings collect high.
    // Collect goes low when trigger parameters have been
    // met.
    output reg                           collect,


    input                                external_trigger,
    
    //
    // This is the interface to the ARM processor
    //
    input                                S_AXI_ACLK,
    input                                S_AXI_ARESETN,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_AWADDR,
    input [2 : 0]                        S_AXI_AWPROT,
    input                                S_AXI_AWVALID,
    output reg                           S_AXI_AWREADY,
    input [AXI_BITS_PER_WORD-1 : 0]      S_AXI_WDATA,
    input [AXI_BYTES_PER_WORD-1 : 0]     S_AXI_WSTRB,
    input                                S_AXI_WVALID,
    output reg                           S_AXI_WREADY,
    output reg [1 : 0]                   S_AXI_BRESP,
    output reg                           S_AXI_BVALID,
    input                                S_AXI_BREADY,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_ARADDR,
    input [2 : 0]                        S_AXI_ARPROT,
    input                                S_AXI_ARVALID,
    output reg                           S_AXI_ARREADY,
    output reg [AXI_BITS_PER_WORD-1 : 0] S_AXI_RDATA,
    output reg [1 : 0]                   S_AXI_RRESP,
    output reg                           S_AXI_RVALID,
    input                                S_AXI_RREADY
);
   
   localparam TRIGGER_BELOW_MIN_THEN_ABOVE_MAX = 2'd0;
   localparam TRIGGER_ABOVE_MAX_THEN_BELOW_MIN = 2'd1;
   localparam TRIGGER_BETWEEN_MIN_AND_MAX      = 2'd2;
   localparam TRIGGER_NOT_BETWEEN_MIN_AND_MAX  = 2'd3;

   genvar                                i, j;

   // Fixed in and passthrough signals
   `w IN_0_AXIS_tready = 1'b1;
   `w IN_1_AXIS_tready = 1'b1;
   `w IN_2_AXIS_tready = 1'b1;
   `w IN_3_AXIS_tready = 1'b1;
   `w IN_4_AXIS_tready = 1'b1;
   `w IN_5_AXIS_tready = 1'b1;
   `w IN_6_AXIS_tready = 1'b1;
   `w IN_7_AXIS_tready = 1'b1;

   reg [AXIS_BUS_WIDTH-1:0]              IN_0_AXIS_tdata_r;
   reg [AXIS_BUS_WIDTH-1:0]              IN_1_AXIS_tdata_r;
   reg [AXIS_BUS_WIDTH-1:0]              IN_2_AXIS_tdata_r;
   reg [AXIS_BUS_WIDTH-1:0]              IN_3_AXIS_tdata_r;
   reg [AXIS_BUS_WIDTH-1:0]              IN_4_AXIS_tdata_r;
   reg [AXIS_BUS_WIDTH-1:0]              IN_5_AXIS_tdata_r;
   reg [AXIS_BUS_WIDTH-1:0]              IN_6_AXIS_tdata_r;
   reg [AXIS_BUS_WIDTH-1:0]              IN_7_AXIS_tdata_r;

   reg                                   IN_0_AXIS_tvalid_r;
   reg                                   IN_1_AXIS_tvalid_r;
   reg                                   IN_2_AXIS_tvalid_r;
   reg                                   IN_3_AXIS_tvalid_r;
   reg                                   IN_4_AXIS_tvalid_r;
   reg                                   IN_5_AXIS_tvalid_r;
   reg                                   IN_6_AXIS_tvalid_r;
   reg                                   IN_7_AXIS_tvalid_r;

   `raxis IN_0_AXIS_tdata_r  <= IN_0_AXIS_tdata;   
   `raxis IN_1_AXIS_tdata_r  <= IN_1_AXIS_tdata;   
   `raxis IN_2_AXIS_tdata_r  <= IN_2_AXIS_tdata;   
   `raxis IN_3_AXIS_tdata_r  <= IN_3_AXIS_tdata;   
   `raxis IN_4_AXIS_tdata_r  <= IN_4_AXIS_tdata;   
   `raxis IN_5_AXIS_tdata_r  <= IN_5_AXIS_tdata;   
   `raxis IN_6_AXIS_tdata_r  <= IN_6_AXIS_tdata;   
   `raxis IN_7_AXIS_tdata_r  <= IN_7_AXIS_tdata;   

   `raxis PASS_0_AXIS_tdata  <= IN_0_AXIS_tdata_r;
   `raxis PASS_1_AXIS_tdata  <= IN_1_AXIS_tdata_r;   
   `raxis PASS_2_AXIS_tdata  <= IN_2_AXIS_tdata_r;   
   `raxis PASS_3_AXIS_tdata  <= IN_3_AXIS_tdata_r;   
   `raxis PASS_4_AXIS_tdata  <= IN_4_AXIS_tdata_r;   
   `raxis PASS_5_AXIS_tdata  <= IN_5_AXIS_tdata_r;   
   `raxis PASS_6_AXIS_tdata  <= IN_6_AXIS_tdata_r;   
   `raxis PASS_7_AXIS_tdata  <= IN_7_AXIS_tdata_r;   

   `raxis IN_0_AXIS_tvalid_r <= IN_0_AXIS_tvalid;
   `raxis IN_1_AXIS_tvalid_r <= IN_1_AXIS_tvalid;
   `raxis IN_2_AXIS_tvalid_r <= IN_2_AXIS_tvalid;
   `raxis IN_3_AXIS_tvalid_r <= IN_3_AXIS_tvalid;
   `raxis IN_4_AXIS_tvalid_r <= IN_4_AXIS_tvalid;
   `raxis IN_5_AXIS_tvalid_r <= IN_5_AXIS_tvalid;
   `raxis IN_6_AXIS_tvalid_r <= IN_6_AXIS_tvalid;
   `raxis IN_7_AXIS_tvalid_r <= IN_7_AXIS_tvalid;

   `raxis PASS_0_AXIS_tvalid <= IN_0_AXIS_tvalid_r;
   `raxis PASS_1_AXIS_tvalid <= IN_1_AXIS_tvalid_r;
   `raxis PASS_2_AXIS_tvalid <= IN_2_AXIS_tvalid_r;
   `raxis PASS_3_AXIS_tvalid <= IN_3_AXIS_tvalid_r;
   `raxis PASS_4_AXIS_tvalid <= IN_4_AXIS_tvalid_r;
   `raxis PASS_5_AXIS_tvalid <= IN_5_AXIS_tvalid_r;
   `raxis PASS_6_AXIS_tvalid <= IN_6_AXIS_tvalid_r;
   `raxis PASS_7_AXIS_tvalid <= IN_7_AXIS_tvalid_r;

   // Registers coming from other clock domain
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg                                     software_arm_sr_cross;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [1:0]                               trigger_mode_sr_cross;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [3:0]                               trigger_adc_select_sr_cross;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [19:0]                              initial_countdown_in_sr_cross;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg signed [AXIS_BITS_PER_SAMPLE-1:0]   min_trigger_sr_cross;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg signed [AXIS_BITS_PER_SAMPLE-1:0]   max_trigger_sr_cross;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [19:0]                              final_countdown_in_sr_cross;

   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)  reg                                     software_arm_pre_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)  reg [1:0]                               trigger_mode_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)  reg [3:0]                               trigger_adc_select_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)  reg [19:0]                              initial_countdown_in_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)  reg signed [AXIS_BITS_PER_SAMPLE-1:0]   min_trigger_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)  reg signed [AXIS_BITS_PER_SAMPLE-1:0]   max_trigger_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)  reg [19:0]                              final_countdown_in_sr;

   reg                                   software_arm_sr;
   
   `raxis software_arm_pre_sr        <= software_arm_sr_cross;
   `raxis trigger_mode_sr            <= trigger_mode_sr_cross;
   `raxis trigger_adc_select_sr      <= trigger_adc_select_sr_cross;
   `raxis initial_countdown_in_sr    <= initial_countdown_in_sr_cross;
   `raxis min_trigger_sr             <= min_trigger_sr_cross;
   `raxis max_trigger_sr             <= max_trigger_sr_cross;
   `raxis final_countdown_in_sr      <= final_countdown_in_sr_cross;


   // For trigger select 8 (external trigger synced), don't start capture until we get an
   // external trigger.  This is to get the FFT input in sync with the DAC frame, since the
   // DAC frame is currently the external trigger.  This is because the FFT is held
   // in reset until collection starts, but acquires a phase associated with the
   // start of collection.  Collection then stops at a later trigger.
   `raxis software_arm_sr <= (trigger_adc_select_sr==8) ? software_arm_pre_sr && ( external_trigger || software_arm_sr) :
                             /**/                         software_arm_pre_sr                                           ;
   
   reg [AXIS_BUS_WIDTH-1:0]              IN_AXIS_tdata_rs;
   reg                                   IN_AXIS_tvalid_rs;

   `raxis IN_AXIS_tdata_rs <= (trigger_adc_select_sr==0) ? IN_0_AXIS_tdata_r :
                              (trigger_adc_select_sr==1) ? IN_1_AXIS_tdata_r :
                              (trigger_adc_select_sr==2) ? IN_2_AXIS_tdata_r :
                              (trigger_adc_select_sr==3) ? IN_3_AXIS_tdata_r :
                              (trigger_adc_select_sr==4) ? IN_4_AXIS_tdata_r :
                              (trigger_adc_select_sr==5) ? IN_5_AXIS_tdata_r :
                              (trigger_adc_select_sr==6) ? IN_6_AXIS_tdata_r :
                              /* case 7 */                 IN_7_AXIS_tdata_r ;

   //
   // Valid isn't actually used, currently.  It's here for later in case we need it.
   // Right now, assume data is valid every clock cycle.
   //
   `raxis IN_AXIS_tvalid_rs <= (trigger_adc_select_sr==0) ? IN_0_AXIS_tvalid_r :
                               (trigger_adc_select_sr==1) ? IN_1_AXIS_tvalid_r :
                               (trigger_adc_select_sr==2) ? IN_2_AXIS_tvalid_r :
                               (trigger_adc_select_sr==3) ? IN_3_AXIS_tvalid_r :
                               (trigger_adc_select_sr==4) ? IN_4_AXIS_tvalid_r :
                               (trigger_adc_select_sr==5) ? IN_5_AXIS_tvalid_r :
                               (trigger_adc_select_sr==6) ? IN_6_AXIS_tvalid_r :
                               (trigger_adc_select_sr==7) ? IN_7_AXIS_tvalid_r :
                               /* case 7 */                 0                  ;  // External trigger.
   
   
   //
   // Initiate delay from software arm to allow a desired number of samples
   // before the trigger.
   //
   reg [19:0]                              initial_countdown_sr;
   wire                                    initial_delay_done_sw;
   reg                                     enable_triggering_sr;
   
   `raxis enable_triggering_sr   <=  (!IN_AXIS_aresetn)     ?  0                      :
                                     (!software_arm_sr)     ?  0                      :
                                     initial_delay_done_sw  ?  1                      :
                                     /* default */             enable_triggering_sr   ;
      
                      
   `raxis initial_countdown_sr   <=  (!IN_AXIS_aresetn)      ?  0                         :
                                     (!software_arm_sr)      ?  initial_countdown_in_sr   :
                                     initial_delay_done_sw   ?  0                         :
                                     /* default */              initial_countdown_sr - 1  ;

   `w     initial_delay_done_sw   =  (initial_countdown_sr == 0);
   
   
   //
   // Generate signals for each sample regarding whether the sample would arm or trigger
   // based on solely its level and the trigger mode.
   //      
   reg [AXIS_SAMPLES_PER_CLOCK-1:0]        arm_condition_sr;
   reg [AXIS_SAMPLES_PER_CLOCK-1:0]        trigger_condition_sr;

   generate
      for(i=0; i<AXIS_SAMPLES_PER_CLOCK; i=i+1)
        begin
           reg signed [AXIS_BITS_PER_SAMPLE-1:0]  in_sample_sr;
           reg signed [AXIS_BITS_PER_SAMPLE:0]    min_diff_sr;
           reg signed [AXIS_BITS_PER_SAMPLE:0]    max_diff_sr;
           
           `raxis in_sample_sr             <= (!IN_AXIS_aresetn)  ?  0  : IN_AXIS_tdata_rs[ i*AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE ];
           
           `raxis min_diff_sr              <= (!IN_AXIS_aresetn)  ?  0  : (  {   in_sample_sr[AXIS_BITS_PER_SAMPLE-1], in_sample_sr   } - 
                                                                             { min_trigger_sr[AXIS_BITS_PER_SAMPLE-1], min_trigger_sr }    ); 
           `raxis max_diff_sr              <= (!IN_AXIS_aresetn)  ?  0  : (  {   in_sample_sr[AXIS_BITS_PER_SAMPLE-1], in_sample_sr   } - 
                                                                             { max_trigger_sr[AXIS_BITS_PER_SAMPLE-1], max_trigger_sr }    ); 
           
           `w less_than_min                 = min_diff_sr[AXIS_BITS_PER_SAMPLE];
           `w less_than_max                 = max_diff_sr[AXIS_BITS_PER_SAMPLE];
           `w greater_than_min              = !less_than_min;  // We don't care about the exactly equal case
           `w greater_than_max              = !less_than_max;
           
           `raxis arm_condition_sr[i]      <= (!IN_AXIS_aresetn)                                        ?  0                                    :
                                              (!enable_triggering_sr)                                   ?  0                                    : 
                                              (trigger_mode_sr==TRIGGER_BELOW_MIN_THEN_ABOVE_MAX)       ?  less_than_min                        :
                                              (trigger_mode_sr==TRIGGER_ABOVE_MAX_THEN_BELOW_MIN)       ?  greater_than_max                     :
                                              (trigger_mode_sr==TRIGGER_BETWEEN_MIN_AND_MAX)            ?  less_than_min || greater_than_max    :
                                              /* (trigger_mode_sr==TRIGGER_NOT_BETWEEN_MIN_AND_MAX) */     greater_than_min && less_than_max    ;

           `raxis trigger_condition_sr[i]  <= (!IN_AXIS_aresetn)                                        ?  0                                    : 
                                              (!enable_triggering_sr)                                   ?  0                                    : 
                                              (trigger_mode_sr==TRIGGER_BELOW_MIN_THEN_ABOVE_MAX)       ?  greater_than_max                     :
                                              (trigger_mode_sr==TRIGGER_ABOVE_MAX_THEN_BELOW_MIN)       ?  less_than_min                        :
                                              (trigger_mode_sr==TRIGGER_BETWEEN_MIN_AND_MAX)            ?  greater_than_min && less_than_max    :
                                              /* (trigger_mode_sr==TRIGGER_NOT_BETWEEN_MIN_AND_MAX) */     less_than_min || greater_than_max    ;   
        end
   endgenerate


   //
   // Need to create registers showing for which samples we're armed.  Once we're armed, we're armed
   // for later samples.  Use the adder carry chain for this, to do it in a way that can meet timing for
   // larger numbers of samples per clock.
   //
   reg [AXIS_SAMPLES_PER_CLOCK-1:0]               this_sample_armed_sr;
   wire                                           armed_on_previous_clock_sw;
   
   `w armed_on_previous_clock_sw = this_sample_armed_sr[AXIS_SAMPLES_PER_CLOCK-1];
   
   generate
      for(i=0; i<AXIS_SAMPLES_PER_CLOCK; i = i + 1)
        begin

           wire signed [i+2:0] add_in_1;
           wire signed [i+2:0] add_in_2;
           wire signed [i+2:0] sum;
           
           `w add_in_1 = -1;
           `w add_in_2 = { 1'b1, arm_condition_sr[i:0], armed_on_previous_clock_sw };
           
           `w sum = add_in_1 + add_in_2;

           `raxis this_sample_armed_sr[i]  <=  (!IN_AXIS_aresetn)       ?  0         :
                                               (!enable_triggering_sr)  ?  0         :
                                               /* default */               sum[i+2]  ;

        end // for (i=0; i<AXIS_SAMPLES_PER_CLOCK; i++)
   endgenerate
           

   //
   // If we're armed on a particular sample and we have a trigger condition, trigger.
   //
   wire                              trigger_sw;
   reg                               triggered_sr;
   reg [AXIS_SAMPLES_PER_CLOCK-1:0]  trigger_condition_srr;
   reg                               external_trigger_condition_sr;
   
   `raxis trigger_condition_srr <= (!IN_AXIS_aresetn)         ?  0                     :
                                   (!enable_triggering_sr)    ?  0                     : 
                                   /* default */                 trigger_condition_sr  ;

   `raxis external_trigger_condition_sr <= external_trigger && (trigger_adc_select_sr==8);
   
   `w     trigger_sw             = (|(trigger_condition_srr & this_sample_armed_sr)) | external_trigger_condition_sr;
 
   `raxis triggered_sr          <= (!IN_AXIS_aresetn)         ?  0                     :
                                   (!enable_triggering_sr)    ?  0                     :
                                   trigger_sw                 ?  1                     :
                                   /* default */                 triggered_sr          ;
   
   
   //
   //  After the trigger, wait to collect samples before we stop collection.
   //
   reg [19:0]                              final_countdown_sr;
   wire                                    final_delay_done_sw;
   reg                                     final_delay_done_sr;
   reg                                     collect_r;
   
   `raxis collect_r            <=  (!IN_AXIS_aresetn)     ?  0           :
                                   final_delay_done_sr    ?  0           :
                                   software_arm_sr        ?  1           :
                                   /* default */             collect_r   ;

   `raxis collect              <=  collect_r;

                            
   `raxis final_countdown_sr   <=  (!IN_AXIS_aresetn)      ?  0                         :
                                   (!triggered_sr)         ?  final_countdown_in_sr     :
                                   final_delay_done_sw     ?  0                         :
                                   /* default */              final_countdown_sr - 1    ;

   `w     final_delay_done_sw   =  (final_countdown_sr == 0);
   `raxis final_delay_done_sr  <=  (!IN_AXIS_aresetn) ? 0 : final_delay_done_sw;


   //
   // Signals from above that get relayed from AXI processor interface on the AXI
   // clock (not the AXIS clock).    
   //
   // reg                                     software_arm_sr;
   // reg [19:0]                              initial_countdown_in_sr;
   // reg [19:0]                              final_countdown_in_sr;
   // reg signed [AXIS_BITS_PER_SAMPLE-1:0]   min_trigger_sr;
   // reg signed [AXIS_BITS_PER_SAMPLE-1:0]   max_trigger_sr;
   // reg [1:0]                               trigger_mode_sr;
   // reg [2:0]                               trigger_adc_select_sr;
   
   
   reg [31:0]                              reg_trigger_arm_ar;              // Register 0
   reg [31:0]                              reg_trigger_mode_ar;             // Register 1
   reg [31:0]                              reg_samples_before_trigger_ar;   // Register 2
   reg [31:0]                              reg_trigger_minimum_ar;          // Register 3
   reg [31:0]                              reg_trigger_maximum_ar;          // Register 4
   reg [31:0]                              reg_samples_after_trigger_ar;    // Register 5


   //
   // Cross from one clock domain to the other.
   //
   `raxis software_arm_sr_cross            <= reg_trigger_arm_ar;
   `raxis trigger_mode_sr_cross            <= reg_trigger_mode_ar[17:16];
   `raxis trigger_adc_select_sr_cross      <= reg_trigger_mode_ar[3:0];
   `raxis initial_countdown_in_sr_cross    <= reg_samples_before_trigger_ar;
   `raxis min_trigger_sr_cross             <= reg_trigger_minimum_ar;
   `raxis max_trigger_sr_cross             <= reg_trigger_maximum_ar;
   `raxis final_countdown_in_sr_cross      <= reg_samples_after_trigger_ar;

   
   //
   // Handle READs/WRITEs from the processor on the AXI bus.
   //    

   localparam                              WORD_ADDR_BITS         = $clog2(NUM_REGISTERS);   
   localparam                              BYTE_ADDR_BITS_IN_WORD = $clog2(AXI_BYTES_PER_WORD);
   
   reg [WORD_ADDR_BITS-1 : 0]              write_address;
   reg [AXI_BITS_PER_WORD-1:0]             write_data;
   wire                                    write_enable;
   wire                                    write_enable_w;
   reg                                     write_enable_r;
   wire                                    write_done;
   
   reg                                     read_enable;
   reg [WORD_ADDR_BITS-1 : 0]              read_address;
   wire [AXI_BITS_PER_WORD-1:0]            read_data;
   wire                                    read_done;

   //
   // The write interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_AWREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_AWREADY && S_AXI_AWVALID)   ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_AWREADY  ;

   `raxi S_AXI_WREADY  <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_WREADY && S_AXI_WVALID)     ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_WREADY   ;

   `raxi S_AXI_BVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          write_done                         ?  1'b1           :
                          (S_AXI_BREADY && S_AXI_BVALID)     ?  1'b0           :
                          /* default */                         S_AXI_BVALID   ;

   `raxi S_AXI_BRESP   <= 2'b0;

   //
   // The read interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_ARREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_ARREADY && S_AXI_ARVALID)   ?  1'b0           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b1           :
                          /* default */                         S_AXI_ARREADY  ;
   

   `raxi S_AXI_RVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          read_done                          ?  1'b1           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b0           :
                          /* default */                         S_AXI_RVALID   ;

   `raxi S_AXI_RRESP   <= 2'b0;
   
   `raxi S_AXI_RDATA   <= (!S_AXI_ARESETN)                   ?  0              : 
                          read_done                          ?  read_data      :
                          /* default */                         S_AXI_RDATA    ;


   //
   // Define write addresses, data, and enables
   //
   `w write_enable_w          = !S_AXI_WREADY && !S_AXI_AWREADY;
   `raxi write_enable_r       = (!S_AXI_ARESETN) ? 0 : write_enable_w;
   `w write_enable            = write_enable_w && !write_enable_r;   // Only pulse write enable for a single clock, regardless of BREADY
   
   `raxi write_address       <= (!S_AXI_ARESETN)                  ?  0                                                              :
                                (S_AXI_AWREADY && S_AXI_AWVALID)  ?  S_AXI_AWADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                        write_address                                                  ;
   
   `raxi write_data          <= (!S_AXI_ARESETN)                  ?  0                :
                                (S_AXI_WREADY && S_AXI_WVALID)    ?  S_AXI_WDATA      :
                                /* default */                        write_data       ;

   `w write_done              = write_enable;  // Write done immediately after address and data are both present
   

   //
   // Define read addresses, data, and enables
   //
   `raxi read_address        <= (!S_AXI_ARESETN)               ?  0                                                              : 
                                S_AXI_ARREADY & S_AXI_ARVALID  ?  S_AXI_ARADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                     read_address                                                   ;
   
   `raxi read_enable         <= S_AXI_ARREADY & S_AXI_ARVALID;

   `w read_done               = read_enable;  // Read data available immediately after address is present.
   
   // read_data  is assigned a value below



   //
   // Handle read data
   //
   reg signed [AXIS_BITS_PER_SAMPLE-1:0]     in_sample_0;
   
   `raxis in_sample_0     <=  IN_AXIS_tdata_rs[0 +: AXIS_BITS_PER_SAMPLE];

   // FALSE_PATH_DEST is used to find these paths and label them as false paths in the TCL.
   // There is no reliance here on timing across the clock domains.
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg signed [31:0]  in_sample_0_32_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg signed [31:0]  in_sample_0_32_arr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg signed [31:0]  min_trigger_32_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg signed [31:0]  min_trigger_32_arr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg signed [31:0]  max_trigger_32_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg signed [31:0]  max_trigger_32_arr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg        [31:0]  read_address_0_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg        [31:0]  read_address_0_arr;

   `raxi  in_sample_0_32_ar  <=  in_sample_0;
   `raxi  in_sample_0_32_arr <=  in_sample_0_32_ar;
   
   `raxi  min_trigger_32_ar  <=  min_trigger_sr;
   `raxi  min_trigger_32_arr <=  min_trigger_32_ar;

   `raxi  max_trigger_32_ar  <=  max_trigger_sr;
   `raxi  max_trigger_32_arr <=  max_trigger_32_ar;

   `raxi read_address_0_ar <= { trigger_mode_sr, 23'b0, collect, final_delay_done_sr, triggered_sr, armed_on_previous_clock_sw, enable_triggering_sr, initial_delay_done_sw, software_arm_sr};
   `raxi read_address_0_arr <= read_address_0_ar;
   
   
   `w read_data = (read_address==0)  ?  read_address_0_arr               :
                  (read_address==1)  ?  reg_trigger_mode_ar              :
                  (read_address==2)  ?  reg_samples_before_trigger_ar    :
                  (read_address==3)  ?  reg_trigger_minimum_ar           :
                  (read_address==4)  ?  reg_trigger_maximum_ar           :
                  (read_address==5)  ?  reg_samples_after_trigger_ar     :
                  (read_address==6)  ?  in_sample_0_32_arr               :
                  (read_address==7)  ?  min_trigger_32_arr               :
                  (read_address==8)  ?  max_trigger_32_arr               :
                  /* default */         0                                ;

   
   //
   // Handle write data.  Registers write on the AXI clock.
   // None of the other registers are supposed to be written unless reg_trigger_arm_ar is 0.
   // Thus they won't be written at a critical time and thus metastability isn't an issue.
   //
   

   `raxi reg_trigger_arm_ar             <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==0)     ?  write_data                     :
                                            /* default */                             reg_trigger_arm_ar             ;

   `raxi reg_trigger_mode_ar            <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==1)     ?  write_data                     :
                                            /* default */                             reg_trigger_mode_ar            ;
  
   `raxi reg_samples_before_trigger_ar  <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==2)     ?  write_data                     :
                                            /* default */                             reg_samples_before_trigger_ar  ;
   
   `raxi reg_trigger_minimum_ar         <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==3)     ?  write_data                     :
                                            /* default */                             reg_trigger_minimum_ar         ;

   `raxi reg_trigger_maximum_ar         <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==4)     ?  write_data                     :
                                            /* default */                             reg_trigger_maximum_ar         ;

   `raxi reg_samples_after_trigger_ar   <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==5)     ?  write_data                     :
                                            /* default */                             reg_samples_after_trigger_ar   ;
   

endmodule
