
//`timescale 1ns / 1ps

`undef w
`undef raxi
`undef rclk

`define w assign
`define raxi   always@(posedge S_AXI_ACLK)
`define rref   always@(posedge REF_CLK)
`define rclk   always@(posedge CLK)

//
// Register Map:
//
//   Reg 0:  ID 
//   Reg 1:  Control.  Bit 0 is enable operation when 1, disable when 0
//   Reg 2:  NUM_CLOCKS, the number of clocks this unit is measuring
//   Reg 3:  REF_CLK_HZ, the frequency expected for the AXI clock
//
//   Reg 0x10 - 0x1F:  Clock freq in Hz for measured clocks 0 through 15, as an unsigned 32-bit integer.
//   Reg 0x40 - 0x4F:  EXPECTED_CLK_0_HZ through EXPECTED_CLOCK_15_HZ
//

(* DONT_TOUCH = "TRUE" *)
module clock_measurement
  #(
    // Basic Parameters
    parameter NUM_CLOCKS                        = 1,
    
    // Clock Frequency Parameters
    parameter REF_CLOCK_FREQ_HZ                 = 100000000,

    parameter EXPECTED_CLK_0_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_1_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_2_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_3_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_4_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_5_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_6_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_7_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_8_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_9_FREQ_HZ            = 491520000,
    parameter EXPECTED_CLK_10_FREQ_HZ           = 491520000,
    parameter EXPECTED_CLK_11_FREQ_HZ           = 491520000,
    parameter EXPECTED_CLK_12_FREQ_HZ           = 491520000,
    parameter EXPECTED_CLK_13_FREQ_HZ           = 491520000,
    parameter EXPECTED_CLK_14_FREQ_HZ           = 491520000,
    parameter EXPECTED_CLK_15_FREQ_HZ           = 491520000,

    // ID value to return to ARM
    parameter CLOCK_MEASUREMENT_ID         = 32'hCCCCCC00,
    
    // ARM interface parameters
    parameter AXI_BITS_PER_WORD            = 32,
    parameter AXI_BYTES_PER_WORD           = AXI_BITS_PER_WORD/8,
    parameter NUM_REGISTERS                = 16+32*2,
    parameter AXI_BYTE_ADDR_WIDTH          = $clog2( NUM_REGISTERS * AXI_BYTES_PER_WORD )
    )
   (
    //
    // Reference clock, against which measurements are made.
    //
    input                                REF_CLK,
    
    //
    // Input clocks to measure.  They're measured against the
    // AXI clock, which is used as a reference
    //
    input                                CLK_0,
    input                                CLK_1,
    input                                CLK_2,
    input                                CLK_3,
    input                                CLK_4,
    input                                CLK_5,
    input                                CLK_6,
    input                                CLK_7,
    input                                CLK_8,
    input                                CLK_9,
    input                                CLK_10,
    input                                CLK_11,
    input                                CLK_12,
    input                                CLK_13,
    input                                CLK_14,
    input                                CLK_15,
    
    //
    // This is the interface to the ARM processor
    //
    input                                S_AXI_ACLK,
    input                                S_AXI_ARESETN,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_AWADDR,
    input [2 : 0]                        S_AXI_AWPROT,
    input                                S_AXI_AWVALID,
    output reg                           S_AXI_AWREADY,
    input [AXI_BITS_PER_WORD-1 : 0]      S_AXI_WDATA,
    input [AXI_BYTES_PER_WORD-1 : 0]     S_AXI_WSTRB,
    input                                S_AXI_WVALID,
    output reg                           S_AXI_WREADY,
    output reg [1 : 0]                   S_AXI_BRESP,
    output reg                           S_AXI_BVALID,
    input                                S_AXI_BREADY,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_ARADDR,
    input [2 : 0]                        S_AXI_ARPROT,
    input                                S_AXI_ARVALID,
    output reg                           S_AXI_ARREADY,
    output reg [AXI_BITS_PER_WORD-1 : 0] S_AXI_RDATA,
    output reg [1 : 0]                   S_AXI_RRESP,
    output reg                           S_AXI_RVALID,
    input                                S_AXI_RREADY
);

   localparam MAX_CLOCKS          = 16;

   localparam FRAC_OF_SECOND      = 10;  // Note:  If this is changed, must also change multiplier "MULX" after measurement is made

   localparam MEASUREMENT_CYCLES  = REF_CLOCK_FREQ_HZ / FRAC_OF_SECOND;  // duration of measurement window
   localparam WAIT_CYCLES         = MEASUREMENT_CYCLES/4; // Wait time for stable measurement.  Measure in the middle.
   localparam CLEAR_CYCLES        = MEASUREMENT_CYCLES/4; // Clear time.  Only clear for first half, to keep clear separate from next measurement
   localparam TOTAL_CYCLES        = MEASUREMENT_CYCLES + WAIT_CYCLES + CLEAR_CYCLES;
   localparam COUNT_BITS          = $clog2(TOTAL_CYCLES);
   
   // Control registers
   reg [31:0]                            reg_control_ar                  = 1;  // Register 1, the only one writable   
   wire [31:0]                           measured_freq_w[0:MAX_CLOCKS-1];

   //
   // Transfer control to REF_CLK
   //
   (* DONT_TOUCH = "TRUE" *)                            reg   enable_ar   = 0;
   (* ASYNC_REG  = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg   enable_xr   = 0;
   (* ASYNC_REG  = "TRUE" *)                            reg   enable_xrr  = 0;
   (* DONT_TOUCH = "TRUE" *)                            reg   enable_xrrr = 0;
   
   `raxi enable_ar    <= reg_control_ar[0];
   `rref enable_xr    <= enable_ar;
   `rref enable_xrr   <= enable_xr;
   `rref enable_xrrr  <= enable_xrr;

   //
   // AXI clock counter
   //
   // Counts out 1 second of AXI clocks during which count_xr is high.  Then
   // counts an additional 1/4 second during which count_xr is low.  Take
   // a measurement at the end.
   //
   reg [COUNT_BITS-1:0]                  ref_count_xr                    = TOTAL_CYCLES-1;
   reg                                   count_xr;
   reg                                   take_measurement_t_xr           = 0;
   reg                                   clear_xr;
   
   `rref  ref_count_xr <= !enable_xrrr                     ?  TOTAL_CYCLES-1      :
                          (ref_count_xr==TOTAL_CYCLES-1)   ?  0                   :
                          /**/                                ref_count_xr+1      ;

   `rref  count_xr            <= (ref_count_xr < MEASUREMENT_CYCLES);
   `rref  clear_xr            <= (ref_count_xr > MEASUREMENT_CYCLES + WAIT_CYCLES) &&
                                 (ref_count_xr < MEASUREMENT_CYCLES + WAIT_CYCLES + CLEAR_CYCLES/2);

   //
   // Toggle the take_measurement_t_xr signal in the middle of the measurement window, so it has 1/8 second
   // for measurement to settle and 1/8 second for it to be taken before the clear.  It is a toggle rather
   // than a pulse because toggles easily cross asynchronous clock boundaries.
   //
   `rref  take_measurement_t_xr <= take_measurement_t_xr ^ (ref_count_xr == MEASUREMENT_CYCLES + WAIT_CYCLES/2);
      
   genvar                                c;
   generate

      //
      // Zero out unused clocks
      //
      for(c=NUM_CLOCKS; c<MAX_CLOCKS; c=c+1)
        begin
           `w measured_freq_w[c] = 0;
        end
           
      //
      // Measure the clocks
      //
      for(c=0; c<NUM_CLOCKS; c=c+1)
        begin

           //
           // Pick the clock we're measuring
           //
           wire CLK;

           `w CLK = (c==0)  ? CLK_0 :
                    (c==1)  ? CLK_1 :
                    (c==2)  ? CLK_2 :
                    (c==3)  ? CLK_3 :
                    (c==4)  ? CLK_4 :
                    (c==5)  ? CLK_5 :
                    (c==6)  ? CLK_6 :
                    (c==7)  ? CLK_7 :
                    (c==8)  ? CLK_8 :
                    (c==9)  ? CLK_9 :
                    (c==10) ? CLK_10 :
                    (c==11) ? CLK_11 :
                    (c==12) ? CLK_12 :
                    (c==13) ? CLK_13 :
                    (c==14) ? CLK_14 :
                    /**/      CLK_15 ;

           //   
           // Transfer count_xr and clear_xr to this clock domain
           //
           
           (* DONT_TOUCH = "TRUE" *)                            reg   count_xrr;
           (* ASYNC_REG  = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg   count_cr;
           (* ASYNC_REG  = "TRUE" *)                            reg   count_crr;
           (* DONT_TOUCH = "TRUE" *)                            reg   count_crrr;

           (* DONT_TOUCH = "TRUE" *)                            reg   clear_xrr;
           (* ASYNC_REG  = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg   clear_cr;
           (* ASYNC_REG  = "TRUE" *)                            reg   clear_crr;
           (* DONT_TOUCH = "TRUE" *)                            reg   clear_crrr;

           `rref count_xrr    <= count_xr;
           `rclk count_cr     <= count_xrr;
           `rclk count_crr    <= count_cr;
           `rclk count_crrr   <= count_crr;

           `rref clear_xrr    <= clear_xr;
           `rclk clear_cr     <= clear_xrr;
           `rclk clear_crr    <= clear_cr;
           `rclk clear_crrr   <= clear_crr;


           //
           // Count the pulses while count_xr is high
           //
           
           reg [31:0] clock_count_cr;

           `rclk clock_count_cr   <=  clear_crrr   ?  0                   :
                                      count_crrr   ?  clock_count_cr + 1  :
                                      /**/            clock_count_cr      ;

           //
           // Transfer the count result to the axi clock domain.
           //
           (* DONT_TOUCH = "TRUE" *)                            reg [31:0]   clock_count_crr = 0;
           (* ASYNC_REG  = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]   clock_count_ar;
           (* ASYNC_REG  = "TRUE" *)                            reg [31:0]   clock_count_arr;
           (* DONT_TOUCH = "TRUE" *)                            reg [31:0]   clock_count_arrr;

           `rclk clock_count_crr  <= clock_count_cr;
           `raxi clock_count_ar   <= clock_count_crr;
           `raxi clock_count_arr  <= clock_count_ar;
           `raxi clock_count_arrr <= clock_count_arr;

           //
           // Transfer the take_measurement_t_xr signal to the axi clock domain, and turn it into
           // a pulse.
           //
           (* DONT_TOUCH = "TRUE" *)                            reg   take_measurement_t_xrr;
           (* ASYNC_REG  = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg   take_measurement_t_ar;
           (* ASYNC_REG  = "TRUE" *)                            reg   take_measurement_t_arr;
           (* DONT_TOUCH = "TRUE" *)                            reg   take_measurement_t_arrr;
           /**/                                                 reg   take_measurement_ar;
           `rref take_measurement_t_xrr  <= take_measurement_t_xr;
           `raxi take_measurement_t_ar   <= take_measurement_t_xrr;
           `raxi take_measurement_t_arr  <= take_measurement_t_ar;
           `raxi take_measurement_t_arrr <= take_measurement_t_arr;

           `raxi take_measurement_ar <= take_measurement_t_arrr ^ take_measurement_t_arr;
           
           //
           // For the case of a lost clock, there will be no change at all on clock_count_cr.  Detect this so that
           // zero frequency can be returned.
           //           
           reg        clock_count_changed_ar;
                         
           `raxi clock_count_changed_ar <= take_measurement_ar                  ?  0                       :
                                           clock_count_arrr != clock_count_arr  ?  1                       :
                                           /**/                                    clock_count_changed_ar  ;

           //
           // Take the measurement.
           //
           reg [31:0] measured_freq_ar;
           reg [31:0] measured_freq_arr;
           
           `raxi measured_freq_ar <=  !enable_ar                                       ?  0                    :
                                      take_measurement_ar && clock_count_changed_ar    ?  clock_count_arr      :
                                      take_measurement_ar                              ?  0                    :
                                      /**/                                                measured_freq_ar     ;

           // MULX, multiply by 10 corresponding to 1/10 of a second during which measurements
           // are taken.  This matches FRAC_OF_SECOND above; they must be changed together.
           `raxi measured_freq_arr <= (measured_freq_ar<<1) + (measured_freq_ar<<3);  // x*2 + x*8 = x*10
           
           `w measured_freq_w[c] = measured_freq_arr;
        end

   endgenerate
   
   
   
   //
   // Handle READs/WRITEs from the processor on the AXI bus.
   //    

   localparam                              WORD_ADDR_BITS         = $clog2(NUM_REGISTERS);   
   localparam                              BYTE_ADDR_BITS_IN_WORD = $clog2(AXI_BYTES_PER_WORD);
   
   reg [WORD_ADDR_BITS-1 : 0]              write_address;
   reg [AXI_BITS_PER_WORD-1:0]             write_data;
   wire                                    write_enable;
   wire                                    write_enable_w;
   reg                                     write_enable_r;
   wire                                    write_done;
   
   reg                                     read_enable;
   reg [WORD_ADDR_BITS-1 : 0]              read_address;
   wire [AXI_BITS_PER_WORD-1:0]            read_data;
   wire                                    read_done;

   //
   // The write interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_AWREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_AWREADY && S_AXI_AWVALID)   ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_AWREADY  ;

   `raxi S_AXI_WREADY  <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_WREADY && S_AXI_WVALID)     ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_WREADY   ;

   `raxi S_AXI_BVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          write_done                         ?  1'b1           :
                          (S_AXI_BREADY && S_AXI_BVALID)     ?  1'b0           :
                          /* default */                         S_AXI_BVALID   ;

   `raxi S_AXI_BRESP   <= 2'b0;

   //
   // The read interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_ARREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_ARREADY && S_AXI_ARVALID)   ?  1'b0           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b1           :
                          /* default */                         S_AXI_ARREADY  ;
   

   `raxi S_AXI_RVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          read_done                          ?  1'b1           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b0           :
                          /* default */                         S_AXI_RVALID   ;

   `raxi S_AXI_RRESP   <= 2'b0;
   
   `raxi S_AXI_RDATA   <= (!S_AXI_ARESETN)                   ?  0              : 
                          read_done                          ?  read_data      :
                          /* default */                         S_AXI_RDATA    ;


   //
   // Define write addresses, data, and enables
   //
   `w write_enable_w          = !S_AXI_WREADY && !S_AXI_AWREADY;
   `raxi write_enable_r      <= (!S_AXI_ARESETN) ? 0 : write_enable_w;
   `w write_enable            = write_enable_w && !write_enable_r;   // Only pulse write enable for a single clock, regardless of BREADY
   
   `raxi write_address       <= (!S_AXI_ARESETN)                  ?  0                                                              :
                                (S_AXI_AWREADY && S_AXI_AWVALID)  ?  S_AXI_AWADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                        write_address                                                  ;
   
   `raxi write_data          <= (!S_AXI_ARESETN)                  ?  0                :
                                (S_AXI_WREADY && S_AXI_WVALID)    ?  S_AXI_WDATA      :
                                /* default */                        write_data       ;

   `w write_done              = write_enable;  // Write done immediately after address and data are both present
   

   //
   // Define read addresses, data, and enables
   //
   `raxi read_address        <= (!S_AXI_ARESETN)               ?  0                                                              : 
                                S_AXI_ARREADY & S_AXI_ARVALID  ?  S_AXI_ARADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                     read_address                                                   ;
   
   `raxi read_enable         <= S_AXI_ARREADY & S_AXI_ARVALID;

   `w read_done               = read_enable;  // Read data available immediately after address is present.
   
   // read_data  is assigned a value below



   //
   // Handle read data
   //
   
   `w read_data = (read_address==0)        ?  CLOCK_MEASUREMENT_ID               :
                  (read_address==1)        ?  reg_control_ar                     :
                  (read_address==2)        ?  NUM_CLOCKS                         :
                  (read_address==3)        ?  REF_CLOCK_FREQ_HZ                  :

                  (read_address==32'h10)   ?  measured_freq_w[0]                 :
                  (read_address==32'h11)   ?  measured_freq_w[1]                 :
                  (read_address==32'h12)   ?  measured_freq_w[2]                 :
                  (read_address==32'h13)   ?  measured_freq_w[3]                 :
                  (read_address==32'h14)   ?  measured_freq_w[4]                 :
                  (read_address==32'h15)   ?  measured_freq_w[5]                 :
                  (read_address==32'h16)   ?  measured_freq_w[6]                 :
                  (read_address==32'h17)   ?  measured_freq_w[7]                 :
                  (read_address==32'h18)   ?  measured_freq_w[8]                 :
                  (read_address==32'h19)   ?  measured_freq_w[9]                 :
                  (read_address==32'h1A)   ?  measured_freq_w[10]                :
                  (read_address==32'h1B)   ?  measured_freq_w[11]                :
                  (read_address==32'h1C)   ?  measured_freq_w[12]                :
                  (read_address==32'h1D)   ?  measured_freq_w[13]                :
                  (read_address==32'h1E)   ?  measured_freq_w[14]                :
                  (read_address==32'h1F)   ?  measured_freq_w[15]                :

                  (read_address==32'h40)   ?  EXPECTED_CLK_0_FREQ_HZ             :
                  (read_address==32'h41)   ?  EXPECTED_CLK_1_FREQ_HZ             :
                  (read_address==32'h42)   ?  EXPECTED_CLK_2_FREQ_HZ             :
                  (read_address==32'h43)   ?  EXPECTED_CLK_3_FREQ_HZ             :
                  (read_address==32'h44)   ?  EXPECTED_CLK_4_FREQ_HZ             :
                  (read_address==32'h45)   ?  EXPECTED_CLK_5_FREQ_HZ             :
                  (read_address==32'h46)   ?  EXPECTED_CLK_6_FREQ_HZ             :
                  (read_address==32'h47)   ?  EXPECTED_CLK_7_FREQ_HZ             :
                  (read_address==32'h48)   ?  EXPECTED_CLK_8_FREQ_HZ             :
                  (read_address==32'h49)   ?  EXPECTED_CLK_9_FREQ_HZ             :
                  (read_address==32'h4A)   ?  EXPECTED_CLK_10_FREQ_HZ            :
                  (read_address==32'h4B)   ?  EXPECTED_CLK_11_FREQ_HZ            :
                  (read_address==32'h4C)   ?  EXPECTED_CLK_12_FREQ_HZ            :
                  (read_address==32'h4D)   ?  EXPECTED_CLK_13_FREQ_HZ            :
                  (read_address==32'h4E)   ?  EXPECTED_CLK_14_FREQ_HZ            :
                  (read_address==32'h4F)   ?  EXPECTED_CLK_15_FREQ_HZ            :
                  
                  /* default */               0                                  ;

   
   //
   // Handle write data.  Registers write on the AXI clock.
   // None of the other registers are supposed to be written unless reg_trigger_arm_ar is 0.
   // Thus they won't be written at a critical time and thus metastability isn't an issue.
   //
   

   `raxi reg_control_ar                <=  (!S_AXI_ARESETN)                        ?  1                              :
                                            write_enable && (write_address==1)     ?  write_data                     :
                                            /* default */                             reg_control_ar                 ;

   

endmodule
