
//    This code is Copyright (C) 2020 by Ross Martin.

//
//  This module is a BxBFFT adapter with input using the AXI4 Streaming
//  protocol.  The protocol is followed except in regards to flow control.  
//
//  If FULL_FORWARD_FLOW_CONTROL is 0, this code expects that data will not stop flowing 
//  once data starts, until tlast.  This corresponds to a BxBFFT with no
//  support for the valid flag.
//
//  If FULL_FORWARD_FLOW_CONTROL is 1, this code passes the valid flag through.
//
//  Parameters are:
//
//      POINTS_PER_CLOCK          -- The number of BxBFFT complex output points (samples).
//      S_AXIS_SAMPLE_WIDTH       -- Bit width and bit separation of input AXIS data.
//      S_AXIS_SHIFT_RIGHT        -- Bits to shift right to adapt input data to BxBFFT data.
//      REAL_WIDTH                -- Bit width of internal BxBFFT data.
//      TID_WIDTH                 -- Select width of AXIS ID bits.  (0 to disable TID).
//      TDEST_WIDTH               -- Select width of AXIS DEST bits.  (0 to disable TDEST).
//      FULL_FORWARD_FLOW_CONTROL -- Support flow control on the AXIS input and BxB output
//
//  These parameters derived from those above determine bit widths of interfaces:
//
//      S_AXIS_BITS               -- Bit width of input data.  (Will be an even number of bytes).
//      S_AXIS_BYTES              -- Byte width of input data.
//      TID_DUMMY_WIDTH           -- Bit width of TID, or 1 if TID is disabled.
//      TDEST_DUMMY_WIDTH         -- Bit width of TDEST, or 1 if TDEST is disabled.
//
// bxb_data_valid_o is synced in timing with bxb_sync_o, which precedes bxb_data_o by
// RESET_ADVANCEMENT clocks.
//


`undef w
`define w assign
`undef r
`define r always@(posedge clk)
`timescale  1 ps / 1 ps

module axis_to_bxb
  #(
    parameter POINTS_PER_CLOCK           = 1,
    parameter S_AXIS_SAMPLE_WIDTH        = 16,
    parameter S_AXIS_SHIFT_RIGHT         = 5,
    parameter REAL_WIDTH                 = 27,
    parameter TID_WIDTH                  = 0,
    parameter TDEST_WIDTH                = 0,
    parameter RESET_ADVANCEMENT          = 16,
    parameter FULL_FORWARD_FLOW_CONTROL  = 0,

    ///////////////////////////////////////////////////////////////////////////////
    // Parameters that are derived but necessary to determine interface widths.

    parameter REALS_PER_COMPLEX          = 2,
    parameter ITEM_NUMBER                = REALS_PER_COMPLEX * POINTS_PER_CLOCK,
    parameter S_AXIS_USED_BITS           = ITEM_NUMBER * S_AXIS_SAMPLE_WIDTH,
    parameter S_AXIS_BYTES               = (S_AXIS_USED_BITS + 7) / 8,
    parameter S_AXIS_BITS                = S_AXIS_BYTES * 8,
    parameter TID_DUMMY_WIDTH            = (TID_WIDTH==0) ? 1 : TID_WIDTH,
    parameter TDEST_DUMMY_WIDTH          = (TDEST_WIDTH==0) ? 1 : TDEST_WIDTH,
    parameter NUM_SYNC                   = 1 + TID_WIDTH + TDEST_WIDTH
   )
   (
    input                          clk,
    input                          resetn,

    /////////////////////////////////////////////////////////////////////////////////////////////////
    // Input, AXI4 Slave Stream.  Once tvalid goes high, it must stay high until the end of the
    // FFT input data.

    input                          s_axis_tvalid,
    output                         s_axis_tready,
    input [S_AXIS_BITS-1:0]        s_axis_tdata,
    input                          s_axis_tlast,
    input [S_AXIS_BYTES-1:0]       s_axis_tstrb,
    input [S_AXIS_BYTES-1:0]       s_axis_tkeep,
    input [TID_DUMMY_WIDTH-1:0]    s_axis_tid,
    input [TDEST_DUMMY_WIDTH-1:0]  s_axis_tdest,

    //
    // BxB stream signals.  clk and resetn are also used for output as for input.
    //
    output [NUM_SYNC-1:0]          bxb_sync_o,
    output                         bxb_data_valid_o,
    output signed [REAL_WIDTH-1:0] bxb_data_o[0:ITEM_NUMBER-1]
   );

   wire [NUM_SYNC-1:0]                    wc_sync_w;
   wire                                   wc_data_valid_w;
   wire signed [S_AXIS_SAMPLE_WIDTH-1:0]  wc_data_w[0:ITEM_NUMBER-1];

   reg [NUM_SYNC-1:0]                     bxb_sync_r         = 0;
   
   wire signed [S_AXIS_BITS-1:0]          s_axis_tdata_d;

   `w s_axis_tready = 1'b1;

   genvar                           i;
   
   /////////////////////////////////////////////////////////////////////////////////////////////
   // Set up bxb sync in three pieces.  Since two pieces are optional, need to avoid issues
   // with zero-width signals.

   generate

      if(FULL_FORWARD_FLOW_CONTROL)
        begin : full_forward_flow_control           
           `r bxb_sync_r[0] <= ( !resetn                          ?  1  :
                                 !s_axis_tready                   ?  1  :
                                 s_axis_tlast && s_axis_tvalid    ?  1  :
                                 /**/                                0  );

           `w wc_data_valid_w = s_axis_tvalid;
        end
      else
        begin : not_full_forward_flow_control
           `r bxb_sync_r[0] <= ( !resetn          ?  1  :
                                 !s_axis_tready   ?  1  :
                                 !s_axis_tvalid   ?  1  :
                                 s_axis_tlast     ?  1  :
                                 /**/                0  );
           `w wc_data_valid_w = 1;
        end // else: !if(FULL_FORWARD_FLOW_CONTROL)

           
   if(TID_WIDTH>0)
     begin : has_tid_width
        `r bxb_sync_r[TID_WIDTH-1+1:1] <= s_axis_tid;
     end
      
      
   if(TDEST_WIDTH>0)
     begin : has_tdest_width
        `r bxb_sync_r[TDEST_WIDTH-1+1+TID_WIDTH:1+TID_WIDTH] <= s_axis_tdest;
     end

   `w wc_sync_w = bxb_sync_r;


   /////////////////////////////////////////////////////////////////////////////////////////////
   // Set up bxb data
   
   delay #(.DELAY(RESET_ADVANCEMENT), .BIT_WIDTH(S_AXIS_BITS)) alignment_delay ( .clk(clk), .in_i(s_axis_tdata), .out_o(s_axis_tdata_d) );

   for(i=0; i<ITEM_NUMBER; i=i+1)
     begin : output_format_change
     `w wc_data_w[i] = s_axis_tdata_d[S_AXIS_SAMPLE_WIDTH*i +: S_AXIS_SAMPLE_WIDTH];
     end
      
   width_convert #( 
                    .INPUT_WIDTH   ( S_AXIS_SAMPLE_WIDTH  ),
                    .OUTPUT_WIDTH  ( REAL_WIDTH           ),
                    .SHIFT_RIGHT   ( S_AXIS_SHIFT_RIGHT   ),
                    .ITEM_NUMBER   ( ITEM_NUMBER          ),
                    .NUM_SYNC      ( NUM_SYNC             )
                  )
   wc             (
                    .clk           ( clk                  ),
                    .resetn        ( resetn               ),
                    .in_sync_i     ( wc_sync_w            ),
                    .in_valid_i    ( wc_data_valid_w      ),
                    .in_i          ( wc_data_w            ),
                    .out_sync_o    ( bxb_sync_o           ),
                    .out_valid_o   ( bxb_data_valid_o     ),
                    .out_o         ( bxb_data_o           )
                  );

   endgenerate


endmodule

