//
// Copyright (C) 2021 Ross Martin
//
// Versions
//
// 2021.11.10  Coding begun

`undef w
`define w assign
`undef r
`define r always@(posedge clk)
`timescale  1 ps / 1 ps   

//
// Take an input value in a larger number of bits and remove the top bits.  This may result
// in a value exceeding the smaller output bit width's range.  If so, saturate to max value
// positive or negative, as appropriate.
//    
module truncate_with_saturate
  #(
    parameter   INPUT_WIDTH             = 25,
    parameter   OUTPUT_WIDTH            = 25
   )
   (
    input                                clk,
    input signed [INPUT_WIDTH-1:0]       in_i,
    output reg signed [OUTPUT_WIDTH-1:0] out_ro
   );

   initial out_ro = 0;
   
   generate

      if(INPUT_WIDTH<OUTPUT_WIDTH)
        begin : input_width_smaller
           error_module error_cannot_truncate_going_up_in_bits(clk);
        end
      else if(INPUT_WIDTH==OUTPUT_WIDTH)
        begin : widths_equal
           `r out_ro <= in_i;
        end
      else
        begin : output_width_smaller

           // Positive overflow when all bits shifted off aren't zero, or the remaining new sign bit isn't also zero.
           // Negative overflow when all bits shifted off aren't one, or the remaining new sign bit isn't also one.
           // Negative overflow also when the remaining non-sign bits "bottom_bits" are all zero, since leaving this 
           // in would result in non-symmetric saturation.
           // i.e.
           // If there was overflow, saturate to the same value positive as negative (don't introduce a saturation bias)
           // Never let the negative value exceed this max.
      
           localparam EXTRA_BITS = INPUT_WIDTH - OUTPUT_WIDTH;
     
           wire                              sign_bit_w;
           wire [EXTRA_BITS:0]               top_bits_w;
           wire [EXTRA_BITS:0]               top_bits_all_zero_w;
           wire [EXTRA_BITS:0]               top_bits_all_one_w;
           wire [OUTPUT_WIDTH-2:0]           bottom_bits_w;
           wire                              overflow_positive_w;
           wire                              overflow_negative_w;
           wire [OUTPUT_WIDTH-1:0]           max_positive_w;
           wire [OUTPUT_WIDTH-1:0]           max_negative_w;

           `w top_bits_all_zero_w = 0;
           `w top_bits_all_one_w  = ~top_bits_all_zero_w;
           
           `w sign_bit_w    =  in_i[INPUT_WIDTH-1];
           `w top_bits_w    =  in_i[INPUT_WIDTH-1:OUTPUT_WIDTH-1];
           `w bottom_bits_w =  in_i[OUTPUT_WIDTH-2:0];

           `w overflow_positive_w =   (sign_bit_w==0) && (top_bits_w != top_bits_all_zero_w);
           `w overflow_negative_w = ( (sign_bit_w==1) && (top_bits_w != top_bits_all_one_w) ) ||
                                    ( (sign_bit_w==1) && (bottom_bits_w == 0) );

           `w max_positive_w = { 1'b0, {(OUTPUT_WIDTH-2){1'b1}}, 1'b1 };
           `w max_negative_w = { 1'b1, {(OUTPUT_WIDTH-2){1'b0}}, 1'b1 };
           
           `r out_ro <= overflow_positive_w  ?  max_positive_w          :
                        overflow_negative_w  ?  max_negative_w          :
                        /**/                    in_i[OUTPUT_WIDTH-1:0]  ;
                                                 
        end

   endgenerate            

endmodule // truncate_with_saturate


