
/*
 * Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., http://www.fsf.org/about/contact/
 *
 */


#include "expression.hh"
#include "parse_support.hh"
#include <ctype.h>
#include <stdio.h>




void Value::setbits(int numbits_in)
{
  if(numbits==AMBIGUOUS_BITS)
    numbits = numbits_in;
}


String valueToString(Value v)
{
  String s;
  char temp[100];

  if(v.value<0 && v.base==10)
    {
      v.value = -v.value;
      s << "-";
    }

  if(v.numbits==AMBIGUOUS_BITS)
    {
      sprintf(temp, "%ld", v.value);
      s << temp;
    }
  else
    {
      if(v.base==10)
	{
	  sprintf(temp, "%d'd%ld", v.numbits, v.value);
	  s << temp;
	}
      else if(v.base==16)
	{
	  sprintf(temp, "%d'h%lx", v.numbits, v.value);
	  s << temp;
	}
      else if(v.base==8)
	{
	  sprintf(temp, "%d'o%lo", v.numbits, v.value);
	  s << temp;
	}
      else if(v.base==2)
	{
	  sprintf(temp, "%d'b", v.numbits);
	  s << temp;
	  bool zeroprintflag = false;
	  for(int i=sizeof(long)*8-1; i>=0; i--)
	    {
	      if(v.value&(1l<<i))
		{
		  s << "1";
		  zeroprintflag = true;
		}
	      else
		{
		  if(zeroprintflag)
		    s << "0";
		}
	    }
	  if(!zeroprintflag)
	    {
	      s << "0";
	    }
	}
      else
	{
	  printf("Invalid base %d in printValue()\n", v.base);
	  exit(20);
	}
    }

  return s;
}


void printValue(Value v)
{
  printf("%s", (char*)valueToString(v));
}

Value readValue(readstream& a)
{
  char buff[100];
  int c;
  int negative = 0;

  buff[0]=0;

  c = skipCommentsVerilog(a);

  if(c=='-')
    {
      negative=1;
      a.getchar();
    }

  c = readStringKnownChars(a, buff, "0123456789", 100);

  if(c==100)
    {
      printf("Size exceeded in readValue\n");
      parse_error(a);
    }

  if(buff[0]==0)
    {
      printf("Invalid integer!\n");
      parse_error(a);
    }

  c = skipCommentsVerilog(a);

  if(c=='`')
    {
      printf("Using the wrong tick (`) in a Verilog constant when it should be a (')\n");
      parse_error(a);
    }

  Value v;
  v.valid = true;

  if(c!='\'')
    {
      v.numbits = AMBIGUOUS_BITS;
      v.value = atol(buff);
      v.base = 10;

      c = a.getchar();
      a.ungetchar();
      if(isalnum(c))
	{
	  printf("Hex constant \"%s\" followed by unexpected character '%c'\n", (char*)valueToString(v), c);
	  parse_error(a);
	}
    }
  else
    {
      v.numbits = atoi(buff);
      
      a.getchar();
      c = a.getchar();
      if(c=='d'||c=='D')
	{
	  // Decimal
	  c = readStringKnownChars(a, buff, "0123456789", 100);
	  if(c==100)
	    {
	      printf("Size exceeded reading decimal value in readValue\n");
	      parse_error(a);
	    }
	  v.value = atol(buff);
	  v.base = 10;

	  c = a.getchar();
	  a.ungetchar();
	  if(isalnum(c))
	    {
	      printf("Hex constant \"%s\" followed by unexpected character '%c'\n", (char*)valueToString(v), c);
	      parse_error(a);
	    }
	}
      else if(c=='h'||c=='H')
	{
	  // Hex
	  c = readStringKnownChars(a, buff, "0123456789abcdefABCDEF", 100);
	  if(c==100)
	    {
	      printf("Size exceeded reading hex value in readValue\n");
	      parse_error(a);
	    }

	  v.value = 0;
	  char* b = buff;
	  while(*b)
	    {
	      char x = *b; 
	      int vv = ( ('0'<=x && x<='9') ? x-'0'    :
			 ('a'<=x && x<='f') ? x-'a'+10 :
			 /**/                 x-'A'+10 );
	      
	      v.value = 16*v.value + vv;
	      b++;
	    }
	  v.base = 16;

	  c = a.getchar();
	  a.ungetchar();
	  if(isalnum(c))
	    {
	      printf("Hex constant \"%s\" followed by unexpected character '%c'\n", (char*)valueToString(v), c);
	      parse_error(a);
	    }
	}
      else if(c=='b'||c=='B')
	{
	  // Binary
	  c = readStringKnownChars(a, buff, "01", 100);
	  if(c==100)
	    {
	      printf("Size exceeded reading binary value in readValue\n");
	      parse_error(a);
	    }

	  v.value = 0;
	  char* b = buff;
	  while(*b)
	    {
	      int vv = (*b)-'0';
	      v.value = 2*v.value + vv;
	      b++;
	    }
	  v.base = 2;

	  c = a.getchar();
	  a.ungetchar();
	  if(isalnum(c))
	    {
	      printf("Binary constant \"%s\" followed by unexpected character '%c'\n", (char*)valueToString(v), c);
	      parse_error(a);
	    }
	}
      else if(c=='o'||c=='O')
	{
	  // Octal
	  c = readStringKnownChars(a, buff, "01234567", 100);
	  if(c==100)
	    {
	      printf("Size exceeded reading octal value in readValue\n");
	      parse_error(a);
	    }	  

	  v.value = 0;
	  char* b = buff;
	  while(*b)
	    {
	      int vv = (*b)-'0';
	      v.value = 8*v.value + vv;
	      b++;
	    }
	  v.base = 8;

	  c = a.getchar();
	  a.ungetchar();
	  if(isalnum(c))
	    {
	      printf("Octal constant \"%s\" followed by unexpected character '%c'\n", (char*)valueToString(v), c);
	      parse_error(a);
	    }
	}
      else
	{
	  printf("\nWhen reading constant, expecting 'd', 'h', 'b' or 'o' after the '\n");
	  a.ungetchar();
	  parse_error(a);
	}
    }

  if(v.numbits != AMBIGUOUS_BITS)
    {
      int nb = 1;
      for(unsigned int i=0; i<sizeof(long)*8-1; i++)
	{
	  if(v.value&(1l<<i))
	    nb = i+1;
	}

      if(negative)
	{
	  if(nb+1>v.numbits)
	    {
	      v.value = -v.value;
	      printf("Negative constant \"%s\" requires %d bits, but only %d bits are specified.\n", 
		     (char*)valueToString(v),
		     nb+1, 
		     v.numbits);
	      parse_error(a);
	    }
	}
      else
	{
	  if(nb>v.numbits)
	    {
	      printf("Numeric constant \"%s\" requires %d bits, but only %d bits are specified.\n", 
		     (char*)valueToString(v),
		     nb, 
		     v.numbits);
	      parse_error(a);
	    }
	}
    }
  
  if(negative)
    v.value = -v.value;

  //printf("Exiting readValue with value %s, bits %d\n", (char*)valueToString(v), v.numbits);

  return v;
}


expressionOperator* operatorMatch(readstream& r, expressionOperator* ops)
{
  expressionOperator* eo;

  for(eo=ops;eo->type;eo++)
    if(matchStringI(r, eo->token))
      return eo;

  return 0;
}

String variableMatch(readstream& r, variable_check_function* vcf, void* data, bool clock_ok)
{
  char temp[200];
  int num;
  String s;

  num = readString(r, temp, " \t\n\r+-=*/^#!@$%&();:.,<>{}|~[]", 200);
  if(num==200)
    {
      printf("String length exceeded in expression::expression!\n");
      parse_error(r);
    }

  if(vcf(temp, data, r, clock_ok))
    {
      s = temp;
      return s;
    }

  r.ungetchars(num);

  return s;
}


expression* parse_unary_expression(readstream&              r, 
				   variable_check_function* vcf, 
				   expressionOperator*      ops,
				   expressionOperator*      uops,
				   void*                    data,
				   bool                     clock_ok)
{
  int c;
  expression* e=0;
  expressionOperator* eo;

  //
  // Read next value.  
  //
  c = skipChars(r, " \t\r\n");

  if(isdigit(c) || c=='.')
    {
      // We get here if it is a numeric constant.

      e = new expression();
      e->data = data;
      e->type = EXPRESSION_CONSTANT;
      e->value = readValue(r);
    }
  else
    {
      // We get here if it is not a numeric constant.

      eo = operatorMatch(r, uops);
      if(eo)
	{
	  // We get here if it is a known operator from the list.

	  if(eo->type == EXPRESSION_BEGINGROUP)
	    {
	      e = new expression();
	      e->data = data;
	      e->type = EXPRESSION_BEGINGROUP;
	      e->exp1 = parse_expression(r, vcf, ops, uops, data, false, 0);
	      eo = operatorMatch(r, ops);
	      if(eo->type!=EXPRESSION_ENDGROUP)
		{
		  int c = r.getchar();
		  r.ungetchar();
		  printf("\n\nExpecting end of group ')', but got '%c' instead.\n", c);
		  parse_error(r);
		}
	    }
	  else
	    {
	      //
	      // Parse unary operator argument at its new priority.
	      //

	      if(eo->parse_function)
		{
		  e = eo->parse_function(r, vcf, ops, uops, data, eo, 0);
		}
	      else
		{
		  e = new expression();
		  e->data = data;
		  e->type = eo->type;
		  e->exp1 = parse_expression(r, vcf, ops, uops, data, false, eo->priority);
		}
	    }
	}
      else
	{
	  String name = variableMatch(r, vcf, data, clock_ok);
	  if(((char*)name)[0])
	    {
	      // We get here if it is a valid variable.
	      
	      e = new expression();
	      e->data = data;
	      e->name = name;
	      e->type = EXPRESSION_VARIABLE;
	      e->value.numbits = vcf((char*)name, data, r, clock_ok);
	      //printf("variable %s has numbits %d\n", (char*)e->name, e->value.numbits);
	    }
	  else
	    {
	      char temp[201];
	      int num = readString(r, temp, " \t\n\r+-=*/^#!@$%&();:.,<>{}|~[]", 200);
	      if(num==200)
		{
		  printf("\n\nString length exceeded in expression::parse_unary_expression!\n");
		  parse_error(r);
		}

	      if(num==0)
		{
		  num=1;
		  temp[0] = r.getchar();
		  temp[1] = 0;
		}

	      r.ungetchars(num);

	      printf("\n\nError parsing expression.  Can't understand \"%s\" or\n"
		     "match it to a valid variable.\n", temp);
	      parse_error(r);
	    }
	}
    }

  if(e==0)
    {
      printf("Unexpected failure of expression parser  -- DEBUG ME.\n");
      parse_error(r);
    }

  Value v = getValue(e, ops, uops);

  if(v.numbits==0)
    {
      printf("Expression has an error; details should have been printed above.\n");
      parse_error(r);
    }


  return e;
}

//
// ops is binary operators.  This includes the final terminator.
//
// uops is unary operators.  This includes begin group.
//
// If priority is even, the operator binds left to right, like subtraction,
// which is also called "left associative".  If it is odd, the operator 
// binds right to left, like the = operator in C and C++, which is also
// called right associative.
//
// For most operators, priority values should be EVEN.
//
expression* parse_expression(readstream&              r, 
			     variable_check_function* vcf, 
			     expressionOperator*      ops,
			     expressionOperator*      uops,
			     void*                    data,
			     bool                     clock_ok,
			     int                      priority)
{
  expressionOperator* eo;
  expression* e;

  //
  // Start by looking for a UNARY expression, such as a constant,
  // a variable, a bitwise or logical NOT with its argument, or 
  // a group with its contents.
  //

  e = parse_unary_expression(r, vcf, ops, uops, data, clock_ok); 

  for(;;)
    {
      // Continue parsing to find next operator.  If it is at a higher
      // priority than we are currently operating at, we need to combine
      // our current expression with it before we return.
      
      skipChars(r, " \t\r\n");
      eo = operatorMatch(r, ops);
      if(!eo)
	{
	  int c = r.getchar();
	  r.ungetchar();
	  printf("\n\nUnidentified operator beginning with '%c'.\n", c);
	  parse_error(r);
	}

      //
      // If we just got the expression terminator, return with the
      // expression we have.
      //
      if(eo->type == EXPRESSION_TERMINATOR)
	{
	  break;
	}

      //
      // If we just got the group terminator, return with the
      // expression we have.
      //
      if(eo->type == EXPRESSION_ENDGROUP)
	{
	  break;
	}

      //
      // If the priority level of the new operator is less than our
      // current priority, we are done.  The next level out of the
      // parser will handle the rest of the line.  So back put the
      // operator name back out to parse, and return with what we have.
      //
      if(eo->priority<priority)
	{
	  break;
	}

      //
      // Left associative operators terminate on identical
      // priority of the next operator.
      //
      if(eo->priority==priority && (priority&1)==0)
	break;

#ifdef DEBUG_EXPRESSIONS
      printf("Operator %s, pri=%d, opri=%d\n", eo->token, priority, eo->priority);
#endif
      
      if(eo->parse_function)
	{
	  e = eo->parse_function(r, vcf, ops, uops, data, eo, e);
	}
      else
	{
	  expression* e2 = parse_expression(r, vcf, ops, uops, data, false, eo->priority);

	  expression* enew = new expression();
	  enew->data = data;
	  enew->exp1 = e;
	  enew->exp2 = e2;
	  enew->type = eo->type;
	  e = enew;
	}
    }

  r.ungetchars(strlen(eo->token));


//
// If this expression is a group, and the priority of the outer operator in the
// group is higher than the current priority, the group can be removed.
//

  while(e->type == EXPRESSION_BEGINGROUP)
    {
      eo = getOperator(e->exp1->type, ops);
      if(!eo)
	eo = getOperator(e->exp1->type, uops);
      if(!eo)
	{
	  printf("Can't determine operator inside group to determine if group can be removed.\n");
	  parse_error(r);
	}

//
// Actually, equality here may be technically incorrect in the case of operators
// that are non-commutative.  However, the operators we deal with are commutative.
//
      if(eo->priority>=priority)
	{
// Remove outer group
	  expression* e3 = e;
	  e = e3->exp1;
	  e3->exp1 = 0;
	  delete e3;
	}
      else
	{
	  break;
	}
    }

  Value v = getValue(e, ops, uops);

  if(v.numbits==0)
    {
      printf("Expression has an error; details should have been printed above.\n");
      parse_error(r);
    }

  return e;
}

expressionOperator* getOperator(int type, expressionOperator* oe)
{
  expressionOperator* x;

  for(x=oe; x->type; x++)
    if(x->type==type)
      return x;

  return 0;
}

Value getValue(expression* e,
	       expressionOperator* oe, 
	       expressionOperator* uoe)
{
  expressionOperator* x;

  if(e->type==EXPRESSION_NULL)
    {
      Value v;
      v.valid = false;
      v.numbits = 0;
      v.value = 0;
      return v;
    }

  x=getOperator(e->type, oe);
  if(x)
    return (x->function)(e, oe, uoe);

  x=getOperator(e->type, uoe);
  if(x)
    return (x->function)(e, oe, uoe);

  printf("In expression.cc getValue():  Cannot find operator of type 0x%x\n", e->type);
  exit(10);
}

String expressionToString(expression* e, 
			  expressionOperator*      ops,
			  expressionOperator*      uops)
{
  String s;

  //printf("e->type=0x%08x\n", e->type);

  if(e->type==EXPRESSION_NULL)
    return s;

  expressionOperator* eo  = getOperator(e->type, ops);
  expressionOperator* ueo = getOperator(e->type, uops);

  if(eo && eo->string_function)
    {
      s = (eo->string_function)(e, ops, uops);
    }
  else if(ueo && ueo->string_function)
    {
      s = (ueo->string_function)(e, ops, uops);
    }
  else if(e->type==EXPRESSION_CONSTANT)
    {
      s << valueToString(e->value);
    }
  else if(e->type==EXPRESSION_VARIABLE)
    {
      s << e->name;
    }
  else
    {
      if(eo)
	{
	  s << expressionToString(e->exp1, ops, uops);
	  //s << " " << eo->token << " ";
	  s << eo->token;
	  s << expressionToString(e->exp2, ops, uops);
	}
      else
	{
	  if(!ueo)
	    {
	      printf("in expression.cc expressionToString(): Unknown operator of type 0x%x\n", e->type);
	      exit(10);
	    }
	  s << " " << ueo->token << " ";
	  s << expressionToString(e->exp1, ops, uops);
	  if(e->type==EXPRESSION_BEGINGROUP)
	    {
	      eo = getOperator(EXPRESSION_ENDGROUP, ops);
	      s << " " << eo->token << " ";
	    }
	}
    }

  return s;
}


void stuff_variables(expression* e, 
		     expressionOperator*      ops,
		     expressionOperator*      uops)
{
  expressionOperator* eo;

  if(e->type==EXPRESSION_CONSTANT)
    {
    }
  else if(e->type == EXPRESSION_VARIABLE)
    {
      e->value = getValue(e, ops, uops);
      e->type = EXPRESSION_CONSTANT;
    }
  else
    {
      eo = getOperator(e->type, ops);
      if(eo)
	{
	  if(e->type != EXPRESSION_OPERATOR_ASSIGNMENT)
	    stuff_variables(e->exp1, ops, uops);
	  stuff_variables(e->exp2, ops, uops);
	}
      else
	{
	  eo = getOperator(e->type, uops);
	  if(!eo)
	    {
	      printf("Unknown operator of type 0x%x\n", e->type);
	      exit(10);
	    }
	  stuff_variables(e->exp1, ops, uops);
	}
    }
}

void evaluate_outer_leaf(expression* e, 
			 expressionOperator*      ops,
			 expressionOperator*      uops)
{
  expressionOperator* eo;

  if(e->type==EXPRESSION_CONSTANT)
    {
      return;
    }
  else if(e->type == EXPRESSION_VARIABLE)
    {
      e->value = getValue(e, ops, uops);
      e->type = EXPRESSION_CONSTANT;
    }
  else
    {
      eo = getOperator(e->type, ops);
      if(eo)
	{
	  if(e->type == EXPRESSION_OPERATOR_ASSIGNMENT)
	    {
	      if( e->exp2->type == EXPRESSION_CONSTANT )
		{
		  e->value = getValue(e, ops, uops);
		  e->type = EXPRESSION_CONSTANT;
		  delete e->exp1;
		  delete e->exp2;
		  e->exp1 = e->exp2 = 0;
		}
	      else
		{
		  evaluate_outer_leaf(e->exp2, ops, uops);
		}
	    }
	  else if(e->exp1->type == EXPRESSION_CONSTANT &&
		  e->exp2->type == EXPRESSION_CONSTANT )
	    {
	      e->value = getValue(e, ops, uops);
	      e->type = EXPRESSION_CONSTANT;
	      delete e->exp1;
	      delete e->exp2;
	      e->exp1 = e->exp2 = 0;
	    }
	  else
	    {
	      evaluate_outer_leaf(e->exp1, ops, uops);
	      evaluate_outer_leaf(e->exp2, ops, uops);
	    }
	}
      else
	{
	  eo = getOperator(e->type, uops);
	  if(!eo)
	    {
	      printf("Unknown operator of type 0x%x\n", e->type);
	      exit(10);
	    }

	  if(e->exp1->type == EXPRESSION_CONSTANT)
	    {
	      e->value = getValue(e, ops, uops);
	      e->type = EXPRESSION_CONSTANT;
	      delete e->exp1;
	      e->exp1 = 0;
	    }
	  else
	    {
	      evaluate_outer_leaf(e->exp1, ops, uops);
	    }
	}
    }
}

void
write(writestream& w,  
      expression* e,
      expressionOperator*      ops,
      expressionOperator*      uops)
{
  expressionOperator* eo;

  if(e->type==EXPRESSION_CONSTANT)
    {
      w << ' ' << valueToString(e->value);
    }
  else if(e->type==EXPRESSION_VARIABLE)
    {
      w << ' ' << e->name;
    }
  else
    {
      eo = getOperator(e->type, ops);
      if(eo)
	{
	  write(w, e->exp1, ops, uops);
	  w << ' ' << eo->token;
	  write(w, e->exp2, ops, uops);
	}
      else
	{
	  eo = getOperator(e->type, uops);
	  if(!eo)
	    {
	      printf("Unknown operator of type 0x%x\n", e->type);
	      exit(10);
	    }
	  w << ' ' << eo->token;
	  write(w, e->exp1, ops, uops);
	  if(e->type==EXPRESSION_BEGINGROUP)
	    {
	      eo = getOperator(EXPRESSION_ENDGROUP, ops);
	      w << ' ' << eo->token;
	    }
	}
    }
}

void
write_nospaces(writestream& w,  
	       expression* e,
	       expressionOperator*      ops,
	       expressionOperator*      uops)
{
  expressionOperator* eo;

  if(e->type==EXPRESSION_CONSTANT)
    {
      w << valueToString(e->value);
    }
  else if(e->type==EXPRESSION_VARIABLE)
    {
      w << e->name;
    }
  else
    {
      eo = getOperator(e->type, ops);
      if(eo)
	{
	  write_nospaces(w, e->exp1, ops, uops);
	  w << eo->token;
	  write_nospaces(w, e->exp2, ops, uops);
	}
      else
	{
	  eo = getOperator(e->type, uops);
	  if(!eo)
	    {
	      printf("Unknown operator of type 0x%x\n", e->type);
	      exit(10);
	    }
	  w << eo->token;
	  write_nospaces(w, e->exp1, ops, uops);
	  if(e->type==EXPRESSION_BEGINGROUP)
	    {
	      eo = getOperator(EXPRESSION_ENDGROUP, ops);
	      w << eo->token;
	    }
	}
    }
}


expression::expression(expression& e)
{
  exp1 = exp2 = 0;

  if(e.exp1)
    exp1 = new expression(*e.exp1);

  if(e.exp2)
    exp2 = new expression(*e.exp2);

  if(e.exp_n)
    {
      exp_n = new expression*[e.exp_count];
      exp_count = e.exp_count;
      for(int i=0; i<exp_count; i++)
	exp_n[i] = e.exp_n[i];
    }

  data = e.data;
  type = e.type;
  value = e.value;
  name = e.name;
}


void write_inner(char*& bigbuff,
		 int& bigbuffsize,
		 expression* e, 
		 expressionOperator*      ops,
		 expressionOperator*      uops)
{
  expressionOperator* eo;
  int n;

  if(e->type==EXPRESSION_CONSTANT)
    {
      n = snprintf(bigbuff, bigbuffsize, "%s ", (char*)valueToString(e->value));
      bigbuffsize -= n;
      bigbuff += n;
      if(bigbuffsize<=0)
	{
	  printf("Buffer size not big enough in expression write.\n");
	  exit(20);
	}
    }
  else if(e->type==EXPRESSION_VARIABLE)
    {
      n = snprintf(bigbuff, bigbuffsize, "%s ", (char*)e->name);
      bigbuffsize -= n;
      bigbuff += n;
      if(bigbuffsize<=0)
	{
	  printf("Buffer size not big enough in expression write.\n");
	  exit(20);
	}
    }
  else
    {
      eo = getOperator(e->type, ops);
      if(eo)
	{
	  write_inner(bigbuff, bigbuffsize, e->exp1, ops, uops);
	  n = snprintf(bigbuff, bigbuffsize, "%s ", eo->token);
	  bigbuffsize -= n;
	  bigbuff += n;
	  if(bigbuffsize<=0)
	    {
	      printf("Buffer size not big enough in expression write.\n");
	      exit(20);
	    }

	  write_inner(bigbuff, bigbuffsize, e->exp2, ops, uops);
	}
      else
	{
	  eo = getOperator(e->type, uops);
	  if(!eo)
	    {
	      printf("Unknown operator of type 0x%x\n", e->type);
	      exit(10);
	    }
	  n = snprintf(bigbuff, bigbuffsize, "%s ", eo->token);
	  bigbuffsize -= n;
	  bigbuff += n;
	  if(bigbuffsize<=0)
	    {
	      printf("Buffer size not big enough in expression write.\n");
	      exit(20);
	    }

	  write_inner(bigbuff, bigbuffsize, e->exp1, ops, uops);
	  if(e->type==EXPRESSION_BEGINGROUP)
	    {
	      eo = getOperator(EXPRESSION_ENDGROUP, ops);
	      n = snprintf(bigbuff, bigbuffsize, "%s ", eo->token);
	      bigbuffsize -= n;
	      bigbuff += n;
	      if(bigbuffsize<=0)
		{
		  printf("Buffer size not big enough in expression write.\n");
		  exit(20);
		}
	    }
	}
    }
}

void write(char* bigbuff,
	   int bigbuffsize,
	   expression* e, 
	   expressionOperator*      ops,
	   expressionOperator*      uops)
{
  char* bb = bigbuff;
  int bbs = bigbuffsize;

  write_inner(bb, bbs, e, ops, uops);
}


void write_nospaces_inner(char*& bigbuff,
			  int& bigbuffsize,
			  expression* e, 
			  expressionOperator*      ops,
			  expressionOperator*      uops)
{
  expressionOperator* eo;
  int n;

  if(e->type==EXPRESSION_CONSTANT)
    {
      n = snprintf(bigbuff, bigbuffsize, "%s", (char*)valueToString(e->value));
      bigbuffsize -= n;
      bigbuff += n;
      if(bigbuffsize<=0)
	{
	  printf("Buffer size not big enough in expression write.\n");
	  exit(20);
	}
    }
  else if(e->type==EXPRESSION_VARIABLE)
    {
      n = snprintf(bigbuff, bigbuffsize, "%s", (char*)e->name);
      bigbuffsize -= n;
      bigbuff += n;
      if(bigbuffsize<=0)
	{
	  printf("Buffer size not big enough in expression write.\n");
	  exit(20);
	}
    }
  else
    {
      eo = getOperator(e->type, ops);
      if(eo)
	{
	  write_nospaces_inner(bigbuff, bigbuffsize, e->exp1, ops, uops);
	  n = snprintf(bigbuff, bigbuffsize, "%s", eo->token);
	  bigbuffsize -= n;
	  bigbuff += n;
	  if(bigbuffsize<=0)
	    {
	      printf("Buffer size not big enough in expression write.\n");
	      exit(20);
	    }

	  write_nospaces_inner(bigbuff, bigbuffsize, e->exp2, ops, uops);
	}
      else
	{
	  eo = getOperator(e->type, uops);
	  if(!eo)
	    {
	      printf("Unknown operator of type 0x%x\n", e->type);
	      exit(10);
	    }
	  n = snprintf(bigbuff, bigbuffsize, "%s", eo->token);
	  bigbuffsize -= n;
	  bigbuff += n;
	  if(bigbuffsize<=0)
	    {
	      printf("Buffer size not big enough in expression write.\n");
	      exit(20);
	    }

	  write_nospaces_inner(bigbuff, bigbuffsize, e->exp1, ops, uops);
	  if(e->type==EXPRESSION_BEGINGROUP)
	    {
	      eo = getOperator(EXPRESSION_ENDGROUP, ops);
	      n = snprintf(bigbuff, bigbuffsize, "%s", eo->token);
	      bigbuffsize -= n;
	      bigbuff += n;
	      if(bigbuffsize<=0)
		{
		  printf("Buffer size not big enough in expression write.\n");
		  exit(20);
		}
	    }
	}
    }
}

void write_nospaces(char* bigbuff,
		    int bigbuffsize,
		    expression* e, 
		    expressionOperator*      ops,
		    expressionOperator*      uops)
{
  char* bb = bigbuff;
  int bbs = bigbuffsize;

  if(!e)
    return;

  write_nospaces_inner(bb, bbs, e, ops, uops);
}
