#!/bin/bash

depend_exe()
{
    tool="$1"
    if [ "`which $tool`" = "" ]; then
	echo ""
	echo "Tool \"$tool\" must be built to run this script.  Run exebuild.sh there."
	echo ""
	exit 20
    fi
}

depends()
{
    tool="$1"
    if [ "`which $tool`" = "" ]; then
	echo ""
	echo "Xilinx's tool $tool must be installed to run this script."
	echo ""
	exit 20
    fi
}

IN_XSA="$1"

if [ ! -f "$IN_XSA" ]; then
    echo "Usage: $0 XSA_FILE.xsa"
    exit 20
fi

rm -rf device-tree-xlnx

if [ ! -f device-tree-xlnx-xilinx_v2024.2.tgz ]; then
    echo ""
    echo ""
    echo "  Have no local copy of device-tree-xlnx.  Fetching and creating local copy."
    echo ""
    echo ""
    git clone https://github.com/Xilinx/device-tree-xlnx
    (cd device-tree-xlnx; git checkout xilinx_v2024.2)
    tar czf device-tree-xlnx-xilinx_v2024.2.tgz device-tree-xlnx
else
    echo ""
    echo ""
    echo "  Extracting device-tree-xlnx from local tar file."
    echo ""
    echo ""
    tar xzf device-tree-xlnx-xilinx_v2024.2.tgz
fi


OUTDIR=out
TMPDIR=tmp

rm -rf "$TMPDIR"

DESIGN=`basename "$IN_XSA" | sed 's/[.].*//'`
DTS="$OUTDIR/${DESIGN}.dts"
DTB="$OUTDIR/${DESIGN}.dtb"
XSA="${DESIGN}.xsa"

if [ ! -f "$DTS" ]; then
    echo "No Device Tree \"$DTS\" exists; building it."
elif [ ! -f "$DTB" ]; then
    echo "No Device Tree \"$DTB\" exists; building it."
else
    if [ "$IN_XSA" -nt "$DTS" ]; then
	echo "Rebuilding \"$DTS\"; the XSA file \"$IN_XSA\" is newer."
    elif [ "$IN_XSA" -nt "$DTB" ]; then
	echo "Rebuilding \"$DTB\"; the XSA file \"$IN_XSA\" is newer."
    else
	echo "Skipping build of \"$DTS\" and \"$DTB\"; the XSA file \"$IN_XSA\" is older."
	exit 0
    fi
fi

depend_exe ../u-boot-xlnx/exe/dtc
depends xsct

rm -f "$DTS" "$DTB"


mkdir -p "$TMPDIR"
mkdir -p "$OUTDIR"


cp "$IN_XSA" "$TMPDIR/$XSA"



cat >"$TMPDIR"/device-tree-conf.yaml <<EOF
{app: {compiler-misc: {add: None}}, bsp: {console_device: {set: psu_uart_1}, dt_setbaud: {set: '115200'}, main_memory: {set: psu_ddr_0}, periph_type_overrides: {set: '{BOARD template}'}, remove_pl: {set: 'TRUE'}}}
EOF

cat >"$TMPDIR"/base-hsi.tcl <<EOF
#requires xsa repo path proc dir yaml-file-location
package require cmdline
package require yaml

proc get_os_config_list {} {
	set prop_data [hsi report_property -all -return_string \\
				[hsi get_os] CONFIG.*]
	set conf_data [split \$prop_data "\n"]
	set config_list {}
	foreach line \$conf_data {
		if { [regexp "^CONFIG..*" \$line matched] == 1 } {
			set config_name [split \$line " "]
			set config_name [lindex \$config_name 0]
			regsub -- "CONFIG." \$config_name "" config_name
			lappend config_list \$config_name
		}
	}
	return \$config_list
}


proc set_properties { yamlconf } {
	set os_config_list [get_os_config_list]
	if { \$yamlconf ne "" } {
		set conf_dict [::yaml::yaml2dict -file \$yamlconf]
		if {[dict exists \$conf_dict "bsp"]} {
			foreach prop [dict keys [dict get \$conf_dict "bsp"]] {
				foreach action [dict keys \\
						[dict get \$conf_dict "bsp" \$prop]] {
					if { [lsearch -exact -nocase \$os_config_list \$prop] < 0} {
						continue
					}
					if { [catch {hsi set_property CONFIG.\$prop \\
						[dict get \$conf_dict "bsp" \$prop \$action] \\
							[hsi get_os]} res] } {
						error "NO BSP configuration available"
					}
				}
			}
		}
	}
}

proc set_hw_design {project hdf hdf_type} {
	file mkdir \$project
	if { [catch { exec cp \$hdf \$project/hardware_description.\$hdf_type } msg] } {
        	puts "\$::errorInfo"
	}

	if { [catch {openhw \$project/hardware_description.\$hdf_type} res] } {
        	error "Failed to open hardware design \\
                       \$project/hardware_description.\$hdf_type"
	}
}
EOF

cat >"$TMPDIR"/dtgen.tcl <<EOF
set dir [file dirname [info script]]
source \$dir/base-hsi.tcl
set option {
	{hdf.arg	""			"hardware Definition file"}
	{hdf_type.arg   "hdf"			"hardware Defination file type: xsa"}
	{processor_ip.arg	""		"target processor_ip"}
	{processor.arg	""			"target processor_ip instance name"}
	{rp.arg		""			"repo path"}
	{app.arg	"empty_application"	"Application project fsbl, empty.."}
	{lib.arg	""			"Add library"}
	{pname.arg	""			"Project Name"}
	{bspname.arg	""			"standalone bsp name"}
	{ws.arg		""			"Work Space Path"}
	{hwpname.arg	""			"hardware project name"}
	{arch.arg	"64"			"32/64 bit architecture"}
	{do_compile.arg	"0"			"Build the project"}
	{forceconf.arg	"0"			"Apply the yaml configs on existing project"}
	{yamlconf.arg	""			"Path to Config File"}
}
set usage "A script to generate and compile device-tree"
array set params [::cmdline::getoptions argv \$option \$usage]
set project "\$params(ws)/\$params(pname)"

set_hw_design \$project \$params(hdf) \$params(hdf_type)

if { [catch {hsi set_repo_path \$params(rp)} res] } {
	error "Failed to set repo path \$params(rp)"
}

if { \$params(processor) ne "" } {
	set processor \$params(processor)
} else {
	set processor [lindex [hsi get_cells -hier -filter IP_NAME==\$params(processor_ip)] 0]
}
if {[catch {hsi create_sw_design \$params(app) \\
		-os device_tree -proc \$processor} res] } {
	error "create_sw_design failed for \$params(app)"
}

#Enable zocl by default when its a expandable xsa
if { [ishwexpandable \$params(hdf)] } {
	hsi set_property CONFIG.dt_zocl true [hsi get_os]
}

if {[file exists \$params(yamlconf)]} {
	set_properties \$params(yamlconf)
}

if {[catch {hsi generate_target -dir \$project} res]} {
	error "generate_target failed"
}
if { [catch {hsi close_hw_design [hsi current_hw_design]} res] } {
	error "Failed to close hw design [hsi current_hw_design]"
}
EOF

(cd "$TMPDIR"; xsct -sdx -nodisp dtgen.tcl -ws . -pname device-tree -rp ../device-tree-xlnx -processor_ip psu_cortexa53 -hdf $XSA -arch 64   -app "device-tree"  -hdf_type xsa -yamlconf device-tree-conf.yaml )

#echo "Before"
#ls -l tmp/device-tree/
#
#cp bsp/zynqmp-clk-ccf.dtsi tmp/device-tree/
#
#echo "After"
#ls -l tmp/device-tree/


cpp -nostdinc -I "$TMPDIR"/device-tree/include -I arch  -undef -x assembler-with-cpp  "$TMPDIR"/device-tree/system-top.dts -o - >$DTS

../u-boot-xlnx/exe/dtc -@ -o "$DTB" "$DTS"

if [ -f "$DTS" -a -f "$DTB" ]; then
    echo ""
    echo "SUCCESS!  Device tree created in \"${DTS}\"."
    echo "          Compiled version in \"${DTB}\"."
    echo ""
    #rm -rf "$TMPDIR" device-tree-xlnx
else
    echo ""
    echo "ERROR!  Couldn't create device tree $DTS of compiled version $DTB."
    echo "Leaving intermediate files for debugging."
    echo ""
fi
