
/*
 * Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., http://www.fsf.org/about/contact/
 *
 */


#include "xml.hh"
#include "parse_support.hh"


xml_entry::xml_entry(readstream& r, bool debug)
{
  int  maxsize=2000;
  char buf[maxsize+1];

  int c = skipCommentsCpp(r);

  if(c!='<')
    {
      printf("Expecting '<' in file \"%s\" at line number %d\n", r.getfilename(), r.getlinenumber());
      parse_error(r);
    }
  
  r.getchar();

  c = skipChars(r, " \t");

  if(!isalpha(c))
    {
      printf("Expecting first character of XML name to be alpha in file \"%s\" at line number %d\n", r.getfilename(), r.getlinenumber());
      parse_error(r);
    }

  readString(r, buf, " ><\n\r>/", maxsize);

  name = buf;

  if(debug)
    printf("Parsing entry \"%s\"\n", buf);
  
  for(;;)
  {
    int c = skipCommentsCpp(r);

    if(c=='>')  // End of attributes, have entries.
      {
	if(debug)
	  printf("Parsing children for \"%s\"\n", (const char*) name);

	r.getchar();
	
	for(;;)
	  {
	    int c = skipCommentsCpp(r);
	    
	    if(c==EOF)
	      {
		printf("Error parsing XML file \"%s\":  got EOF when parsing \"%s\".\n", r.getfilename(), (const char*)name);
		parse_error(r);
	      }

	    // Add one new child
	    auto xmle = new xml_entry(r, debug);
	    children.addAtEnd(xmle, true);

	    skipCommentsCpp(r);

	    if(debug)
	      printf("Checking for term of %s.\n", (const char*) name);
	    
	    // See if next item is an end of this entry.  If so, end it out.  If not, loop back for another child
	    if(matchString(r, "</"))
	      {
		if(debug)
		  printf("Parsing end of children for \"%s\"\n", (const char*) name);
		skipCommentsCpp(r);
		if(!matchString(r,  (const char*)name))
		  {
		    readString(r, buf, "\n\r>", maxsize);
		    printf("Error parsing XML file \"%s\" line %d:  got \"</%s\" when looking for term of \"%s\".\n", r.getfilename(),
			   r.getlinenumber(), buf, (const char*)name);
		    parse_error(r);
		  }
		int c = skipCommentsCpp(r);
		if(c!='>')
		  {
		    printf("Error parsing XML file \"%s\" line %d:  expecting \">\" to terminate \"</%s\".\n", r.getfilename(),
			   r.getlinenumber(), (const char*)name);
		    parse_error(r);
		  }
		r.getchar();
		return;
	      }
	  }
      }
    else if(matchString(r, "/>"))  // End of attributes, no entries
      {
	if(debug)
	printf("No children for \"%s\"\n", (const char*) name);
	return;
      }
    else  // Have an attribute
      {
	if(debug)
	  printf("Parsing an attribute for \"%s\"\n", (const char*) name);
	skipCommentsCpp(r);
	readString(r, buf, "\n\r><=", maxsize);
	skipCommentsCpp(r);
	c = r.getchar();
	if(c!='=')
	  {
	    printf("Error parsing XML file \"%s\" line %d:  expecting \"=\" to define attribute \"%s\".\n", r.getfilename(),
		   r.getlinenumber(), buf);
	    parse_error(r);
	  }

	skipCommentsCpp(r);
	
	String name = buf;
	String value = readQuotedString(r);
	
	auto xmla = new xml_attribute(name, value);
	attributes.addAtEnd(xmla, true);
      }    
  }
}


xml::xml(const char* filename, bool debug)
{
  int  maxsize=2000;
  char buf[maxsize+1];
  
  readstream r(filename);

  skipCommentsCpp(r);

  //
  // The first 4 characters of the XML file may be garbage.  Look for the header start at the
  // start or after any number of the first 4 characters.
  //
  bool ok = false;
  int fuzz;
  for(fuzz=0; fuzz<=4; fuzz++)
    {    
      r.rewindToStartOfLine();

      for(int i=0; i<fuzz; i++)
	r.getchar();
      
      if(matchString(r, "<?xml"))
	{
	  if(debug && fuzz)
	    printf("GOT a MATCH for header with fuzz %d.\n", fuzz);
	  ok = true;
	  break;
	}
    }
  
  if(!ok)
    {
      printf("Input file \"%s\" in xml::xml doesn't appear to be an XML file.\n", filename);
      printf("Expecting line beginning with \"<?xml\".\n");
      readString(r, buf, "\n\r", maxsize);
      printf("Input line is \"%s\".\n", buf);
      parse_error(r);
    }

  r.rewindToStartOfLine();
  for(int i=0; i<fuzz; i++)
    r.getchar();


  readString(r, buf, "\n\r", maxsize);

  if(debug)
    printf("Got header \"%s\".\n", buf);
  
  header = buf;

  for(;;)
    {
      int c = skipCommentsCpp(r);

      if(c==EOF)
	break;
      
      auto xmle = new xml_entry(r, debug);
      entries.addAtEnd(xmle, true);
    }

}


void xml_attribute::print(writestream& w)
{
  w << name << "=\"" << value << "\"";
}


void xml_entry::print_with_indent(writestream& w, String indent)
{
  w << indent << name;

  ListElement<xml_attribute>* lea;
  for(lea=attributes.first(); lea; lea=lea->next())
    {
      w << " ";
      lea->data()->print(w);
    }

  w << "\n";
  
  ListElement<xml_entry>* lee = children.first();

  if(lee)
    {
      String newindent = indent;
      newindent << "  ";
      for(;lee;lee=lee->next())
	{
	  lee->data()->print_with_indent(w, newindent);
	}
    }

}

void xml::print_with_indent(writestream& w)
{
  w << header << "\n\n";

  ListElement<xml_entry>* lee = entries.first();

  if(lee)
    {
      String indent = "";
      for(;lee;lee=lee->next())
	{
	  lee->data()->print_with_indent(w, indent);
	}
    }

}


void xml_entry::print_with_path(writestream& w, String path)
{
  w << path << name << " Begins     ============\n";

  ListElement<xml_attribute>* lea;
  for(lea=attributes.first(); lea; lea=lea->next())
    {
      w << path << name << "@";
      lea->data()->print(w);
      w << "\n";
    }
  
  ListElement<xml_entry>* lee = children.first();
  if(lee)
    {
      String newpath;
      newpath << path << name << ":";
      for(;lee;lee=lee->next())
	{
	  lee->data()->print_with_path(w, newpath);
	}
    }

}

void xml::print_with_path(writestream& w)
{
  w << header << "\n\n";

  ListElement<xml_entry>* lee = entries.first();

  if(lee)
    {
      String path = "";
      for(;lee;lee=lee->next())
	{
	  lee->data()->print_with_path(w, path);
	}
    }

}


xml_attribute* xml_entry::find_attribute(String name)
{
  ListElement<xml_attribute>* lea = attributes.first();
  for(;lea;lea=lea->next())
    {
      if(name.equals(lea->data()->name))
	return lea->data();
    }
  
  return 0;
}


xml_entry*     xml_entry::find_child(String name)
{
  ListElement<xml_entry>* lee = children.first();
  for(;lee;lee=lee->next())
    {
      if(name.equals(lee->data()->name))
	return lee->data();
    }
  
  return 0;
}

xml_entry*     xml_entry::find_child_after(String name, xml_entry* xmle)
{
  ListElement<xml_entry>* lee = children.first();
  for(;lee;lee=lee->next())
    {
      if(lee->data()==xmle)
	break;
    }
  lee=lee->next();
  for(;lee;lee=lee->next())
    {
      if(name.equals(lee->data()->name))
	return lee->data();
    }
  
  return 0;
}

xml_entry*     xml::find_entry(String name)
{
  ListElement<xml_entry>* lee = entries.first();
  for(;lee;lee=lee->next())
    {
      if(name.equals(lee->data()->name))
	return lee->data();
    }
  
  return 0;
}
