
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

`timescale 1ns / 1ps

`define w assign
`define r always@(posedge clk)

//
// Uses a linear approximation to a sine.  The table is generated
// elsewhere, but is made to simply interpolate between sine points.
// Table length is typically 1024, with a 36-bit table for a single
// 1k x 36 bit memory.  Accuracy is around +/-2e-5 full scale, which
// is a little less than +/- 1 LSB of a 16-bit output word.
// Input is typically 32-bit, but that's clearly overkill.  Just
// because it's convenient to not figure out how small it can be
// made, and having smaller doesn't really save many resources.
//
   
module sample_interleave
  #(
    // Input parameters
    parameter IN_SAMPLES_PER_CLOCK         = 1,
    parameter BITS_PER_SAMPLE              = 16,
    parameter MULTIPLIER                   = 2,
 
    // AXIS side data parameters
    parameter TID_WIDTH                    = 0,
    parameter TDEST_WIDTH                  = 0,


    // Dependent parameters
    parameter  OUT_SAMPLES_PER_CLOCK       = IN_SAMPLES_PER_CLOCK * MULTIPLIER,
    
    parameter  IN_AXIS_BUS_WIDTH           = BITS_PER_SAMPLE * IN_SAMPLES_PER_CLOCK,
    parameter  OUT_AXIS_BUS_WIDTH          = BITS_PER_SAMPLE * OUT_SAMPLES_PER_CLOCK,

    parameter  TID_DUMMY_WIDTH             = (TID_WIDTH==0)   ? 1 : TID_WIDTH,
    parameter  TDEST_DUMMY_WIDTH           = (TDEST_WIDTH==0) ? 1 : TDEST_WIDTH
    )
   (
    // Data input and output

    (* X_INTERFACE_INFO = "xilinx.com:signal:clock:1.0 clk CLK" *)
    (* X_INTERFACE_PARAMETER = "ASSOCIATED_BUSIF IN_AXIS:OUT_AXIS:OUT_DUP_AXIS, ASSOCIATED_RESET resetn" *)
    input                                 clk,

    (* X_INTERFACE_INFO = "xilinx.com:signal:reset:1.0 resetn RST" *)
    input                                 resetn,

    input                                 IN_AXIS_tvalid,
    output wire                           IN_AXIS_tready,
    input [IN_AXIS_BUS_WIDTH-1:0]         IN_AXIS_tdata,
    input                                 IN_AXIS_tlast,
    input [TID_DUMMY_WIDTH-1:0]           IN_AXIS_tid,
    input [TDEST_DUMMY_WIDTH-1:0]         IN_AXIS_tdest,

    output wire                           OUT_AXIS_tvalid,
    input                                 OUT_AXIS_tready,
    output wire [OUT_AXIS_BUS_WIDTH-1:0]  OUT_AXIS_tdata,
    output wire                           OUT_AXIS_tlast,
    output wire [TID_DUMMY_WIDTH-1:0]     OUT_AXIS_tid,
    output wire [TDEST_DUMMY_WIDTH-1:0]   OUT_AXIS_tdest
    );

   genvar i;
                                           
   generate
      for(i=0; i<OUT_SAMPLES_PER_CLOCK; i=i+1)
        begin

	   //
	   // Commenting out this replicates samples instead of zero-inserting between them
	   //
	   //if(i % MULTIPLIER == 0)
	     //begin : input_to_output

		localparam j = i / MULTIPLIER;

		wire [BITS_PER_SAMPLE-1:0] sample_w;

		`w sample_w = IN_AXIS_tdata[ j * BITS_PER_SAMPLE +: BITS_PER_SAMPLE ];

		`w OUT_AXIS_tdata[ i * BITS_PER_SAMPLE +: BITS_PER_SAMPLE ] = sample_w;
		
	     //end
	   //else
	   //  begin : zero_value
//
//		`w OUT_AXIS_tdata[ i * BITS_PER_SAMPLE +: BITS_PER_SAMPLE ] = 0;
//
//	     end
	   
        end
   endgenerate   

   
   `w OUT_AXIS_tvalid     = IN_AXIS_tvalid;
   `w OUT_AXIS_tlast      = IN_AXIS_tlast;

   
   if(TID_WIDTH>0)
     begin
        `w OUT_AXIS_tid   = IN_AXIS_tid;
     end

   if(TDEST_WIDTH>0)
     begin
        `w OUT_AXIS_tdest = IN_AXIS_tdest;
     end
      
endmodule
