
`ifndef BRAM_BASED_RAM_VH
`define BRAM_BASED_RAM_VH

`ifndef BRAM_EXTRA_REGISTERS
 `define BRAM_EXTRA_REGISTERS 0
`endif

`ifndef BRAM_EXTRA_ADDRESS_REGISTERS
 `define BRAM_EXTRA_ADDRESS_REGISTERS 0
`endif

//
//
// Functions to determine cascade length, bram_address_length, and column width.
// We support synchronous cascading of up to 4 BRAMs, handling up to 16384 address length.
// If the address length is larger, we can't handle it without excessive cascading or adding a
// multiplexer at the output or going to a less efficiently-packed BRAM mode.  So use the less
// efficient packing.  Put each bit into a separate BRAM36E2, since 1-bit column width
// is as efficient as any can allows precise widths.  For 65536 and 131072, do the
// same but cascade 2 or 4 BRAM36E2s.
//
// Note that if 1536 length is needed, that could be accomplished with a 3-length cascade
// of width 72 instead of a 1-length cascade of width 18.  That would mean that to cover a 72-bit
// width only 3 BRAM36E2's are required, not 4.  For now, leave some of these unusual special cases
// unaddressed.  But maybe handle some.
//
// This could be extended to handle truly large addressed_length by allowing higher cascade
// or by adding muxes, but for now 131072 is plenty.
//
function automatic integer bram_based_ram_cascade(integer addressed_length);

`ifndef VERILATOR
   assert(addressed_length <= 131072);
`endif
   
   if      (addressed_length<=512)
     bram_based_ram_cascade=1;  // No cascade, 512x72
   else if (addressed_length<=1024)
     bram_based_ram_cascade=1;  // No cascade, 1024x36
   //else if (addressed_length<=1536)
   //  bram_based_ram_cascade=3;  // Use 512x72, cascade 3 times
   else if (addressed_length<=2048)
     bram_based_ram_cascade=1;  // No cascade, 2048x18
   //else if (addressed_length<=3072)
   //  bram_based_ram_cascade=3;  // Use 1024x36, cascade 3 times
   else if (addressed_length<=4096)
     bram_based_ram_cascade=1;  // No cascade, 4096x9
   //else if (addressed_length<=6144)
   //  bram_based_ram_cascade=3;  // Use 2048x18, cascade 3 times
   else if (addressed_length<=8192)
     bram_based_ram_cascade=2;  // Use 4096x9, cascade 2 times
   //else if (addressed_length<=12288)
   //  bram_based_ram_cascade=3;  // Use 4096x9, cascade 3 times
   //else if (addressed_length<=16384)
   //  bram_based_ram_cascade=4;  // Use 4096x9, cascade 4 times
   else if (addressed_length<=16384)
     bram_based_ram_cascade=2;  // Use 8192x4, cascade 2 times
   else if (addressed_length<=32768)
     bram_based_ram_cascade=1;  // No cascade, 32768x1
   else if (addressed_length<=65536)
     bram_based_ram_cascade=2;  // Use 32768x1, cascade 2 times
   else if (addressed_length<=98304)
     bram_based_ram_cascade=3;  // Use 32768x1, cascade 3 times
   else if (addressed_length<=131072)
     bram_based_ram_cascade=4;  // Use 32768x1, cascade 4 times
   else
     bram_based_ram_cascade=-1;  //error_cant_handle_truly_large_memories();

`ifndef VERILATOR
   assert(bram_based_ram_cascade>0);
`endif
endfunction

function automatic integer bram_based_ram_post_cascade_address_bits(integer addressed_length);
   bram_based_ram_post_cascade_address_bits = $clog2( (addressed_length+bram_based_ram_cascade(addressed_length)-1) / bram_based_ram_cascade(addressed_length));
endfunction

function automatic integer bram_based_ram_post_cascade_address_length(integer addressed_length);
   bram_based_ram_post_cascade_address_length = 1<<bram_based_ram_post_cascade_address_bits(addressed_length);
endfunction


`define post_cascade_addressed_length bram_based_ram_post_cascade_address_length(addressed_length)


function automatic integer bram_based_ram_width(integer addressed_length);
   if      (`post_cascade_addressed_length<=512)
     bram_based_ram_width=72;  // No cascade
   else if (`post_cascade_addressed_length==1024)
     bram_based_ram_width=36;  // No cascade
   else if (`post_cascade_addressed_length==2048)
     bram_based_ram_width=18;  // No cascade
   else if (`post_cascade_addressed_length==4096)
     bram_based_ram_width=9;
   else if (`post_cascade_addressed_length==8192)
     bram_based_ram_width=4;
   else if (`post_cascade_addressed_length==16384)
     bram_based_ram_width=-1; // error_16384_case_should_not_happen();
   else if (`post_cascade_addressed_length==32768)
     bram_based_ram_width=1;
   else
     bram_based_ram_width=-1; //error_greater_than_32768_or_non_power_of_2_case_should_not_happen();

`ifndef VERILATOR
   assert(bram_based_ram_width>0);
`endif
endfunction   

//
// Use a BRAM18E2 instead of a BRAM36E2 when the addressed_length and required_column_width allow.
// Note that we may not provide all of the needed_column_width in this column.  If not, another
// column must be added, with its required width reduced by the width of this column.  So the last
// column might go to a BRAM18E2 when earlier columns go to a BRAM36E2.
//
// Also note that the BRAM18E2 supports modes that are half the bit width of the BRAM36E2, rounded
// down.  So a 9-bit mode on the BRAM36E2 becomes a 4-bit mode on the BRAM18E2.
//   
function automatic logic bram_based_ram_use_bram18(integer addressed_length, integer needed_column_width);
   if(needed_column_width<=bram_based_ram_width(addressed_length)/2)
     bram_based_ram_use_bram18 = 1; // true, we can use a bram18
   else
     bram_based_ram_use_bram18 = 0; // false, we need a bram36
endfunction


//
// Extra registers, if needed to meet timing.
//     
function automatic integer bram_based_ram_extra_registers(integer addressed_length);
   bram_based_ram_extra_registers = `BRAM_EXTRA_REGISTERS;
   //bram_based_ram_extra_registers = (addressed_length>32768) ? 2 : 
   //                                 (addressed_length>4096)  ? 1 : 
   //                                 /**/                       0 ;
endfunction

//
// Column has a base delay for the BRAM, plus one extra delay for every level of cascade.
// No longer -- took too many registers.
//     
function automatic integer bram_based_ram_register_delay(integer addressed_length);
   //bram_based_ram_register_delay = 2 + bram_based_ram_cascade(addressed_length);
   bram_based_ram_register_delay = 2 + bram_based_ram_extra_registers(addressed_length);
endfunction

`endif //  `ifndef BRAM_BASED_RAM_VH

