
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2021-2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


`undef w
`define w assign
`undef r
`define r always@(posedge clk)
`timescale  1 ps / 1 ps   


   //
   // Convert INPUT_WIDTH to OUTPUT_WIDTH.  If SHIFT_RIGHT is 0, top bits match.
   // Otherwise output is reduced in amplitude if SHIFT_RIGHT is positive, and
   // increased in amplitude if SHIFT_RIGHT is negative.  Module takes up to 3
   // clocks. If SHIFT_RIGHT is negative, value may exceeed limits, in which case
   // it saturates.  If bits are cut off, rounding will occur.  Rounding is to
   // nearest, with exact 1/2 being rounded towards zero.  
   //
   
module width_convert
  #(    
     parameter   INPUT_WIDTH          = 25,
     parameter   OUTPUT_WIDTH         = 25,
     parameter   SHIFT_RIGHT          = 1,
     parameter   ITEM_NUMBER          = 4,
     parameter   NUM_SYNC             = 1
   )
   (
     input                            clk,
     input                            resetn,
     input [NUM_SYNC-1:0]             in_sync_i,
     input                            in_valid_i,
     input signed [INPUT_WIDTH-1:0]   in_i [0:ITEM_NUMBER-1],
     output [NUM_SYNC-1:0]            out_sync_o,
     output                           out_valid_o,
     output signed [OUTPUT_WIDTH-1:0] out_o [0:ITEM_NUMBER-1]
   );

   localparam MAX_WIDTH          =    INPUT_WIDTH > OUTPUT_WIDTH  ?   INPUT_WIDTH  : OUTPUT_WIDTH  ;
   localparam SHIFT_LEFT_VALUE   =    (SHIFT_RIGHT<0)             ?  -SHIFT_RIGHT  :            0  ;
   localparam SHIFT_RIGHT_VALUE  =  ( (SHIFT_RIGHT>0)             ?   SHIFT_RIGHT  :            0  )  +  (MAX_WIDTH-OUTPUT_WIDTH);
   localparam ENABLE_SHIFT_LEFT  =    (SHIFT_LEFT_VALUE!=0);
   localparam ENABLE_SHIFT_RIGHT =    (SHIFT_RIGHT_VALUE!=0);
   localparam ENABLE_TRUNCATE    =    (MAX_WIDTH!=OUTPUT_WIDTH);
   localparam TOTAL_DELAY        =    (ENABLE_SHIFT_LEFT?1:0) + (ENABLE_SHIFT_RIGHT?1:0) + (ENABLE_TRUNCATE?1:0);

   genvar                             i;
   
   generate
      
      for(i=0; i<ITEM_NUMBER; i=i+1)
        begin : item_number_loop
           wire [MAX_WIDTH-1:0]            in_w;
           wire [MAX_WIDTH-1:0]            shift_left_w;
           wire [MAX_WIDTH-1:0]            shift_right_w;

           if(MAX_WIDTH==INPUT_WIDTH)
	     begin : max_width_equal_to_input_width
		`w in_w = in_i[i];
	     end
           else
	     begin : max_width_not_equal_to_input_width
		`w in_w = { in_i[i], {(MAX_WIDTH-INPUT_WIDTH){1'b0}}};
	     end
	   
           if(ENABLE_SHIFT_LEFT)
	     begin : enable_shift_left
		shift_left_with_saturate #( .REAL_WIDTH(MAX_WIDTH), .SHIFT_LEFT(SHIFT_LEFT_VALUE))   slws ( .clk(clk), .in_i(in_w),         .out_ro(shift_left_w) );
	     end
           else
	     begin : no_shift_left
		`w shift_left_w = in_w;
	     end
           
           if(ENABLE_SHIFT_RIGHT)
	     begin : enable_shift_right
		shift_right_with_round   #( .REAL_WIDTH(MAX_WIDTH), .SHIFT_RIGHT(SHIFT_RIGHT_VALUE)) srwr ( .clk(clk), .in_i(shift_left_w), .out_ro(shift_right_w) );
	     end
           else
	     begin : no_shift_right
		`w shift_right_w = shift_left_w;
	     end
	   
           if(ENABLE_TRUNCATE)
	     begin : enable_truncate
		truncate_with_saturate   #( .INPUT_WIDTH(MAX_WIDTH), .OUTPUT_WIDTH(OUTPUT_WIDTH)) tr ( .clk(clk), .in_i(shift_right_w), .out_ro(out_o[i]) );
	     end
           else
	     begin : no_enable_truncate
		`w out_o[i] = shift_right_w;
	     end
	   
        end // for (i=0; i<ITEM_NUMBER; i=i+1)
      
   endgenerate            

   bxb_sync_delay_2 #(.DELAY(TOTAL_DELAY), .BIT_WIDTH(NUM_SYNC)) sync_delay_instance  ( .clk(clk), .resetn(resetn), .in_i(in_sync_i), .out_o(out_sync_o)  ); 
   delay      #(.DELAY(TOTAL_DELAY), .BIT_WIDTH(1))        valid_delay_instance ( .clk(clk), .in_i(in_valid_i),   .out_o(out_valid_o) ); 

endmodule // width_convert


