
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


`timescale 1ps / 1ps

`define w assign
`define r always@(posedge clk)


module pn_generator
  #(
    parameter   PN_CODE          =  32'h80000401,  // Have no idea if this is a good code.  Bit 0 should be set though.
    parameter   DYNAMIC_PN_CODE  =  1              // If dynamic, use the input.  Otherwise use the static PN_CODE.
    )
   (
    input        clk,
    input        enable_i,
    input [31:0] pn_code_i,
    output reg   out_o
    );

   reg [31:0]    pn_r           = 0;
   wire [31:0]   pn_shift_w;
   reg [31:0]    pn_code_r      = 0;
   reg           pn_is_zero_r   = 0;

   initial out_o = 0;
   
   generate
      if(DYNAMIC_PN_CODE)
        `r pn_code_r <= pn_code_i;
      else
        `r pn_code_r <= PN_CODE;
   endgenerate

   // If pn_r zero, it would stay zero forever without a mechanism to knock it out.
   // Inserting a nonzero bit in bit zero if an all-zero case is detected is such a mechanism.
   // This doesn't affect normal operation, since all-zero never occurs in normal operation.
   `r pn_is_zero_r <= (pn_r==0);
   `w pn_shift_w = { pn_r[30:0], pn_is_zero_r };

   //
   // A 5-input mux:  1:enable_i   2: pn_r   3:pn_code_r   4: pn_r[31]    5:pn_shift_w
   //
   // It might be 3-input if the enable really goes to the register enable, because we don't need to
   // feed back the value pn_r or feed in enable_i in that case.
   //
   `r pn_r  <=  !enable_i      ?   pn_r                    :
                pn_r[31]       ?   pn_shift_w ^ pn_code_r  :
                /**/               pn_shift_w              ;

   `r out_o <= !enable_i ? out_o : out_o ^ pn_r[31];
   
endmodule
