
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2021-2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


`timescale 1ps / 1ps

`define w assign
`define r always@(posedge clk)


module gauss_rand
  #(
    parameter   WIDTH           = 16,
    parameter   UWIDTH          = 13,
    parameter   SELF_RESET      = 1,
    parameter   GAUSS_RAND_TOP  = 1
    )
   (
    input                     clk,
    input                     enable,
    input                     randomizer_in_i,
    output                    randomizer_out_o,
    output signed [WIDTH-1:0] out_o
    );
   
   localparam NUM_STAGES = WIDTH - UWIDTH;

   reg signed [WIDTH-1:0]     out_r = 0;
   wire                       randomizer_w;
   

   generate
      
      if(NUM_STAGES==0)
        begin

           uniform_rand #(.WIDTH(UWIDTH), .SELF_RESET(SELF_RESET)) 
           unirand (.clk(clk), .enable(enable), .randomizer_in_i(randomizer_w), .randomizer_out_o(randomizer_out_o), .out_o(out_o));
           
        end
      else
        begin

           wire signed [WIDTH-2:0] rand1_w;
           wire                    randomizer_out_1_w;
           wire signed [WIDTH-2:0] rand2_w;
                                  
           gauss_rand #(.WIDTH(WIDTH-1), .UWIDTH(UWIDTH), .SELF_RESET(SELF_RESET), .GAUSS_RAND_TOP(0)) 
           grand1 ( .clk(clk), .enable(enable), .randomizer_in_i(randomizer_w),  .randomizer_out_o(randomizer_out_1_w), .out_o(rand1_w) );

           gauss_rand #(.WIDTH(WIDTH-1), .UWIDTH(UWIDTH), .SELF_RESET(0), .GAUSS_RAND_TOP(0)) 
           grand2 ( .clk(clk), .enable(enable), .randomizer_in_i(randomizer_out_1_w), .randomizer_out_o(randomizer_out_o), .out_o(rand2_w) );

           `r out_r <= rand1_w + rand2_w;

           if(GAUSS_RAND_TOP)
             begin
                reg signed [WIDTH-1:0] out_2_r = 0;

                `r out_2_r <= randomizer_w ? -out_r : out_r;
                `w out_o = out_2_r;
             end
           else
             begin
                `w out_o = out_r;
             end
        end 

      if(GAUSS_RAND_TOP)
        begin
           localparam MAX_CLOCKS_WITHOUT_MASTER_RANDOM_1 = 63;
           localparam GUARD_COUNTER_BITS = $clog2(MAX_CLOCKS_WITHOUT_MASTER_RANDOM_1+1);
           reg randomizer_in_r = 0;
           reg [GUARD_COUNTER_BITS-1:0] guard_counter_r = MAX_CLOCKS_WITHOUT_MASTER_RANDOM_1;

           pn32 master_randomizer(.clk(clk), .enable(enable), .in_i(randomizer_in_r), .out_o(randomizer_w));

           `r randomizer_in_r <= randomizer_in_i | (guard_counter_r==0);
           
           `r guard_counter_r <= (!enable)     ?  guard_counter_r                     : 
                                 randomizer_w  ?  MAX_CLOCKS_WITHOUT_MASTER_RANDOM_1  :
                                 /**/             guard_counter_r - 1                 ;
        end
      else
        begin
           `w randomizer_w = randomizer_in_i;
        end
      
   endgenerate
   
endmodule
