
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


`timescale 1ps / 1ps

`define w assign
`define raxis always@(posedge OUT_AXIS_aclk)
`define raxi  always@(posedge S_AXI_ACLK)

//
// AXIS_SAMPLES_PER_CLOCK must be 1, 2, 4, or 8.
//
module white_gaussian_noise
  #(
    // Capture parameters
    parameter AXIS_SAMPLES_PER_CLOCK       = 8,
    parameter AXIS_BITS_PER_SAMPLE         = 16,
    parameter AXIS_BUS_WIDTH               = AXIS_BITS_PER_SAMPLE * AXIS_SAMPLES_PER_CLOCK,

    // Processing parameters
    parameter ENABLE_RANDOMIZER            = 0,
    
    // ARM interface parameters
    parameter AXI_BITS_PER_WORD            = 32,
    parameter AXI_BYTES_PER_WORD           = AXI_BITS_PER_WORD/8,
    parameter NUM_REGISTERS                = 16,  // Must be greater than 1
    parameter AXI_BYTE_ADDR_WIDTH          = $clog2( NUM_REGISTERS * AXI_BYTES_PER_WORD )
    )
   (
    // Data collection options, muxed to output

    (* X_INTERFACE_INFO = "xilinx.com:signal:clock:1.0 OUT_AXIS_aclk CLK" *)
    (* X_INTERFACE_PARAMETER = "ASSOCIATED_BUSIF OUT_0_AXIS:OUT_1_AXIS:OUT_2_AXIS:OUT_3_AXIS:OUT_4_AXIS:OUT_5_AXIS:OUT_6_AXIS:OUT_7_AXIS, ASSOCIATED_RESET OUT_AXIS_aresetn" *)
    input                                OUT_AXIS_aclk,

    (* X_INTERFACE_INFO = "xilinx.com:signal:reset:1.0 OUT_AXIS_aresetn RST" *)
    input                                OUT_AXIS_aresetn,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_0_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_0_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_0_AXIS TREADY" *)  input                                 OUT_0_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_0_AXIS TVALID" *)  output reg                            OUT_0_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_1_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_1_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_1_AXIS TREADY" *)  input                                 OUT_1_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_1_AXIS TVALID" *)  output reg                            OUT_1_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_2_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_2_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_2_AXIS TREADY" *)  input                                 OUT_2_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_2_AXIS TVALID" *)  output reg                            OUT_2_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_3_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_3_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_3_AXIS TREADY" *)  input                                 OUT_3_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_3_AXIS TVALID" *)  output reg                            OUT_3_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_4_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_4_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_4_AXIS TREADY" *)  input                                 OUT_4_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_4_AXIS TVALID" *)  output reg                            OUT_4_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_5_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_5_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_5_AXIS TREADY" *)  input                                 OUT_5_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_5_AXIS TVALID" *)  output reg                            OUT_5_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_6_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_6_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_6_AXIS TREADY" *)  input                                 OUT_6_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_6_AXIS TVALID" *)  output reg                            OUT_6_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_7_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_7_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_7_AXIS TREADY" *)  input                                 OUT_7_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_7_AXIS TVALID" *)  output reg                            OUT_7_AXIS_tvalid,

    input                                randomizer_in,
    output                               randomizer_out,
    
    //
    // This is the interface to the ARM processor
    //
    input                                S_AXI_ACLK,
    input                                S_AXI_ARESETN,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_AWADDR,
    input [2 : 0]                        S_AXI_AWPROT,
    input                                S_AXI_AWVALID,
    output reg                           S_AXI_AWREADY,
    input [AXI_BITS_PER_WORD-1 : 0]      S_AXI_WDATA,
    input [AXI_BYTES_PER_WORD-1 : 0]     S_AXI_WSTRB,
    input                                S_AXI_WVALID,
    output reg                           S_AXI_WREADY,
    output reg [1 : 0]                   S_AXI_BRESP,
    output reg                           S_AXI_BVALID,
    input                                S_AXI_BREADY,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_ARADDR,
    input [2 : 0]                        S_AXI_ARPROT,
    input                                S_AXI_ARVALID,
    output reg                           S_AXI_ARREADY,
    output reg [AXI_BITS_PER_WORD-1 : 0] S_AXI_RDATA,
    output reg [1 : 0]                   S_AXI_RRESP,
    output reg                           S_AXI_RVALID,
    input                                S_AXI_RREADY
);
   
   genvar                                i, j;


   //
   // Registers coming from other clock domain
   //
   // Two registers after a clock transition are needed to avoid metastability.  Truthfully it's not a
   // big issue since these will only change once in a blue moon, and metastability will *eventually*
   // die out.  Also, this isn't sufficient to make sure that all bits in the register update at the
   // same time.  Still, it's something in the right direction.
   //

   reg [7:0]                                                                                 enable_sr;
   reg [7:0]                                                                                 max_amp_sr;
   reg                                                                                       enable_full_sr;

   /**/                                                                          reg [31:0]  reg_shift_right_0_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_0_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_0_srr     = 0;

   /**/                                                                          reg [31:0]  reg_shift_right_1_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_1_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_1_srr     = 0;

   /**/                                                                          reg [31:0]  reg_shift_right_2_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_2_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_2_srr     = 0;
   
   /**/                                                                          reg [31:0]  reg_shift_right_3_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_3_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_3_srr     = 0;
   
   /**/                                                                          reg [31:0]  reg_shift_right_4_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_4_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_4_srr     = 0;

   /**/                                                                          reg [31:0]  reg_shift_right_5_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_5_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_5_srr     = 0;

   /**/                                                                          reg [31:0]  reg_shift_right_6_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_6_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_6_srr     = 0;

   /**/                                                                          reg [31:0]  reg_shift_right_7_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  shift_right_7_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]   shift_right_7_srr     = 0;


   `raxis  shift_right_0_sr   <=  reg_shift_right_0_ar;
   `raxis  shift_right_0_srr  <=  shift_right_0_sr;

   `raxis  shift_right_1_sr   <=  reg_shift_right_1_ar;
   `raxis  shift_right_1_srr  <=  shift_right_1_sr;

   `raxis  shift_right_2_sr   <=  reg_shift_right_2_ar;
   `raxis  shift_right_2_srr  <=  shift_right_2_sr;

   `raxis  shift_right_3_sr   <=  reg_shift_right_3_ar;
   `raxis  shift_right_3_srr  <=  shift_right_3_sr;

   `raxis  shift_right_4_sr   <=  reg_shift_right_4_ar;
   `raxis  shift_right_4_srr  <=  shift_right_4_sr;

   `raxis  shift_right_5_sr   <=  reg_shift_right_5_ar;
   `raxis  shift_right_5_srr  <=  shift_right_5_sr;

   `raxis  shift_right_6_sr   <=  reg_shift_right_6_ar;
   `raxis  shift_right_6_srr  <=  shift_right_6_sr;

   `raxis  shift_right_7_sr   <=  reg_shift_right_7_ar;
   `raxis  shift_right_7_srr  <=  shift_right_7_sr;

   `raxis  enable_sr          <=  { shift_right_7_sr[31],  shift_right_6_sr[31],  shift_right_5_sr[31],  shift_right_4_sr[31],  shift_right_3_sr[31],  shift_right_2_sr[31],  shift_right_1_sr[31],  shift_right_0_sr[31] };
   `raxis  enable_full_sr     <=  |enable_sr;

   `raxis  max_amp_sr         <=  { shift_right_7_sr[30],  shift_right_6_sr[30],  shift_right_5_sr[30],  shift_right_4_sr[30],  shift_right_3_sr[30],  shift_right_2_sr[30],  shift_right_1_sr[30],  shift_right_0_sr[30] };

   //////////////////////////////////////////////////////////////////////////////////////////////////////
   //
   // Main code here, on AXIS clock.  Control registers after clock transition are 
   //
   //    enable_full_sr   enable_sr   shift_right_*_srr
   //

   initial OUT_0_AXIS_tvalid = 0;
   initial OUT_0_AXIS_tdata  = 1'b1;

   initial OUT_1_AXIS_tvalid = 0;
   initial OUT_1_AXIS_tdata  = 1'b1;

   initial OUT_2_AXIS_tvalid = 0;
   initial OUT_2_AXIS_tdata  = 1'b1;

   initial OUT_3_AXIS_tvalid = 0;
   initial OUT_3_AXIS_tdata  = 1'b1;

   initial OUT_4_AXIS_tvalid = 0;
   initial OUT_4_AXIS_tdata  = 1'b1;

   initial OUT_5_AXIS_tvalid = 0;
   initial OUT_5_AXIS_tdata  = 1'b1;

   initial OUT_6_AXIS_tvalid = 0;
   initial OUT_6_AXIS_tdata  = 1'b1;

   initial OUT_7_AXIS_tvalid = 0;
   initial OUT_7_AXIS_tdata  = 1'b1;

   wire [AXIS_SAMPLES_PER_CLOCK:0]       randomizer_w;
   
   `w randomizer_w[0] = ENABLE_RANDOMIZER ? randomizer_in : 0;
   `w randomizer_out  = randomizer_w[AXIS_SAMPLES_PER_CLOCK];
   
   generate      
      
      for(i=0; i<AXIS_SAMPLES_PER_CLOCK; i=i+1)
        begin

           wire signed [AXIS_BITS_PER_SAMPLE-1:0] grand_value;
           reg signed [AXIS_BITS_PER_SAMPLE-1:0] grand_value_maxed;
           
           gauss_rand 
               #(
                 .WIDTH             (  AXIS_BITS_PER_SAMPLE      ),
                 .UWIDTH            (  AXIS_BITS_PER_SAMPLE-3    ),
                 .SELF_RESET        (  i==0                      )
                )
           gr   (
                 .clk               (  OUT_AXIS_aclk             ),
                 .enable            (  enable_full_sr            ),
                 .randomizer_in_i   (  randomizer_w[i]           ),
                 .randomizer_out_o  (  randomizer_w[i+1]         ),
                 .out_o             (  grand_value               )
                );

           `raxis grand_value_maxed = { 
                                        grand_value[AXIS_BITS_PER_SAMPLE-1], 
                                        { (AXIS_BITS_PER_SAMPLE-2) { ~grand_value[AXIS_BITS_PER_SAMPLE-1] }}, 
                                        1'b1
                                      };
           
           `raxis OUT_0_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[0]  ? 0                                      :
                                                                                        !max_amp_sr[0] ? ( grand_value >>> shift_right_0_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

           `raxis OUT_1_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[1]  ? 0                                      :
                                                                                        !max_amp_sr[1] ? ( grand_value >>> shift_right_1_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

           `raxis OUT_2_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[2]  ? 0                                      :
                                                                                        !max_amp_sr[2] ? ( grand_value >>> shift_right_2_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

           `raxis OUT_3_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[3]  ? 0                                      :
                                                                                        !max_amp_sr[3] ? ( grand_value >>> shift_right_3_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

           `raxis OUT_4_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[4]  ? 0                                      :
                                                                                        !max_amp_sr[4] ? ( grand_value >>> shift_right_4_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

           `raxis OUT_5_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[5]  ? 0                                      :
                                                                                        !max_amp_sr[5] ? ( grand_value >>> shift_right_5_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

           `raxis OUT_6_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[6]  ? 0                                      :
                                                                                        !max_amp_sr[6] ? ( grand_value >>> shift_right_6_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

           `raxis OUT_7_AXIS_tdata[i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE] <= !enable_sr[7]  ? 0                                      :
                                                                                        !max_amp_sr[7] ? ( grand_value >>> shift_right_7_srr )  :
                                                                                         /**/            grand_value_maxed                      ;

        end
      
   endgenerate
   
   `raxis OUT_0_AXIS_tvalid <= 1'b1;
   `raxis OUT_1_AXIS_tvalid <= 1'b1;
   `raxis OUT_2_AXIS_tvalid <= 1'b1;
   `raxis OUT_3_AXIS_tvalid <= 1'b1;
   `raxis OUT_4_AXIS_tvalid <= 1'b1;
   `raxis OUT_5_AXIS_tvalid <= 1'b1;
   `raxis OUT_6_AXIS_tvalid <= 1'b1;
   `raxis OUT_7_AXIS_tvalid <= 1'b1;


   //
   // End main code
   //
   //////////////////////////////////////////////////////////////////////////////////////////////////////


     
   //
   // Handle READs/WRITEs from the processor on the AXI bus.
   //    

   localparam                              WORD_ADDR_BITS         = $clog2(NUM_REGISTERS);   
   localparam                              BYTE_ADDR_BITS_IN_WORD = $clog2(AXI_BYTES_PER_WORD);
   
   reg [WORD_ADDR_BITS-1 : 0]              write_address     = 0;
   reg [AXI_BITS_PER_WORD-1:0]             write_data        = 0;
   wire                                    write_enable;
   wire                                    write_enable_w;
   reg                                     write_enable_r    = 0;
   wire                                    write_done;
   
   reg                                     read_enable       = 0;
   reg [WORD_ADDR_BITS-1 : 0]              read_address      = 0;
   wire [AXI_BITS_PER_WORD-1:0]            read_data;
   wire                                    read_done;

   //
   // The write interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_AWREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_AWREADY && S_AXI_AWVALID)   ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_AWREADY  ;

   `raxi S_AXI_WREADY  <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_WREADY && S_AXI_WVALID)     ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_WREADY   ;

   `raxi S_AXI_BVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          write_done                         ?  1'b1           :
                          (S_AXI_BREADY && S_AXI_BVALID)     ?  1'b0           :
                          /* default */                         S_AXI_BVALID   ;

   `raxi S_AXI_BRESP   <= 2'b0;

   //
   // The read interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_ARREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_ARREADY && S_AXI_ARVALID)   ?  1'b0           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b1           :
                          /* default */                         S_AXI_ARREADY  ;
   

   `raxi S_AXI_RVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          read_done                          ?  1'b1           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b0           :
                          /* default */                         S_AXI_RVALID   ;

   `raxi S_AXI_RRESP   <= 2'b0;
   
   `raxi S_AXI_RDATA   <= (!S_AXI_ARESETN)                   ?  0              : 
                          read_done                          ?  read_data      :
                          /* default */                         S_AXI_RDATA    ;


   //
   // Define write addresses, data, and enables
   //
   `w write_enable_w          = !S_AXI_WREADY && !S_AXI_AWREADY;
   `raxi write_enable_r       = (!S_AXI_ARESETN) ? 0 : write_enable_w;
   `w write_enable            = write_enable_w && !write_enable_r;   // Only pulse write enable for a single clock, regardless of BREADY
   
   `raxi write_address       <= (!S_AXI_ARESETN)                  ?  0                                                              :
                                (S_AXI_AWREADY && S_AXI_AWVALID)  ?  S_AXI_AWADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                        write_address                                                  ;
   
   `raxi write_data          <= (!S_AXI_ARESETN)                  ?  0                :
                                (S_AXI_WREADY && S_AXI_WVALID)    ?  S_AXI_WDATA      :
                                /* default */                        write_data       ;

   `w write_done              = write_enable;  // Write done immediately after address and data are both present
   

   //
   // Define read addresses, data, and enables
   //
   `raxi read_address        <= (!S_AXI_ARESETN)               ?  0                                                              : 
                                S_AXI_ARREADY & S_AXI_ARVALID  ?  S_AXI_ARADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                     read_address                                                   ;
   
   `raxi read_enable         <= S_AXI_ARREADY & S_AXI_ARVALID;

   `w read_done               = read_enable;  // Read data available immediately after address is present.
   
   // read_data  is assigned a value below



   //
   // Handle read data.  For this module we don't have any data to read on the AXIS clock, so we don't need to
   // handle any clock transitions here.
   //
      
   `w read_data = (read_address==0)  ?  reg_shift_right_0_ar  :
                  (read_address==1)  ?  reg_shift_right_1_ar  :
                  (read_address==2)  ?  reg_shift_right_2_ar  :
                  (read_address==3)  ?  reg_shift_right_3_ar  :
                  (read_address==4)  ?  reg_shift_right_4_ar  :
                  (read_address==5)  ?  reg_shift_right_5_ar  :
                  (read_address==6)  ?  reg_shift_right_6_ar  :
                  (read_address==7)  ?  reg_shift_right_7_ar  :
                  /* default */         0                     ;

   
   //
   // Handle write data.  Registers write on the AXI clock.
   // None of the other registers are supposed to be written unless reg_trigger_arm_ar is 0.
   // Thus they won't be written at a critical time and thus metastability isn't an issue.
   //


`ifdef DEBUG
   reg [31:0]  init_reg_shift_right_ar      = 0;              // Reigster 1
`else
   reg [31:0]  init_reg_shift_right_ar      = 0;              // Reigster 1
`endif

   
   `raxi reg_shift_right_0_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==0)     ?  write_data                :
                                          /* default */                             reg_shift_right_0_ar      ;
     
   `raxi reg_shift_right_1_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==1)     ?  write_data                :
                                          /* default */                             reg_shift_right_1_ar      ;
     
   `raxi reg_shift_right_2_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==2)     ?  write_data                :
                                          /* default */                             reg_shift_right_2_ar      ;
     
   `raxi reg_shift_right_3_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==3)     ?  write_data                :
                                          /* default */                             reg_shift_right_3_ar      ;
     
   `raxi reg_shift_right_4_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==4)     ?  write_data                :
                                          /* default */                             reg_shift_right_4_ar      ;
     
   `raxi reg_shift_right_5_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==5)     ?  write_data                :
                                          /* default */                             reg_shift_right_5_ar      ;
     
   `raxi reg_shift_right_6_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==6)     ?  write_data                :
                                          /* default */                             reg_shift_right_6_ar      ;
     
   `raxi reg_shift_right_7_ar         <=  (!S_AXI_ARESETN)                       ?  init_reg_shift_right_ar   :
                                          write_enable && (write_address==7)     ?  write_data                :
                                          /* default */                             reg_shift_right_7_ar      ;
     

endmodule
