
`timescale 1 ns / 1 ps

`define r always @( posedge S_AXI_ACLK )
`define w assign

   
module AXI_DNA 
  #(
    parameter C_S_AXI_DATA_WIDTH = 32,
    parameter C_S_AXI_ADDR_WIDTH = 5
    )
   (
    input wire                                S_AXI_ACLK,
    input wire                                S_AXI_ARESETN,
    input wire [C_S_AXI_ADDR_WIDTH-1 : 0]     S_AXI_AWADDR,
    input wire                                S_AXI_AWVALID,
    output reg                                S_AXI_AWREADY,
    input wire [C_S_AXI_DATA_WIDTH-1 : 0]     S_AXI_WDATA,
    input wire                                S_AXI_WVALID,
    output reg                                S_AXI_WREADY,
    output reg [1 : 0]                        S_AXI_BRESP,
    output reg                                S_AXI_BVALID,
    input wire                                S_AXI_BREADY,
    input wire [C_S_AXI_ADDR_WIDTH-1 : 0]     S_AXI_ARADDR,
    input wire                                S_AXI_ARVALID,
    output reg                                S_AXI_ARREADY,
    output reg [C_S_AXI_DATA_WIDTH-1 : 0]     S_AXI_RDATA,
    output reg [1 : 0]                        S_AXI_RRESP,
    output reg                                S_AXI_RVALID,
    input wire                                S_AXI_RREADY
    );


   ////////////////////////////////////////////////////////////////////////////////////
   //
   // Version info
   //
   ////////////////////////////////////////////////////////////////////////////////////

   localparam                                 PRODUCT_ID     = 32'hDC;
   localparam                                 VERSION_MAJOR  = 32'd02;
   localparam                                 VERSION_MINOR  = 32'd00;
   localparam                                 REVISION       = 32'd01;
   localparam                                 VERSION_REG    = (PRODUCT_ID<<24)+(VERSION_MAJOR<<16)+(VERSION_MINOR<<8)+REVISION;
   
   ////////////////////////////////////////////////////////////////////////////////////
   //
   // User sets NUM_REGISTERS to the number of registers made visible on the AMBA bus
   //
   ////////////////////////////////////////////////////////////////////////////////////
   localparam                                 NUM_REGISTERS     = 5;  // Must be greater than 1
   localparam                                 ADDR_BITS         = $clog2(NUM_REGISTERS);   
   
   localparam                                 BYTES_IN_WORD     = C_S_AXI_DATA_WIDTH/8;
   localparam                                 ADDR_BITS_IN_WORD = $clog2(BYTES_IN_WORD);


   //generate
   //   if(NUM_REGISTERS<2 || ADDR_BITS+ADDR_BITS_IN_WORD > C_S_AXI_ADDR_WIDTH)
   //     error_module("Too few registers, or more registers than will fit in address space.");  // Call a nonexistent module to throw an error   
   //endgenerate

   
   reg [ADDR_BITS-1 : 0]                      write_address;
   reg [C_S_AXI_DATA_WIDTH-1:0]               write_data;
   wire                                       write_enable;
   wire                                       write_done;

   reg                                        read_enable;
   reg [ADDR_BITS-1 : 0]                      read_address;
   wire [C_S_AXI_DATA_WIDTH-1:0]              read_data;
   wire                                       read_done;

   //
   // The write interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `r S_AXI_AWREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                       (S_AXI_AWREADY && S_AXI_AWVALID)   ?  1'b0           :
                       (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                       /* default */                         S_AXI_AWREADY  ;

   `r S_AXI_WREADY  <= (!S_AXI_ARESETN)                   ?  1'b1           :
                       (S_AXI_WREADY && S_AXI_WVALID)     ?  1'b0           :
                       (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                       /* default */                         S_AXI_WREADY   ;

   `r S_AXI_BVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                       write_done                         ?  1'b1           :
                       (S_AXI_BREADY && S_AXI_BVALID)     ?  1'b0           :
                       /* default */                         S_AXI_BVALID   ;

   `r S_AXI_BRESP   <= 2'b0;

   //
   // The read interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `r S_AXI_ARREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                       (S_AXI_ARREADY && S_AXI_ARVALID)   ?  1'b0           :
                       (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b1           :
                       /* default */                         S_AXI_ARREADY  ;
   

   `r S_AXI_RVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                       read_done                          ?  1'b1           :
                       (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b0           :
                       /* default */                         S_AXI_RVALID   ;

   `r S_AXI_RRESP   <= 2'b0;
   
   `r S_AXI_RDATA   <= (!S_AXI_ARESETN)                   ?  0              : 
                       read_done                          ?  read_data      :
                       /* default */                         S_AXI_RDATA    ;


   //
   // Define write addresses, data, and enables
   //
   `w write_enable         = !S_AXI_WREADY && !S_AXI_AWREADY;

   `r write_address       <= (!S_AXI_ARESETN)                  ?  0                                                              :
                             (S_AXI_AWREADY && S_AXI_AWVALID)  ?  S_AXI_AWADDR[ADDR_BITS-1+ADDR_BITS_IN_WORD:ADDR_BITS_IN_WORD]  :
                             /* default */                        write_address                                                  ;
   
   `r write_data          <= (!S_AXI_ARESETN)                  ?  0                :
                             (S_AXI_WREADY && S_AXI_WVALID)    ?  S_AXI_WDATA      :
                             /* default */                        write_data       ;

   `w write_done           = write_enable;  // Write done immediately after address and data are both present
   

   //
   // Define read addresses, data, and enables
   //
   `r read_address        <= (!S_AXI_ARESETN)               ?  0                                                              : 
                             S_AXI_ARREADY & S_AXI_ARVALID  ?  S_AXI_ARADDR[ADDR_BITS-1+ADDR_BITS_IN_WORD:ADDR_BITS_IN_WORD]  :
                             /* default */                     read_address                                                   ;
   
   `r read_enable         <= S_AXI_ARREADY & S_AXI_ARVALID;

   `w read_done            = read_enable;  // Read data available immediately after address is present.
   
   // read_data  is assigned a value below

   
   ////////////////////////////////////////////////////////////////////////////////////
   //
   // User logic, to define read_data for reads based on read_address and to 
   // handle byte writes based on byte_write_enable, write_data, and write_address.
   //
   ////////////////////////////////////////////////////////////////////////////////////

   //
   // Read the DNA shift register.  Initially, fill with zeroes with a 1 in the high bit.
   // keep shifting until that 1 shifts out.
   //
   localparam DNA_BITS = 96;

   reg [DNA_BITS-1:0] DNA;
   wire DOUT;
   reg  DNA_read_done;
    
   DNA_PORTE2 dna_read(.DOUT(DOUT), .CLK(S_AXI_ACLK), .DIN(1'b0), .READ(!S_AXI_ARESETN), .SHIFT(1'b1));
      
   `r DNA <= (!S_AXI_ARESETN)  ?  1<<(DNA_BITS-1)      :
             DNA_read_done     ?  DNA                  : 
             /* default */        { DOUT, DNA[95:1] }  ;

   `r DNA_read_done <= (!S_AXI_ARESETN) ?  1'b0   : 
                       DNA_read_done    ?  1'b1   :
                       /* default */       DNA[0] ;



   //
   // One read-write register.  Not needed for the DNA, but it serves as a good example for when write
   // capability is required.
   //
   // Will return 0xDEADBEEF on first read.  Second read returns VERSION_REG.  Or if something
   // is written, it will return that for a single read, then VERSION_REG will be returned after.
   // This behavior is mostly for testing.
   //
   //
   // The idea of the contents of the register changing on read doesn't work with caching.  It's
   // exposed a bug in mmap of /dev/mem with O_SYNC, that caching is supposed to be turned off
   // but turns itself back on.  This persists across runs, which indicates it's either in
   // the processor, hardware, or kernel.
   //
   reg [31:0] register;

   `r register <= ( (!S_AXI_ARESETN)                   ?  32'hDEADBEEF      :
                    write_enable && write_address==3   ?  write_data        :
                    read_enable && read_address==3     ?  VERSION_REG       :
                    /* default */                         register          );
   
   
   //
   // Handle read data
   //
   `w read_data = (read_address==0) ? DNA[95:64]   :
                  (read_address==1) ? DNA[63:32]   :
                  (read_address==2) ? DNA[31:0]    :
                  (read_address==3) ? VERSION_REG  :
                  /* default */       register     ;
                       
   
endmodule
