#!/bin/sh -e


usage()
{
    echo ""
    echo "  This script adds utilities and a demonstration program to a Debian base"
    echo "  filesystem for the RFSoC4x2 board."
    echo ""
    exit 20
}

#
# Must be run as root
#
if [ `id -u` != "0" ]; then
    echo ""
    echo "This script must be run as root.  Use sudo."
    echo ""
    exit 20
fi

BASE=../STYNQ_BASE/filesystem/STYNQ_filesystem.tgz

if [ ! -f "$BASE" ]; then
    echo ""
    echo "Can't find base filesystem at $BASE."
    echo "Run the build.sh script in that directory to"
    echo "build a base filesystem before running this"
    echo "script."
    echo ""
    exit 20
fi

ROOT_NOPATH=STYNQ_filesystem
ROOT="$PWD/$ROOT_NOPATH"

if [ -d "$ROOT" ]; then
    echo "##########################################################################################"
    echo "##"
    echo "##  This will delete the old STYNQ filesytem in $ROOT"
    echo "##"
    echo "##  A new STYNQ filesystem will be made at $ROOT from Debian sources."
    echo "##"
    echo "##  ctrl-d to continue, ctrl-c to abort."
    echo "##"
    echo "##########################################################################################"
    cat
    #
    # These umounts are in case the script is re-run after a failure.
    # in that case, these may be left mounted.  They should not be mounted
    # as we start again, so umount them just in case.  Failures in the
    # umount are likely.
    #
    umount "$ROOT"/sys  >/dev/null 2>&1 || true
    umount "$ROOT"/dev  >/dev/null 2>&1 || true
    umount "$ROOT"/proc >/dev/null 2>&1 || true
    
    umount "$ROOT"/sys  >/dev/null 2>&1 || true
    umount "$ROOT"/dev  >/dev/null 2>&1 || true
    umount "$ROOT"/proc >/dev/null 2>&1 || true

    rm -rf "$ROOT"
fi

rm -f "$ROOT_NOPATH".tgz

tar xzpf "$BASE"


if [ ! -d "$ROOT" ]; then
    echo ""
    echo "Failed to extract STYNQ filesystem to $ROOT."
    echo ""
    exit 20
fi

USER=`basename $ROOT/home/*`

if [ ! -d $ROOT/home/$USER ]; then
   echo ""
   echo "The installed username was identified as \"$USER\", but this is incorrect."
   echo "Aborting."
   echo ""
   exit 20
fi

echo ""
echo "######################################################################"
echo "##"
echo "##  Adding packages to the base filesystem."
echo "##"
echo "######################################################################"
echo ""

mv "$ROOT"/etc/resolv.conf "$ROOT"/etc/resolv.conf-bak
echo "nameserver 8.8.8.8" >"$ROOT"/etc/resolv.conf

sudo mount /sys  "$ROOT"/sys  -o bind
sudo mount /dev  "$ROOT"/dev  -o bind
sudo mount /proc "$ROOT"/proc -o bind
    
#
# Install needed packages
#    
chroot "$ROOT" apt update
chroot "$ROOT" apt --yes upgrade

chroot "$ROOT" apt -y install libdrm-dev
chroot "$ROOT" apt -y install librsvg2-dev
chroot "$ROOT" apt -y install libwebsockets-dev
chroot "$ROOT" apt -y install make
chroot "$ROOT" apt -y install g++
chroot "$ROOT" apt -y install cmake
chroot "$ROOT" apt -y install libsysfs-dev
chroot "$ROOT" apt -y install libasound2-dev
chroot "$ROOT" apt -y install device-tree-compiler
chroot "$ROOT" apt -y install git
chroot "$ROOT" apt -y install flex
chroot "$ROOT" apt -y install bison
chroot "$ROOT" apt -y install bc
chroot "$ROOT" apt -y install fortunes

chroot "$ROOT" apt -y clean

cat >$ROOT/opt/STYNQ/startup_scripts/splash.sh <<EOF
#!/bin/bash
program=\`ls /home/*/bin/set_newhaven_display 2>/dev/null | head -1\`

if [ -x "\$program" ]; then
   "\$program" "Debian STYNQ IP " "    None Yet    "
fi 

while true; do

  IP=\`ip -4 -br addr | grep eth0 | sed 's@/.*@@; s/.*[ ]//'\`
  IPCHECK=\`echo \$IP | sed 's/[0-9]*[.][0-9]*[.][0-9]*[.][0-9]*//'\`

  if [ "\$IP" != "" -a "\$IPCHECK" = "" ]; then

     echo "Got IP address \$IP"
     break
  fi

  sleep 1

done

if [ -x "\$program" ]; then
   "\$program" "Debian STYNQ IP " "\$IP"
fi 
EOF
chmod ugo+x $ROOT/opt/STYNQ/startup_scripts/splash.sh

   
cat STYNQ_arm_sw.tgz | (cd $ROOT/home/$USER; tar xzpf -)
mv $ROOT/home/$USER/STYNQ_arm_sw/* $ROOT/home/$USER/
rmdir $ROOT/home/$USER/STYNQ_arm_sw

cat >$ROOT/home/$USER/runme.sh <<EOF
#!/bin/bash
(cd arm_sw; make)
echo ""
echo "Running the program needs a password for root permissions."
echo "Once it runs, you need a mouse on the USB and monitor"
echo "on the DisplayPort to see the demo.  USB touchscreens also"
echo "work.  Or alternately, connect with a browser to the IP address."
echo "(The IP address might be on the LED display now. Write it down!)"
echo ""
sudo arm_sw/RFSoC_graphing/exe/RFSoC_graphing -configfile bxbapp.config
EOF
chmod ugo+x $ROOT/home/$USER/runme.sh

cat >$ROOT/home/$USER/README.TXT <<EOF
Welcome to STYNQ!

*************************************************************************

Files in the STYNQ distribution have various copyrights.  Often, binaries
cannot be distributed.  Refer to the source distribution for copyright
information.

*************************************************************************

To run the example program, type ./runme.sh

Note that this example program is graphical, so you need a monitor
and a mouse, or you need to connect to the board with a web browser.

EOF

cat >>$ROOT/home/$USER/.bashrc <<EOF
if [[ "\$0" == -* ]]; then
   # This is a login shell
   cat README.TXT
   echo "*************************************************************************"
   echo ""
   echo "Today's fortune:"
   echo ""
   fortune | sed 's/^/    /'
   echo ""
   echo "*************************************************************************"
   echo ""
fi
EOF

echo ""
echo "###########################################################################"
echo "##"
echo "##  Copying in the PL binfile."
echo "##"
echo "###########################################################################"
echo ""

cp Vivado/best_results/RFSoC4x2_BxBDemo1.bin $ROOT/home/$USER/


chroot "$ROOT" chown -R $USER:$USER /home/$USER

echo ""
echo ""
echo "###########################################################################"
echo "##"
echo "##  Creating a tar archive of the STYNQ filesystem."
echo "##"
echo "###########################################################################"
echo ""

mv "$ROOT"/etc/resolv.conf-bak "$ROOT"/etc/resolv.conf

sudo umount "$ROOT"/sys
sudo umount "$ROOT"/dev
sudo umount "$ROOT"/proc

#BACKUP="$ROOT_NOPATH"_`date +%Y_%m_%d_%H.%M`.tgz

BACKUP=$ROOT_NOPATH.tgz

tar czf $BACKUP $ROOT_NOPATH

#rm -rf "$ROOT"

echo ""
echo ""
echo "###########################################################################################"
echo "##"
echo "##  Finished successfully with creation of an RFSoC4x2-specific STYNQ OS."
echo "##  The output is in the tar file $BACKUP"
echo "##"
echo "##  To use this, an SD card needs to be formatted with two partitions, typically by"
echo "##  using the program \"fdisk\"."
echo "##"
echo "##  The first partition needs to be formatted to a VFAT filesystem, typicallly bu using"
echo "##  the program \"mkfs.vfat\".  Then it needs to have the files BOOT.BIN,"
echo "##  BOOT.SCR, and image.ub placed on it."
echo "##"
echo "##  The second partition needs to be formatted to EXT4, typicaly by using \"mkfs.ext4\"."
echo "##  Then it is mounted and equipped with the files just generated, as root, like so:"
echo "##"
echo "##     cd DIRETORY_OF_MOUNTED_SECOND_PARTITION"
echo "##     tar xzpf PATH/$BACKUP"
echo "##     mv $ROOT_NOPATH/* ."
echo "##     rmdir $ROOT_NOPATH"
echo "##"
echo "##  Then sync the disks with \"sync\" and cleanly unmount the partitions.  Remove"
echo "##  the SD card, set the board to boot from it, insert it in the board, and the"
echo "##  board should boot."
echo "##"
echo "##  You can connect to the board using its serial debug interface, or by ssh to its IP."
echo "##"
echo "##  This filesystem includes an example program, which you can compile and run"
echo "##  with the command \"./runme.sh\"."
echo "##"
echo "##  Debian packages can be installed by running \"apt install package\"."
echo "##  on the booted FPGA board."
echo "##"
echo "###########################################################################################"
echo ""

