// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include "hwalloc.hh"

#include <stdio.h>


#define DP_BASE_ADDR 0xfd4a0000
#define REG_LENGTH (0x03FF>>2)

typedef uint64_t UL;
#define BITS_PER_LONG 64

#define BIT(nr)			(UL(1) << (nr))

// GENMASK_ULL(39, 21) gives us the 64bit vector 0x000000ffffe00000.

#define GENMASK(h, l) (((~UL(0)) - (UL(1) << (l)) + 1) & (~UL(0) >> (BITS_PER_LONG - 1 - (h))))



/* Link configuration registers */
#define ZYNQMP_DP_TX_LINK_BW_SET			0x0
#define ZYNQMP_DP_TX_LANE_CNT_SET			0x4
#define ZYNQMP_DP_TX_ENHANCED_FRAME_EN			0x8
#define ZYNQMP_DP_TX_TRAINING_PATTERN_SET		0xc
#define ZYNQMP_DP_TX_SCRAMBLING_DISABLE			0x14
#define ZYNQMP_DP_TX_DOWNSPREAD_CTL			0x18
#define ZYNQMP_DP_TX_SW_RESET				0x1c
#define ZYNQMP_DP_TX_SW_RESET_STREAM1			BIT(0)
#define ZYNQMP_DP_TX_SW_RESET_STREAM2			BIT(1)
#define ZYNQMP_DP_TX_SW_RESET_STREAM3			BIT(2)
#define ZYNQMP_DP_TX_SW_RESET_STREAM4			BIT(3)
#define ZYNQMP_DP_TX_SW_RESET_AUX			BIT(7)
#define ZYNQMP_DP_TX_SW_RESET_ALL			(ZYNQMP_DP_TX_SW_RESET_STREAM1 | \
							 ZYNQMP_DP_TX_SW_RESET_STREAM2 | \
							 ZYNQMP_DP_TX_SW_RESET_STREAM3 | \
							 ZYNQMP_DP_TX_SW_RESET_STREAM4 | \
							 ZYNQMP_DP_TX_SW_RESET_AUX)

/* Core enable registers */
#define ZYNQMP_DP_TX_ENABLE				0x80
#define ZYNQMP_DP_TX_ENABLE_MAIN_STREAM			0x84
#define ZYNQMP_DP_TX_FORCE_SCRAMBLER_RESET		0xc0
#define ZYNQMP_DP_TX_VERSION				0xf8
#define ZYNQMP_DP_TX_VERSION_MAJOR_MASK			GENMASK(31, 24)
#define ZYNQMP_DP_TX_VERSION_MAJOR_SHIFT		24
#define ZYNQMP_DP_TX_VERSION_MINOR_MASK			GENMASK(23, 16)
#define ZYNQMP_DP_TX_VERSION_MINOR_SHIFT		16
#define ZYNQMP_DP_TX_VERSION_REVISION_MASK		GENMASK(15, 12)
#define ZYNQMP_DP_TX_VERSION_REVISION_SHIFT		12
#define ZYNQMP_DP_TX_VERSION_PATCH_MASK			GENMASK(11, 8)
#define ZYNQMP_DP_TX_VERSION_PATCH_SHIFT		8
#define ZYNQMP_DP_TX_VERSION_INTERNAL_MASK		GENMASK(7, 0)
#define ZYNQMP_DP_TX_VERSION_INTERNAL_SHIFT		0

/* Core ID registers */
#define ZYNQMP_DP_TX_CORE_ID				0xfc
#define ZYNQMP_DP_TX_CORE_ID_MAJOR_MASK			GENMASK(31, 24)
#define ZYNQMP_DP_TX_CORE_ID_MAJOR_SHIFT		24
#define ZYNQMP_DP_TX_CORE_ID_MINOR_MASK			GENMASK(23, 16)
#define ZYNQMP_DP_TX_CORE_ID_MINOR_SHIFT		16
#define ZYNQMP_DP_TX_CORE_ID_REVISION_MASK		GENMASK(15, 8)
#define ZYNQMP_DP_TX_CORE_ID_REVISION_SHIFT		8
#define ZYNQMP_DP_TX_CORE_ID_DIRECTION			GENMASK(1)

/* AUX channel interface registers */
#define ZYNQMP_DP_TX_AUX_COMMAND			0x100
#define ZYNQMP_DP_TX_AUX_COMMAND_CMD_SHIFT		8
#define ZYNQMP_DP_TX_AUX_COMMAND_ADDRESS_ONLY		BIT(12)
#define ZYNQMP_DP_TX_AUX_COMMAND_BYTES_SHIFT		0
#define ZYNQMP_DP_TX_AUX_WRITE_FIFO			0x104
#define ZYNQMP_DP_TX_AUX_ADDRESS			0x108
#define ZYNQMP_DP_TX_CLK_DIVIDER			0x10c
#define ZYNQMP_DP_TX_CLK_DIVIDER_MHZ			1000000
#define ZYNQMP_DP_TX_CLK_DIVIDER_AUX_FILTER_SHIFT	8
#define ZYNQMP_DP_TX_INTR_SIGNAL_STATE			0x130
#define ZYNQMP_DP_TX_INTR_SIGNAL_STATE_HPD		BIT(0)
#define ZYNQMP_DP_TX_INTR_SIGNAL_STATE_REQUEST		BIT(1)
#define ZYNQMP_DP_TX_INTR_SIGNAL_STATE_REPLY		BIT(2)
#define ZYNQMP_DP_TX_INTR_SIGNAL_STATE_REPLY_TIMEOUT	BIT(3)
#define ZYNQMP_DP_TX_AUX_REPLY_DATA			0x134
#define ZYNQMP_DP_TX_AUX_REPLY_CODE			0x138
#define ZYNQMP_DP_TX_AUX_REPLY_CODE_AUX_ACK		(0)
#define ZYNQMP_DP_TX_AUX_REPLY_CODE_AUX_NACK		BIT(0)
#define ZYNQMP_DP_TX_AUX_REPLY_CODE_AUX_DEFER		BIT(1)
#define ZYNQMP_DP_TX_AUX_REPLY_CODE_I2C_ACK		(0)
#define ZYNQMP_DP_TX_AUX_REPLY_CODE_I2C_NACK		BIT(2)
#define ZYNQMP_DP_TX_AUX_REPLY_CODE_I2C_DEFER		BIT(3)
#define ZYNQMP_DP_TX_AUX_REPLY_CNT			0x13c
#define ZYNQMP_DP_TX_AUX_REPLY_CNT_MASK			0xff
#define ZYNQMP_DP_TX_INTR_STATUS			0x140
#define ZYNQMP_DP_TX_INTR_MASK				0x144
#define ZYNQMP_DP_TX_INTR_HPD_IRQ			BIT(0)
#define ZYNQMP_DP_TX_INTR_HPD_EVENT			BIT(1)
#define ZYNQMP_DP_TX_INTR_REPLY_RECV			BIT(2)
#define ZYNQMP_DP_TX_INTR_REPLY_TIMEOUT			BIT(3)
#define ZYNQMP_DP_TX_INTR_HPD_PULSE			BIT(4)
#define ZYNQMP_DP_TX_INTR_EXT_PKT_TXD			BIT(5)
#define ZYNQMP_DP_TX_INTR_LIV_ABUF_UNDRFLW		BIT(12)
#define ZYNQMP_DP_TX_INTR_VBLANK_START			BIT(13)
#define ZYNQMP_DP_TX_INTR_PIXEL0_MATCH			BIT(14)
#define ZYNQMP_DP_TX_INTR_PIXEL1_MATCH			BIT(15)
#define ZYNQMP_DP_TX_INTR_CHBUF_UNDERFLW_MASK		0x3f0000
#define ZYNQMP_DP_TX_INTR_CHBUF_OVERFLW_MASK		0xfc00000
#define ZYNQMP_DP_TX_INTR_CUST_TS_2			BIT(28)
#define ZYNQMP_DP_TX_INTR_CUST_TS			BIT(29)
#define ZYNQMP_DP_TX_INTR_EXT_VSYNC_TS			BIT(30)
#define ZYNQMP_DP_TX_INTR_VSYNC_TS			BIT(31)
#define ZYNQMP_DP_TX_INTR_ALL				(ZYNQMP_DP_TX_INTR_HPD_IRQ | \
							 ZYNQMP_DP_TX_INTR_HPD_EVENT | \
							 ZYNQMP_DP_TX_INTR_REPLY_RECV | \
							 ZYNQMP_DP_TX_INTR_REPLY_TIMEOUT | \
							 ZYNQMP_DP_TX_INTR_HPD_PULSE | \
							 ZYNQMP_DP_TX_INTR_EXT_PKT_TXD | \
							 ZYNQMP_DP_TX_INTR_LIV_ABUF_UNDRFLW | \
							 ZYNQMP_DP_TX_INTR_CHBUF_UNDERFLW_MASK | \
							 ZYNQMP_DP_TX_INTR_CHBUF_OVERFLW_MASK)
#define ZYNQMP_DP_TX_NO_INTR_ALL			(ZYNQMP_DP_TX_INTR_PIXEL0_MATCH | \
							 ZYNQMP_DP_TX_INTR_PIXEL1_MATCH | \
							 ZYNQMP_DP_TX_INTR_CUST_TS_2 | \
							 ZYNQMP_DP_TX_INTR_CUST_TS | \
							 ZYNQMP_DP_TX_INTR_EXT_VSYNC_TS | \
							 ZYNQMP_DP_TX_INTR_VSYNC_TS)
#define ZYNQMP_DP_TX_REPLY_DATA_CNT			0x148
#define ZYNQMP_DP_SUB_TX_INTR_STATUS			0x3a0
#define ZYNQMP_DP_SUB_TX_INTR_MASK			0x3a4
#define ZYNQMP_DP_SUB_TX_INTR_EN			0x3a8
#define ZYNQMP_DP_SUB_TX_INTR_DS			0x3ac

/* Main stream attribute registers */
#define ZYNQMP_DP_TX_MAIN_STREAM_HTOTAL			0x180
#define ZYNQMP_DP_TX_MAIN_STREAM_VTOTAL			0x184
#define ZYNQMP_DP_TX_MAIN_STREAM_POLARITY		0x188
#define ZYNQMP_DP_TX_MAIN_STREAM_POLARITY_HSYNC_SHIFT	0
#define ZYNQMP_DP_TX_MAIN_STREAM_POLARITY_VSYNC_SHIFT	1
#define ZYNQMP_DP_TX_MAIN_STREAM_HSWIDTH		0x18c
#define ZYNQMP_DP_TX_MAIN_STREAM_VSWIDTH		0x190
#define ZYNQMP_DP_TX_MAIN_STREAM_HRES			0x194
#define ZYNQMP_DP_TX_MAIN_STREAM_VRES			0x198
#define ZYNQMP_DP_TX_MAIN_STREAM_HSTART			0x19c
#define ZYNQMP_DP_TX_MAIN_STREAM_VSTART			0x1a0
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0			0x1a4
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0_SYNC		BIT(0)
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0_FORMAT_SHIFT	1
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0_DYNAMIC_RANGE	BIT(3)
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0_YCBCR_COLRIMETRY	BIT(4)
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0_BPC_SHIFT	5
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC1			0x1a8
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0_INTERLACED_VERT	BIT(0)
#define ZYNQMP_DP_TX_MAIN_STREAM_MISC0_STEREO_VID_SHIFT	1
#define ZYNQMP_DP_TX_M_VID				0x1ac
#define ZYNQMP_DP_TX_TRANSFER_UNIT_SIZE			0x1b0
#define ZYNQMP_DP_TX_DEF_TRANSFER_UNIT_SIZE		64
#define ZYNQMP_DP_TX_N_VID				0x1b4
#define ZYNQMP_DP_TX_USER_PIXEL_WIDTH			0x1b8
#define ZYNQMP_DP_TX_USER_DATA_CNT_PER_LANE		0x1bc
#define ZYNQMP_DP_TX_MIN_BYTES_PER_TU			0x1c4
#define ZYNQMP_DP_TX_FRAC_BYTES_PER_TU			0x1c8
#define ZYNQMP_DP_TX_INIT_WAIT				0x1cc

/* PHY configuration and status registers */
#define ZYNQMP_DP_TX_PHY_CONFIG				0x200
#define ZYNQMP_DP_TX_PHY_CONFIG_PHY_RESET		BIT(0)
#define ZYNQMP_DP_TX_PHY_CONFIG_GTTX_RESET		BIT(1)
#define ZYNQMP_DP_TX_PHY_CONFIG_PHY_PMA_RESET		BIT(8)
#define ZYNQMP_DP_TX_PHY_CONFIG_PHY_PCS_RESET		BIT(9)
#define ZYNQMP_DP_TX_PHY_CONFIG_ALL_RESET		(ZYNQMP_DP_TX_PHY_CONFIG_PHY_RESET | \
							 ZYNQMP_DP_TX_PHY_CONFIG_GTTX_RESET | \
							 ZYNQMP_DP_TX_PHY_CONFIG_PHY_PMA_RESET | \
							 ZYNQMP_DP_TX_PHY_CONFIG_PHY_PCS_RESET)
#define ZYNQMP_DP_TX_PHY_PREEMPHASIS_LANE_0		0x210
#define ZYNQMP_DP_TX_PHY_PREEMPHASIS_LANE_1		0x214
#define ZYNQMP_DP_TX_PHY_PREEMPHASIS_LANE_2		0x218
#define ZYNQMP_DP_TX_PHY_PREEMPHASIS_LANE_3		0x21c
#define ZYNQMP_DP_TX_PHY_VOLTAGE_DIFF_LANE_0		0x220
#define ZYNQMP_DP_TX_PHY_VOLTAGE_DIFF_LANE_1		0x224
#define ZYNQMP_DP_TX_PHY_VOLTAGE_DIFF_LANE_2		0x228
#define ZYNQMP_DP_TX_PHY_VOLTAGE_DIFF_LANE_3		0x22c
#define ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING		0x234
#define ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_162	0x1
#define ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_270	0x3
#define ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_540	0x5
#define ZYNQMP_DP_TX_PHY_POWER_DOWN			0x238
#define ZYNQMP_DP_TX_PHY_POWER_DOWN_LANE_0		BIT(0)
#define ZYNQMP_DP_TX_PHY_POWER_DOWN_LANE_1		BIT(1)
#define ZYNQMP_DP_TX_PHY_POWER_DOWN_LANE_2		BIT(2)
#define ZYNQMP_DP_TX_PHY_POWER_DOWN_LANE_3		BIT(3)
#define ZYNQMP_DP_TX_PHY_POWER_DOWN_ALL			0xf
#define ZYNQMP_DP_TX_PHY_PRECURSOR_LANE_0		0x23c
#define ZYNQMP_DP_TX_PHY_PRECURSOR_LANE_1		0x240
#define ZYNQMP_DP_TX_PHY_PRECURSOR_LANE_2		0x244
#define ZYNQMP_DP_TX_PHY_PRECURSOR_LANE_3		0x248
#define ZYNQMP_DP_TX_PHY_POSTCURSOR_LANE_0		0x24c
#define ZYNQMP_DP_TX_PHY_POSTCURSOR_LANE_1		0x250
#define ZYNQMP_DP_TX_PHY_POSTCURSOR_LANE_2		0x254
#define ZYNQMP_DP_TX_PHY_POSTCURSOR_LANE_3		0x258
#define ZYNQMP_DP_SUB_TX_PHY_PRECURSOR_LANE_0		0x24c
#define ZYNQMP_DP_SUB_TX_PHY_PRECURSOR_LANE_1		0x250
#define ZYNQMP_DP_TX_PHY_STATUS				0x280
#define ZYNQMP_DP_TX_PHY_STATUS_PLL_LOCKED_SHIFT	4
#define ZYNQMP_DP_TX_PHY_STATUS_FPGA_PLL_LOCKED		BIT(6)

/* Audio registers */
#define ZYNQMP_DP_TX_AUDIO_CONTROL			0x300
#define ZYNQMP_DP_TX_AUDIO_CHANNELS			0x304
#define ZYNQMP_DP_TX_AUDIO_INFO_DATA			0x308
#define ZYNQMP_DP_TX_AUDIO_M_AUD			0x328
#define ZYNQMP_DP_TX_AUDIO_N_AUD			0x32c
#define ZYNQMP_DP_TX_AUDIO_EXT_DATA			0x330

#define ZYNQMP_DP_MISC0_RGB				(0)
#define ZYNQMP_DP_MISC0_YCRCB_422			(5 << 1)
#define ZYNQMP_DP_MISC0_YCRCB_444			(6 << 1)
#define ZYNQMP_DP_MISC0_FORMAT_MASK			0xe
#define ZYNQMP_DP_MISC0_BPC_6				(0 << 5)
#define ZYNQMP_DP_MISC0_BPC_8				(1 << 5)
#define ZYNQMP_DP_MISC0_BPC_10				(2 << 5)
#define ZYNQMP_DP_MISC0_BPC_12				(3 << 5)
#define ZYNQMP_DP_MISC0_BPC_16				(4 << 5)
#define ZYNQMP_DP_MISC0_BPC_MASK			0xe0
#define ZYNQMP_DP_MISC1_Y_ONLY				(1 << 7)

#define ZYNQMP_DP_MAX_LANES				2
#define ZYNQMP_MAX_FREQ					3000000

#define DP_REDUCED_BIT_RATE				162000
#define DP_HIGH_BIT_RATE				270000
#define DP_HIGH_BIT_RATE2				540000
#define DP_MAX_TRAINING_TRIES				5
#define DP_V1_2						0x12

volatile uint32_t* dp_regs;

uint32_t dp(int reg)
{
  //printf("Accessing register 0x%03x\n", reg);
  //fflush(stdout);
  //sleep(1);
  
  return dp_regs[reg/4];  // Convert to 32-bit-word address to byte address
}

int main()
{
  //printf("Allocating dp_regs\n");
  //fflush(stdout);
  //sleep(1);
  
  dp_regs = hwalloc<uint32_t>(DP_BASE_ADDR, REG_LENGTH);
    
  uint32_t DP_Version = dp(ZYNQMP_DP_TX_CORE_ID);
  int DP_Major = (DP_Version & ZYNQMP_DP_TX_CORE_ID_MAJOR_MASK)     >> ZYNQMP_DP_TX_CORE_ID_MAJOR_SHIFT;
  int DP_Minor = (DP_Version & ZYNQMP_DP_TX_CORE_ID_MINOR_MASK)     >> ZYNQMP_DP_TX_CORE_ID_MINOR_SHIFT;
  int DP_Rev   = (DP_Version & ZYNQMP_DP_TX_CORE_ID_REVISION_MASK)  >> ZYNQMP_DP_TX_CORE_ID_REVISION_SHIFT;
  printf("ZynqMP DP core version is %d.%d.%d\n", DP_Major, DP_Minor, DP_Rev);

  for(;;)
    {
      int width = dp(ZYNQMP_DP_TX_MAIN_STREAM_HRES);
      int height = dp(ZYNQMP_DP_TX_MAIN_STREAM_VRES);

      uint32_t phy_status     = dp(ZYNQMP_DP_TX_PHY_STATUS);
      bool     phy_pll_locked = !!(phy_status & (1<<ZYNQMP_DP_TX_PHY_STATUS_PLL_LOCKED_SHIFT));

      bool xmit_enable     = !!(dp(ZYNQMP_DP_TX_ENABLE) & 1);

      uint32_t link_rate_setting = dp(ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING);
      const char* link_rate_txt = ( link_rate_setting==ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_162 ? "1.62Gb/s" :
				    link_rate_setting==ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_270 ? "2.70Gb/s" :
				    link_rate_setting==ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_540 ? "5.40Gb/s" :
				    /**/                                                             "bad     " );
      bool link_rate_good = ( link_rate_setting==ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_162 ? true  :
			      link_rate_setting==ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_270 ? true  :
			      link_rate_setting==ZYNQMP_DP_TX_PHY_CLOCK_FEEDBACK_SETTING_540 ? true  :
			      /**/                                                             false );

      
      uint32_t training = dp(ZYNQMP_DP_TX_TRAINING_PATTERN_SET);
      const char* training_txt = ( (training==0) ? "OFF" :
				   (training==1) ? "p1 " :
				   (training==2) ? "p2 " :
				   (training==3) ? "p3 " :
				   /**/            "BAD" );
      
      uint32_t scramble_disable = dp(ZYNQMP_DP_TX_SCRAMBLING_DISABLE);
      const char* scramble_txt = ( (scramble_disable==1) ? "NO "  :
				   (scramble_disable==0) ? "YES"  :
				   /**/                    "BAD"  );


      bool OK = (scramble_disable==0) && (training==0) && link_rate_good && xmit_enable && phy_pll_locked;
      const char* OK_txt = OK ? "YES" : "NO ";
      
      printf("DP %dx%d, PHY_PLL_LOCKED=%s, XMIT_ENABLE=%s, LINK_RATE=%s, SCRAMBLE=%s, TRAIN=%s, OK=%s\n", width, height, phy_pll_locked?"YES":"NO ", xmit_enable?"YES":"NO ", link_rate_txt, scramble_txt, training_txt, OK_txt);
      sleep(1);
    }
  

  hwfree(dp_regs, REG_LENGTH);
  return 0;
}
