// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#ifndef MENU_ADC_CONTROL_HH
#define MENU_ADC_CONTROL_HH

#include "menu.hh"
#include "select_one.hh"
#include "event_types.hh"
#include "number_box.hh"
#include "xrfdc.h"


#define MAX_NUM_ADCS 8
extern int NUM_ADCS;

class menu_adc_control;

struct adc_info
{
  int                                     adc_number_hw;
  int                                     adc_tile;
  int                                     adc_block;
  int                                     nyquist_zone;
  bool                                    dither_enabled;
  u8                                      calibration_mode;
  bool                                    calibration_freeze;
  menu_adc_control*                       control;
};


extern adc_info adcs[MAX_NUM_ADCS];

class menu_adc_control : public menu
{
  void add(menu* m); // This type of menu is terminal; can't add children.  Thus make this private.
  
public:

  adc_info*  adc;

  select_one*  nyquist_button;
  select_one*  dither_button;
  select_one*  calibration_freeze_button;
  select_one*  calibration_mode_button;

  int nyquist_button_x_offset;
  int nyquist_button_y_offset;
  int nyquist_button_width;
  int nyquist_button_height;

  int dither_button_x_offset;
  int dither_button_y_offset;
  int dither_button_width;
  int dither_button_height;

  int calibration_freeze_button_x_offset;
  int calibration_freeze_button_y_offset;
  int calibration_freeze_button_width;
  int calibration_freeze_button_height;

  int calibration_mode_button_x_offset;
  int calibration_mode_button_y_offset;
  int calibration_mode_button_width;
  int calibration_mode_button_height;

  connection<select_one, selection_changed_event, menu_adc_control, int, int>* select_trigger;

  const static color REGIONCOLOR        = GREY3;

  virtual bool handle_event(my_event& me);
  virtual void draw_dirty();
  virtual void draw_dynamic();
  virtual void layout();

  virtual void this_selected();

  void apply_settings();
  void update_settings_from_hw();
  
  menu_adc_control(adc_info&, color bg=BLACK);
  virtual ~menu_adc_control();
};


//
// 0 = Gen 1
// 1 = Gen 2
// 2 = Gen 3  ( XRFDC_GEN3 is 2 )
//
extern int XRFdc_Version;

//
// Calibration mode for Gen1 & Gen2 may be MODE_1 or MODE_2.
// Calibration mode for Gen 3 may be AUTO, MODE_1, or MODE_2.
//
inline const char* number_to_calibration_mode(int mode)
{
  const char* ret;
  
  if(XRFdc_Version < XRFDC_GEN3)
    {
      if(mode == XRFDC_CALIB_MODE1)
	ret = "Mode 1";
      else if(mode == XRFDC_CALIB_MODE2)
	ret = "Mode 2";
      else
	ret = "Bad";
    }
  else
    {
      if(mode == XRFDC_CALIB_MODE_AUTO)
	ret = "AUTO";
      else if(mode == XRFDC_CALIB_MODE1)
	ret = "Mode 1";
      else if(mode == XRFDC_CALIB_MODE2)
	ret = "Mode 2";
      else
	ret = "Bad";
    }

  return ret;
}


#endif
