// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include "bxbapp_config.hh"
#include "parse_support.hh"
#include "stdlib.h"

String                 config_board_name;
int                    current_config_number = -1;
struct bxbapp_config   current_config;
class  bxbapp_configs* allowable_configurations = 0;

double sampling_rate_kHz        = 0; //3932160;
double sampling_rate_MHz        = 0; //sampling_rate_kHz / 1000.0;
double dac_sampling_rate_kHz    = 0; //3932160;
double dac_sampling_rate_MHz    = 0; //sampling_rate_kHz / 1000.0;
int    num_freq_bins            = 0; //49152;
double freq_bin_separation_kHz  = 0; //sampling_rate_kHz / num_freq_bins;
double freq_bin_separation_MHz  = 0; //freq_bin_separation_kHz / 1000.0;
int    adc_samples_per_clock    = 0; //8;
int    dac_samples_per_clock    = 0; //8;
int    time_capture_length      = 0; //65536
double spectral_gain_adjust_dB  = 0; // 0dB

void mark_current_config(int number)
{
  if(number<0 || number>=allowable_configurations->num_configs)
    number = 0;
  
  if(number==current_config_number)
    return;

  current_config_number = number;
  current_config = allowable_configurations->configs[number];

  sampling_rate_kHz        = current_config.adc_sampling_rate_hz / 1000.0;
  sampling_rate_MHz        = sampling_rate_kHz / 1000.0;
  dac_sampling_rate_kHz    = current_config.dac_sampling_rate_hz / 1000.0;
  dac_sampling_rate_MHz    = dac_sampling_rate_kHz / 1000.0;
  num_freq_bins            = current_config.fft_output_size;
  freq_bin_separation_kHz  = sampling_rate_kHz / num_freq_bins;
  freq_bin_separation_MHz  = freq_bin_separation_kHz / 1000.0;
  adc_samples_per_clock    = current_config.adc_points_per_clock;
  dac_samples_per_clock    = current_config.dac_points_per_clock;
  time_capture_length      = current_config.time_capture_length;
  spectral_gain_adjust_dB  = current_config.spectral_gain_adjust_dB;
  
  printf("Setting current configuration to number %d\n", number);
  current_config.print();

  //
  // Some boards require the PL to be loaded prior to clock configuration.
  // So load the FPGA first, even though it seems backwards to load the
  // FPGA before the clocks that are required to make it work.  Also,
  // for this app the PL is accessed to get some info for the splash screen.
  //
  printf("Executing command '%s'\n", (char*)current_config.fpga_config_command);
  int status = system(current_config.fpga_config_command);
  printf("return status was %d\n", status);
}

void activate_current_config()
{
  int status;

  if(current_config.adc_multiplier != 1 || current_config.dac_multiplier != 1)
    {
      char temp[1000];
      double base_freq = sampling_rate_MHz / current_config.adc_multiplier;
      
      sprintf(temp, "%s %f %d %d",
	      (char*)current_config.clock_config_command,
	      base_freq,
	      current_config.adc_multiplier,
	      current_config.dac_multiplier);
	      
      printf("Executing command '%s'\n", temp);
      status = system(temp);
      printf("return status was %d\n", status);
    }
  else
    {
      printf("Executing command '%s'\n", (char*)current_config.clock_config_command);
      status = system(current_config.clock_config_command);
      printf("return status was %d\n", status);
    }
}

void bxbapp_config::set_adc_sampling_rate_hz(double freq_Hz)
{
  adc_sampling_rate_hz     = freq_Hz;
  sampling_rate_kHz        = adc_sampling_rate_hz / 1000.0;
  sampling_rate_MHz        = sampling_rate_kHz / 1000.0;

  freq_bin_separation_kHz  = sampling_rate_kHz / num_freq_bins;
  freq_bin_separation_MHz  = freq_bin_separation_kHz / 1000.0;

  // Currently, DAC sampling rate changes in sync with ADC sampling rate.
  dac_sampling_rate_hz         = freq_Hz * dac_multiplier / adc_multiplier;
  dac_sampling_rate_kHz        = dac_sampling_rate_hz / 1000.0;
  dac_sampling_rate_MHz        = dac_sampling_rate_kHz / 1000.0;
}

void bxbapp_config::set_dac_sampling_rate_hz(double freq_Hz)
{
  // Currently, change in sync with adc sampling rate.
  //dac_sampling_rate_hz         = freq_Hz;
  //dac_sampling_rate_kHz        = dac_sampling_rate_hz / 1000.0;
  //dac_sampling_rate_MHz        = dac_sampling_rate_kHz / 1000.0;
}

bxbapp_configs::bxbapp_configs(const char* filename)
{
  int c;
  
  num_configs = 0;
  
  FILE* fp = fopen(filename, "r");

  if(!fp)
    {
      printf("ERROR:  Can't open configuration file \"%s\".\n", filename);
      exit(20);
    }

  readstream r(filename);

  printf("Parsing BxBApp configuration file \"%s\".\n", r.getfilename());

  c = skipCommentsCpp(r);

  if(matchString(r, "board"))
    {
      c = skipCommentsCpp(r);
      config_board_name = readQuotedString(r);
    }

  c = skipCommentsCpp(r);

  if(!matchString(r, "config"))
    {
      printf("Expecting \"config\" statement.\n");
      parse_error(r);
    }
  
  for(;;) // Loop for configurations
    {

      c = skipCommentsCpp(r);

      if(c==EOF)
	break;
      
      if(num_configs>=MAX_CONFIGS)
	{
	  printf("ERROR: Exceeded maximum number of configurations, which is currently %d\n", MAX_CONFIGS);
	  parse_error(r);
	}
	      
      configs[num_configs].name = readQuotedString(r);
      
      for(;;) // loop for arguments for this configuration
	{
	  c = skipCommentsCpp(r);

	  if(c==EOF)
	    break;
	  
	  if(matchString(r, "description"))
	    {
	      c = skipCommentsCpp(r);
	      configs[num_configs].description = readExtendedQuotedString(r);
	    }
	  else if(matchString(r, "fpga_config_command"))
	    {
	      c = skipCommentsCpp(r);
	      configs[num_configs].fpga_config_command = readExtendedQuotedString(r);
	    }
	  else if(matchString(r, "clock_config_command"))
	    {
	      c = skipCommentsCpp(r);
	      configs[num_configs].clock_config_command = readExtendedQuotedString(r);
	    }
	  else if(matchString(r, "adc_sampling_rate_hz"))
	    {
	      configs[num_configs].adc_sampling_rate_hz = readLong(r);
	    }
	  else if(matchString(r, "dac_sampling_rate_hz"))
	    {
	      configs[num_configs].dac_sampling_rate_hz = readLong(r);
	    }
	  else if(matchString(r, "adc_multiplier"))
	    {
	      configs[num_configs].adc_multiplier = readInt(r);
	    }
	  else if(matchString(r, "dac_multiplier"))
	    {
	      configs[num_configs].dac_multiplier = readInt(r);
	    }
	  else if(matchString(r, "dac_loopback"))
	    {
	      configs[num_configs].dac_loopback = (readInt(r)!=0);
	    }
	  else if(matchString(r, "points_per_clock"))
	    {
	      configs[num_configs].dac_points_per_clock = configs[num_configs].adc_points_per_clock = readInt(r);
	    }
	  else if(matchString(r, "adc_points_per_clock"))
	    {
	      configs[num_configs].adc_points_per_clock = readInt(r);
	    }
	  else if(matchString(r, "dac_points_per_clock"))
	    {
	      configs[num_configs].dac_points_per_clock = readInt(r);
	    }
	  else if(matchString(r, "time_capture_length"))
	    {
	      configs[num_configs].time_capture_length = readInt(r);
	    }
	  else if(matchString(r, "fft_output_size"))
	    {
	      configs[num_configs].fft_output_size = readInt(r);
	    }
	  else if(matchString(r, "spectral_gain_adjust_dB"))
	    {
	      configs[num_configs].spectral_gain_adjust_dB = readFloat(r);
	    }
	  else if(matchString(r, "selectable_sampling_rate"))
	    {
	      configs[num_configs].selectable_sampling_rate = (readInt(r)!=0);
	    }
	  else if(matchString(r, "restart_app"))
	    {
	      configs[num_configs].restart_app = (readInt(r)!=0);
	    }
	  else if(matchString(r, "spec_max_adc_sample_hz"))
	    {
	      configs[num_configs].spec_max_adc_sample_hz = readLong(r);
	    }
	  else if(matchString(r, "config"))
	    {
	      break;
	    }
	  else
	    {
	      printf("Expecting one of these keywords:\n"
		     "\n"
		     "  description\n"
		     "  fpga_config_command\n"
		     "  clock_config_command\n"
		     "  adc_sampling_rate_hz\n"
		     "  dac_sampling_rate_hz\n"
		     "  adc_multiplier\n"
		     "  dac_multiplier\n"
		     "  dac_loopback\n"
		     "  points_per_clock\n"
		     "  adc_points_per_clock\n"
		     "  dac_points_per_clock\n"
		     "  time_capture_length\n"
		     "  fft_output_size\n"
		     "  spectral_gain_adjust_dB\n"
		     "  selectable_sampling_rate\n"
		     "  restart_app\n"
		     "  spec_max_adc_sample_hz\n"
		     "\n");
	      parse_error(r);
	    }
	}

      printf("Read configuration %d from file \"%s\":\n", num_configs, r.getfilename());
      configs[num_configs].print();

      num_configs++;
    }
}


void bxbapp_config::print()
{
  printf("\n"
	 "config \"%s\"\n"
	 "\n"
	 "  description              \"%s\"\n"
	 "  fpga_config_command      \"%s\"\n"
	 "  clock_config_command     \"%s\"\n"
	 "  adc_sampling_rate_hz     %ld\n"
	 "  dac_sampling_rate_hz     %ld\n"
	 "  adc_multiplier           %d\n"
	 "  dac_multiplier           %d\n"
	 "  dac_loopback             %d\n"
	 "  adc_points_per_clock     %d\n"
	 "  dac_points_per_clock     %d\n"
	 "  time_capture_length      %d\n"
	 "  fft_output_size          %d\n"
	 "  spectral_gain_adjust_dB  %f\n"
	 "  selectable_sampling_rate %d\n"
	 "  restart_app              %d\n"
	 "  spec_max_adc_sample_hz   %ld\n"
	 "\n",
	 (char*)name,
	 (char*)description,
	 (char*)fpga_config_command,
	 (char*)clock_config_command,
	 adc_sampling_rate_hz,
	 dac_sampling_rate_hz,
	 adc_multiplier,
	 dac_multiplier,
	 dac_loopback,
	 adc_points_per_clock,
	 dac_points_per_clock,
	 time_capture_length,
	 fft_output_size,
	 spectral_gain_adjust_dB,
	 selectable_sampling_rate,
	 restart_app,
	 spec_max_adc_sample_hz);
}


bxbapp_config::bxbapp_config()
{
  // These are bad values, but I didn't want to leave them totally undefined by default.
  adc_sampling_rate_hz     = 1000000;
  dac_sampling_rate_hz     = 1000000;
  adc_multiplier           = 1;
  dac_multiplier           = 1;
  dac_loopback             = true;
  adc_points_per_clock     = 8;
  dac_points_per_clock     = 8;
  time_capture_length      = 8192;
  fft_output_size          = 32768;
  spectral_gain_adjust_dB  = 0.0;
  selectable_sampling_rate = false;
  restart_app              = true;
  spec_max_adc_sample_hz   = 4096000000l;
}
