
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//



#include "data_update.hh"
#include "hwalloc.hh"
#include "bxbapp_config.hh"
#include "data_sources.hh"
#include "axis_scale.hh"
#include "RFSoC4x2_BxBDemo1.hh"

extern bool RFSoC4x2;


// DEBUG_PLOTS are enabled in graph_data_base.hh

#define MAX_HW_SOURCES 200
hw_source hw_source_list[MAX_HW_SOURCES];
int num_hw_sources = 0;

//bool freq2_got_data = false;

void add_x_axis_scale(graph_type       type,
		      int              source_number_start_sw,
		      int              source_number_count,
		      graph_data_type  data_type,
		      int              num_points,
		      int              interleave,
		      double           x_data_start_step_1,
		      double           xmin_step_1,
		      double           xmax_step_1,
		      double           ymin_scale_1,
		      double           ymax_scale_1)
{
  if(num_hw_sources==MAX_HW_SOURCES)
    {
      printf("ERROR:  More hardware data sources than are allowed for in data_update.cc.\n");
      exit(20);
    }

  for(int i=0; i<source_number_count; i++)
    {
      int source_number_sw = source_number_start_sw + i;

      hw_source_list[num_hw_sources].type                 = type;
      hw_source_list[num_hw_sources].source_number_sw     = source_number_sw;
      hw_source_list[num_hw_sources].data_type            = data_type;
      hw_source_list[num_hw_sources].num_points           = num_points;
      hw_source_list[num_hw_sources].interleave           = interleave;
      hw_source_list[num_hw_sources].x_data_start_step_1  = x_data_start_step_1;
      hw_source_list[num_hw_sources].xmin_step_1          = xmin_step_1;
      hw_source_list[num_hw_sources].xmax_step_1          = xmax_step_1;
      hw_source_list[num_hw_sources].ymin_scale_1         = ymin_scale_1;
      hw_source_list[num_hw_sources].ymax_scale_1         = ymax_scale_1;
      num_hw_sources++;
    }
}


void hw_source_list_init()
{
  num_hw_sources = 0;

  int    TCAPL   = time_capture_length;
  double TCAPSTR = -1.0 * sampling_rate_MHz; // -1 microsecond  default x graph start
  double TCAPEND =  2.0 * sampling_rate_MHz; // +2 microseconds default x graph end

  int    FBINS   = num_freq_bins;
  double FLFBINS = num_freq_bins;
  double F2FBINS = 2*num_freq_bins-1;
  double FLBINSM = num_freq_bins-1;

  int num_adcs = RFSoC4x2 ? 4 : 8;
  int num_dacs = RFSoC4x2 ? 2 : 8;

  add_x_axis_scale( TYPE_TIME,                    0,  num_adcs, TYPE_SINT16, TCAPL, 1,     0.0, TCAPSTR, TCAPEND, 0.0, 0.0 );

  add_x_axis_scale( TYPE_FREQ_Z1,                 0,  num_adcs, TYPE_FLOAT , FBINS, 4,     0.0,     0.0, FLBINSM, 0.0, 0.0 );
  add_x_axis_scale( TYPE_FREQ_Z2,                 0,  num_adcs, TYPE_FLOAT , FBINS, 4, FLFBINS, FLFBINS, F2FBINS, 0.0, 0.0 );
  
  if(current_config.dac_loopback)
    {
      add_x_axis_scale( TYPE_FREQ_Z1,             8,  num_dacs, TYPE_FLOAT , FBINS, 4,     0.0,     0.0, FLBINSM, 0.0, 0.0 );
      add_x_axis_scale( TYPE_FREQ_Z2,             8,  num_dacs, TYPE_FLOAT , FBINS, 4, FLFBINS, FLFBINS, F2FBINS, 0.0, 0.0 );
    }

  add_x_axis_scale( TYPE_Z1_TRANSFER_MAG,         0,  num_adcs, TYPE_FLOAT , FBINS, 4,     0.0,     0.0, FLBINSM, 0.0, 0.0 );
  add_x_axis_scale( TYPE_Z2_TRANSFER_MAG,         0,  num_adcs, TYPE_FLOAT , FBINS, 4, FLFBINS, FLFBINS, F2FBINS, 0.0, 0.0 );

  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE,       0,  num_adcs, TYPE_FLOAT , FBINS, 4,     0.0,     0.0, FLBINSM, 0.0, 0.0 );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE,       0,  num_adcs, TYPE_FLOAT , FBINS, 4, FLFBINS, FLFBINS, F2FBINS, 0.0, 0.0 );

  add_x_axis_scale( TYPE_Z1_TRANSFER_GROUP_DELAY, 0,  num_adcs, TYPE_FLOAT , FBINS, 4,     0.0,     0.0, FLBINSM, 0.0, 0.0 );
  add_x_axis_scale( TYPE_Z2_TRANSFER_GROUP_DELAY, 0,  num_adcs, TYPE_FLOAT , FBINS, 4, FLFBINS, FLFBINS, F2FBINS, 0.0, 0.0 );

  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE_DEBUG, 0,  num_adcs, TYPE_FLOAT , FBINS, 4,     0.0,     0.0, FLBINSM, 0.0, 0.0 );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE_DEBUG, 0,  num_adcs, TYPE_FLOAT , FBINS, 4, FLFBINS, FLFBINS, F2FBINS, 0.0, 0.0 );
     
  //add_x_axis_scale( TYPE_STATIC_TIME, -1,  TYPE_SINT16,  "DAC",    WHITE,      0xB1000000, TCAPL, 1,     0.0, 32000.0, 33000.0, 0.0, 0.0 );
}




#define MAX_X_AXIS_SCALES 100
axis_scale x_axis_scale_list[MAX_X_AXIS_SCALES];
int num_x_axis_scales = 0;

void add_x_axis_scale(graph_type type, const char *name, double scale, bool is_default)
{
  if(num_x_axis_scales==MAX_X_AXIS_SCALES)
    {
      printf("ERROR:  More x axis scale possibiliities than are allowed for in data_update.cc.\n");
      exit(20);
    }

  x_axis_scale_list[num_x_axis_scales].type       = type;
  x_axis_scale_list[num_x_axis_scales].scale      = scale;
  x_axis_scale_list[num_x_axis_scales].is_default = is_default;

  const char* s = name;
  char* d = x_axis_scale_list[num_x_axis_scales].name;

  while(*s)
    *d++ = *s++;
  *d=*s;

  num_x_axis_scales++;
}


void  x_axis_scale_list_init()
{
  num_x_axis_scales = 0;

  add_x_axis_scale( TYPE_TIME,        "Samples",         1.0,                               false );
  add_x_axis_scale( TYPE_TIME,        "Milliseconds",    1.0e-3 / sampling_rate_MHz,        false );
  add_x_axis_scale( TYPE_TIME,        "Microseconds",    1.0    / sampling_rate_MHz,        true  );
  add_x_axis_scale( TYPE_TIME,        "Nanoseconds",     1.0e3  / sampling_rate_MHz,        false );
  
  //    add_x_axis_scale( TYPE_STATIC_TIME, "Samples",         1.0,                               false );
  //    add_x_axis_scale( TYPE_STATIC_TIME, "Milliseconds",    1.0e-3 / sampling_rate_MHz,        false );
  //    add_x_axis_scale( TYPE_STATIC_TIME, "Microseconds",    1.0    / sampling_rate_MHz,        true  );
  //    add_x_axis_scale( TYPE_STATIC_TIME, "Nanoseconds",     1.0e3  / sampling_rate_MHz,        false );

  add_x_axis_scale( TYPE_FREQ_Z1,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_FREQ_Z1,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_FREQ_Z1,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_FREQ_Z1,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
  
  add_x_axis_scale( TYPE_FREQ_Z2,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_FREQ_Z2,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_FREQ_Z2,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_FREQ_Z2,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
  
  add_x_axis_scale( TYPE_Z1_TRANSFER_MAG,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_MAG,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_MAG,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z1_TRANSFER_MAG,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );

  add_x_axis_scale( TYPE_Z2_TRANSFER_MAG,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_MAG,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_MAG,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z2_TRANSFER_MAG,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
  
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
  
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );

  add_x_axis_scale( TYPE_Z1_TRANSFER_GROUP_DELAY,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_GROUP_DELAY,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_GROUP_DELAY,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z1_TRANSFER_GROUP_DELAY,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
  
  add_x_axis_scale( TYPE_Z2_TRANSFER_GROUP_DELAY,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_GROUP_DELAY,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_GROUP_DELAY,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z2_TRANSFER_GROUP_DELAY,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
  
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE_DEBUG,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE_DEBUG,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE_DEBUG,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z1_TRANSFER_PHASE_DEBUG,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
  
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE_DEBUG,     "Bins",            1.0,                               false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE_DEBUG,     "GHz",             sampling_rate_MHz/num_freq_bins/1000.0/2.0,  false );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE_DEBUG,     "MHz",             sampling_rate_MHz/num_freq_bins/2.0,     true  );
  add_x_axis_scale( TYPE_Z2_TRANSFER_PHASE_DEBUG,     "kHz",             sampling_rate_MHz/num_freq_bins*1000.0/2.0,      false );
};




#define MAX_Y_AXIS_SCALES 50

axis_scale y_axis_scale_list[MAX_Y_AXIS_SCALES];
int num_y_axis_scales = 0;

  
void add_y_axis_scale(graph_type type, const char *name, double scale, bool is_default)
{
  if(num_y_axis_scales==MAX_Y_AXIS_SCALES)
    {
      printf("ERROR:  More y axis scale possibiliities than are allowed for in data_update.cc.\n");
      exit(20);
    }

  y_axis_scale_list[num_y_axis_scales].type       = type;
  y_axis_scale_list[num_y_axis_scales].scale      = scale;
  y_axis_scale_list[num_y_axis_scales].is_default = is_default;

  const char* s = name;
  char* d = y_axis_scale_list[num_y_axis_scales].name;

  while(*s)
    *d++ = *s++;
  *d=*s;

  num_y_axis_scales++;
}

void y_axis_scale_list_init()
{
  num_y_axis_scales = 0;

  add_y_axis_scale( TYPE_TIME,        "Raw Value",  1.0,                true   );
  add_y_axis_scale( TYPE_TIME,        "Volts",      0.5  / 32768,       false  );
  add_y_axis_scale( TYPE_TIME,        "Millivolts", 0.5e3  / 32768,     false  );
  //    add_y_axis_scale( TYPE_TIME,        "Microvolts", 1.0e6  / 32768 }
  
  //    add_y_axis_scale( TYPE_STATIC_TIME, "Raw Value",  1.0,                false );
  //    add_y_axis_scale( TYPE_STATIC_TIME, "Volts",      1.0  / 32768,       false );
  //    add_y_axis_scale( TYPE_STATIC_TIME, "Millivolts", 1.0e3  / 32768,     true  );
  //    add_y_axis_scale( TYPE_STATIC_TIME, "Microvolts", 1.0e6  / 32768 }
  
  add_y_axis_scale( TYPE_FREQ_Z1,     "dBFS",       1.0,                true   );
  add_y_axis_scale( TYPE_FREQ_Z1,     "~dBm",       1.125*1.125,        false  );
  add_y_axis_scale( TYPE_FREQ_Z2,     "dBFS",       1.0,                true   );
  add_y_axis_scale( TYPE_FREQ_Z2,     "~dBm",       1.125*1.125,        false  );
  
  add_y_axis_scale( TYPE_Z1_TRANSFER_MAG, "dB",        1.0,                true   );
  add_y_axis_scale( TYPE_Z2_TRANSFER_MAG, "dB",        1.0,                true   );
  
  add_y_axis_scale( TYPE_Z1_TRANSFER_PHASE, "Cycles",          1.0,        true   );
  add_y_axis_scale( TYPE_Z1_TRANSFER_PHASE, "Degrees",       360.0,        false  );
  add_y_axis_scale( TYPE_Z2_TRANSFER_PHASE, "Cycles",          1.0,        true   );
  add_y_axis_scale( TYPE_Z2_TRANSFER_PHASE, "Degrees",       360.0,        false  );
  
  add_y_axis_scale( TYPE_Z1_TRANSFER_GROUP_DELAY, "Nanoseconds",                        1.0,    true   );
  add_y_axis_scale( TYPE_Z1_TRANSFER_GROUP_DELAY, "Samples",        sampling_rate_MHz/1.0e3,    false  );
  add_y_axis_scale( TYPE_Z2_TRANSFER_GROUP_DELAY, "Nanoseconds",                        1.0,    true   );
  add_y_axis_scale( TYPE_Z2_TRANSFER_GROUP_DELAY, "Samples",        sampling_rate_MHz/1.0e3,    false  );
  
  add_y_axis_scale( TYPE_Z1_TRANSFER_PHASE_DEBUG, "Cycles",        1.0,    true   );
  add_y_axis_scale( TYPE_Z1_TRANSFER_PHASE_DEBUG, "Degrees",     360.0,    false  );
  add_y_axis_scale( TYPE_Z2_TRANSFER_PHASE_DEBUG, "Cycles",        1.0,    true   );
  add_y_axis_scale( TYPE_Z2_TRANSFER_PHASE_DEBUG, "Degrees",     360.0,    false  );
};
  


struct pfb_trigger
{
  uint32_t         mux_select;
  uint32_t         capture;
};



volatile hw_pfb_spectrum*  pfb_spectrum    = 0;
volatile pfb_trigger*      freq_trigger_1  = 0;
volatile pfb_trigger*      freq_trigger_2  = 0;

void data_init()
{
  if(pfb_spectrum)
    {
      printf("Error in data_update.cc::data_init():  Double allocate.\n");
      exit(20);
    }

  
  hw_source_list_init();
  x_axis_scale_list_init();
  y_axis_scale_list_init();
  
  
  pfb_spectrum   = hwalloc<hw_pfb_spectrum>  ( HW_RFSoC4x2_BxBDemo1::spect_spectral_capture_960_0.C_BASEADDR  /*0xA0400000*/);


  freq_trigger_1  = hwalloc<pfb_trigger> ( HW_RFSoC4x2_BxBDemo1::spectrum_capture_con_0.C_BASEADDR /*0xA0080000*/);
  freq_trigger_2  = hwalloc<pfb_trigger> ( HW_RFSoC4x2_BxBDemo1::spectrum_capture_con_1.C_BASEADDR /*0xA0090000*/);

  
  pfb_spectrum->collect = 0;
  freq_trigger_1->mux_select = 0;
  freq_trigger_2->mux_select = 0;

  
 #ifdef NOTDEF
  printf("\n"
	 "pfb_spectrum registers:\n"
         "  collect              = %d\n"
	 "  late_delay           = %d\n"
	 "  spectrum_done        = %d\n"
	 "  capture_offset       = 0x%08X\n"
	 "  samples_per_section  = %d\n"
	 "  num_sections         = %d\n"
	 "  num_streams_early    = %d\n"
	 "  num_streams_late     = %d\n"
	 "\n",
	 pfb_spectrum->collect,
	 pfb_spectrum->late_delay,
	 pfb_spectrum->spectrum_done,
	 pfb_spectrum->capture_offset,
	 pfb_spectrum->samples_per_section,
	 pfb_spectrum->num_sections,
	 pfb_spectrum->num_streams_early,
	 pfb_spectrum->num_streams_late);

  pfb_spectrum->collect = 1;

  printf("\n"
	 "pfb_spectrum registers:\n"
         "  collect              = %d\n"
	 "  late_delay           = %d\n"
	 "  spectrum_done        = %d\n"
	 "  capture_offset       = 0x%08X\n"
	 "  samples_per_section  = %d\n"
	 "  num_sections         = %d\n"
	 "  num_streams_early    = %d\n"
	 "  num_streams_late     = %d\n"
	 "\n",
	 pfb_spectrum->collect,
	 pfb_spectrum->late_delay,
	 pfb_spectrum->spectrum_done,
	 pfb_spectrum->capture_offset,
	 pfb_spectrum->samples_per_section,
	 pfb_spectrum->num_sections,
	 pfb_spectrum->num_streams_early,
	 pfb_spectrum->num_streams_late);

  sleep(1);


  printf("\n"
	 "pfb_spectrum registers:\n"
         "  collect              = %d\n"
	 "  late_delay           = %d\n"
	 "  spectrum_done        = %d\n"
	 "  capture_offset       = 0x%08X\n"
	 "  samples_per_section  = %d\n"
	 "  num_sections         = %d\n"
	 "  num_streams_early    = %d\n"
	 "  num_streams_late     = %d\n"
	 "\n",
	 pfb_spectrum->collect,
	 pfb_spectrum->late_delay,
	 pfb_spectrum->spectrum_done,
	 pfb_spectrum->capture_offset,
	 pfb_spectrum->samples_per_section,
	 pfb_spectrum->num_sections,
	 pfb_spectrum->num_streams_early,
	 pfb_spectrum->num_streams_late);
#endif
}


#define MAX_NUM_ADCS_DACS 16  // Include DACs also here, which can be captured for spectrum.


int adc_captured_at_this_time_1 = 0;
int adc_captured_at_this_time_2 = 0;

static int capture_time = -1;
static int adc_capture_times[MAX_NUM_ADCS_DACS];
static int adc_pair_capture_times[MAX_NUM_ADCS_DACS][MAX_NUM_ADCS_DACS];
static bool adc_capture_desired[MAX_NUM_ADCS_DACS];
static bool adc_pair_capture_desired[MAX_NUM_ADCS_DACS][MAX_NUM_ADCS_DACS];

void update_adc_not_captured_but_desired(int adc)
{
  adc_capture_desired[adc] = true;
}

void update_adc_pair_not_captured_but_desired(int adc1, int adc2)
{
  adc_pair_capture_desired[adc1][adc2] = true;
}


void update_adc_captured(int adc)
{
  adc_capture_desired[adc] = true;
  adc_capture_times[adc] = capture_time;
}

void update_adc_pair_captured(int adc1, int adc2)
{
  adc_pair_capture_desired[adc1][adc2] = true;
  adc_pair_capture_times[adc1][adc2] = capture_time;
}


//
// Also used by freq2_graph.hh in update_data().
//
bool update_data_done = true;

void data_start_capture()
{
  int adc = -1;
  int adc_capture_time = capture_time+1;
  int adc1 = -1;
  int adc2 = 0;
  int adc_pair_capture_time = capture_time+1;

  capture_time++;

  if(capture_time<0)
    {
      capture_time = 0;
      for(int i=0; i<MAX_NUM_ADCS_DACS; i++)
	{
	  adc_capture_times[i] = 0;
	  for(int j=0; j<MAX_NUM_ADCS_DACS; j++)
	    {
	      adc_pair_capture_times[i][j] = 0;
	    }
	}
    }

  for(int i=0; i<MAX_NUM_ADCS_DACS; i++)
    {
      if(adc_capture_desired[i] && adc_capture_times[i]<adc_capture_time)
	{
	  adc = i;
	  adc_capture_time = adc_capture_times[i];
	}
    }
  
  for(int i=0; i<MAX_NUM_ADCS_DACS; i++)
    for(int j=0; j<MAX_NUM_ADCS_DACS; j++)
      {
      if(adc_pair_capture_desired[i][j] && adc_pair_capture_times[i][j]<adc_pair_capture_time)
	{
	  adc1 = i;
	  adc2 = j;
	  adc_pair_capture_time = adc_pair_capture_times[i][j];
	}
    }

  if(adc_pair_capture_time>adc_capture_time)
    {
      // Don't collect a pair, but collect a second singleton if that is useful.
      adc1 = adc;
      adc2 = (adc1 + 1) % MAX_NUM_ADCS_DACS;

      // Find a second adc to capture
      adc_capture_time = capture_time+1;
      for(int i=0; i<MAX_NUM_ADCS_DACS; i++)
	{
	  if(i!=adc1 && adc_capture_desired[i] && adc_capture_times[i]<adc_capture_time)
	    {
	      adc2 = i;
	      adc_capture_time = adc_capture_times[i];
	    }
	}
    }
  else
    {
      // Collect a pair
    }
  
  if(adc1!=-1)
    {
      freq_trigger_1->mux_select = sw_source_to_hw_source(adc1);
      freq_trigger_2->mux_select = sw_source_to_hw_source(adc2);
      adc_captured_at_this_time_1 = adc1;
      adc_captured_at_this_time_2 = adc2;

      static int wait_timer = 0;

      //
      // Only cause a new collect to happen if the data has been picked
      // up and analyzed, as signified by update_data_done being true.
      // However, to prevent any possibility of an infinite loop
      // also have a timeout counter.
      //
      if(update_data_done)
	{
	  update_data_done = false;
	  pfb_spectrum->collect = 0;
	  pfb_spectrum->collect = 1;
	  wait_timer=0;
	}
      else
	{
	  if(wait_timer==10)
	    {
	      printf("Timout for analyzing data.  This shouldn't occur.  Resetting.\n");
	      update_data_done = true;
	      wait_timer=0;
	    }
	  wait_timer++;
	}	  
    }

  for(int i=0; i<MAX_NUM_ADCS_DACS; i++)
    {
      adc_capture_desired[i] = false;
      for(int j=0; j<MAX_NUM_ADCS_DACS; j++)
	{
	  adc_pair_capture_desired[i][j] = false;
	}
    }
}
