
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include "graph_data_hw.hh"
#include "List.hh"
#include "time_graph.hh"
#include "freq2_graph.hh"
#include "xfer_graph.hh"
#include "menu_adc_control.hh"
#include "data_update.hh"

graph* hw_graph_create(graph_config& gc)
{
  graph_data_base*  gd = 0;
  int               source_numbers_sw[graph::MAX_GRAPHS];
  int               reference = gc.reference;
  
  for(int i=0; i<gc.num_hw_sources; i++)
    {
      source_numbers_sw[i]  = hw_source_list[gc.source_list_number[i]].source_number_sw;
    }
  

  if(gc.type==TYPE_TIME)
    {
      gd = create_time_graph(gc.type,
			     gc.num_hw_sources,
			     source_numbers_sw,
			     gc.num_points,
			     gc.x_data_start,
			     gc.x_data_step,
			     gc.y_scale_factor);


      printf("Setting up time graph with length %d points.\n", gc.num_points);
    }
  else if(gc.type==TYPE_FREQ_Z1 || gc.type==TYPE_FREQ_Z2)
    {
      gd = create_freq2_graph(gc.type,
			      gc.num_hw_sources,
			      source_numbers_sw,
			      gc.num_points,
			      gc.x_data_start,
			      gc.x_data_step,
			      gc.y_scale_factor,
			      gc.averaging);

    }
  else if(gc.type==TYPE_Z1_TRANSFER_MAG          ||
	  gc.type==TYPE_Z2_TRANSFER_MAG          ||
	  gc.type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
	  gc.type==TYPE_Z2_TRANSFER_PHASE_DEBUG  ||
	  gc.type==TYPE_Z1_TRANSFER_PHASE        ||
	  gc.type==TYPE_Z2_TRANSFER_PHASE        ||
	  gc.type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
	  gc.type==TYPE_Z2_TRANSFER_GROUP_DELAY  )
    {
      gd = create_xfer_graph(gc.type,
			     gc.num_hw_sources,
			     source_numbers_sw,
			     reference,
			     gc.num_points,
			     gc.x_data_start,
			     gc.x_data_step,
			     gc.y_scale_factor,
			     gc.averaging);
    }

  
  if(!gd)
    {
      printf("Error creating hardware graph data.\n");
      abort();
    }
  
  auto g = new graph(gd,            // graph_data
		     gc.title,      // title
		     gc.x_label,    // x axis label
		     gc.y_label,    // y axis label
		     gc.xmin,       // xmin
		     gc.xmax,       // xmax
		     gc.ymin,       // ymin
		     gc.ymax);      // ymax

  if(gc.type==TYPE_Z1_TRANSFER_MAG          ||
     gc.type==TYPE_Z2_TRANSFER_MAG          ||
     gc.type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
     gc.type==TYPE_Z2_TRANSFER_PHASE_DEBUG  ||
     gc.type==TYPE_Z1_TRANSFER_PHASE        ||
     gc.type==TYPE_Z2_TRANSFER_PHASE        ||
     gc.type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z2_TRANSFER_GROUP_DELAY  )
    {
      for(int i=0; i<gd->num_graphs; i++)
	{
	  ((graph_data<float>*)gd)->y_data[i]->g = g;
	}

      // Hack below to give colors to reference DACs to match ADCs
      //g->GRAPH_COLORS[graph::MAX_GRAPHS-1] = RED;
      //for(int i=0; i<NUM_ADCS; i++)
      //{
      //  if(hw_source_list[i].adc_number==reference)
      //    {
      //      g->GRAPH_COLORS[graph::MAX_GRAPHS-1] = hw_source_list[i].source_color;
      //      break;
      //    }
      //}

      g->GRAPH_COLORS[graph::MAX_GRAPHS-1] = source_colors_sw(hw_source_list[reference].source_number_sw);
    }
  
  if(gc.type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
     gc.type==TYPE_Z2_TRANSFER_PHASE_DEBUG  ||
     gc.type==TYPE_Z1_TRANSFER_PHASE        ||
     gc.type==TYPE_Z2_TRANSFER_PHASE        ||
     gc.type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z2_TRANSFER_GROUP_DELAY  )
    {
      g->GRAPH_COLORS[1]                   = source_colors_sw(hw_source_list[gc.source_list_number[0]].source_number_sw);
      g->GRAPH_COLORS[0]                   = WHITE;
    }
  else
    {
      for(int i=0; i<gc.num_hw_sources; i++)
	g->GRAPH_COLORS[i] = source_colors_sw(hw_source_list[gc.source_list_number[i]].source_number_sw);
    }
  
  g->set_xmin_xmax(gc.xmin, gc.xmax);

  double ymin = gc.ymin;
  double ymax = gc.ymax;
  
  if(ymin==ymax)
    {
      ymin=ymax=0;
      g->gd->find_ymin_ymax(g->ymin_in_last_dataset, g->ymax_in_last_dataset);
    }
  
  g->set_ymin_ymax(ymin, ymax);

  //g->gc = &gc;
  g->external_xmin = &gc.xmin;
  g->external_xmax = &gc.xmax;
  
  return g;
}



graph_config default_gc(graph_type gt)
{
  graph_config gc;

  gc.type = gt;

  if(gt==TYPE_FREQ_Z1 || gt==TYPE_FREQ_Z2)
    gc.averaging = AVERAGING_HUGE;
  else
    gc.averaging = AVERAGING_HUGE;
  
  // Find first matching source.
  for(int i=0; i<num_hw_sources; i++)
    {
      struct hw_source* s = &hw_source_list[i];
      if(s->type==gt)
	{
	  // found a source
	  gc.data_type               = s->data_type;
	  gc.num_hw_sources          = 0;
	  gc.source_list_number[0]     = i;
	  gc.reference               = 0; //s->default_reference;
	  gc.num_points              = s->num_points;
	  gc.interleave              = s->interleave;
	  sprintf(gc.title, "Graph with no Specified Sources");

	  // Find first matching x-axis scale
	  bool first_x = true;
	  for(int i=0; i<num_x_axis_scales; i++)
	    {
	      struct axis_scale* xas = &x_axis_scale_list[i];
	      if(xas->type==gc.type && (first_x || xas->is_default))
		{
		  first_x = false;
		  gc.x_data_step   = xas->scale;
		  gc.x_data_start  = s->x_data_start_step_1 * xas->scale;
		  gc.xmin          = s->xmin_step_1         * xas->scale;
		  gc.xmax          = s->xmax_step_1         * xas->scale;
		  sprintf(gc.x_label, "%s", xas->name);
		}

	      if(i==num_x_axis_scales-1 && first_x)
		{
		  printf("ERROR: graph source has no allowable x-axis scales.\n");
		  abort();
		}
	    }

	  // Find first matching y-axis scale
	  bool first_y = true;
	  for(int i=0; i<num_y_axis_scales; i++)
	    {
	      struct axis_scale* yas = &y_axis_scale_list[i];
	      if(yas->type==gc.type && (first_y || yas->is_default))
		{
		  first_y = false;
		  gc.y_scale_factor  = yas->scale;
		  gc.ymin            = s->ymin_scale_1 * yas->scale;
		  gc.ymax            = s->ymax_scale_1 * yas->scale;
		  sprintf(gc.y_label, "%s", yas->name);
		}

	      if(i==num_y_axis_scales-1 && first_y)
		{
		  printf("ERROR: graph source has no allowable y-axis scales.\n");
		  abort();
		}
	    }
	  
	  break;
	}

      if(i==num_hw_sources-1)
	{
	  printf("ERROR: no source matching graph type.\n");
	  abort();
	}
    }
    
  return gc;
}



void print_gc (const graph_config& gc)
{
  printf("graph_config contents:\n"
	 "  type              = %d\n"
         "  data_type         = %d\n"
	 "  num_hw_sources    = %d\n",
	 gc.type, gc.data_type, gc.num_hw_sources);
	 
  for(int i=0; i<gc.num_hw_sources; i++)
    printf("    source_list_number[%d] = 0x%08x with reference 0x%08x\n", i, gc.source_list_number[i], gc.reference);

  printf("  num_points        = %d\n"
	 "  x_data_start      = %f\n"
	 "  x_data_step       = %f\n"
	 "  y_scale_factor    = %f\n"
	 "  title             = \"%s\"\n"
	 "  x_label           = \"%s\"\n"
	 "  y_label           = \"%s\"\n"
	 "  xmin              = %f\n"
	 "  xmax              = %f\n"
	 "  ymin              = %f\n"
	 "  ymax              = %f\n"
	 "\n",
	 gc.num_points,
	 gc.x_data_start,
	 gc.x_data_step,
	 gc.y_scale_factor,
	 gc.title,
	 gc.x_label,
	 gc.y_label,
	 gc.xmin,
	 gc.xmax,
	 gc.ymin,
	 gc.ymax);

}
