
// SPDX-License-Identifier: CC-BY-NC-ND-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial NoDerivatives 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-ND-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-ND-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  This work may be distributed in unmodified form,
// but derivatives of this work may not be distributed.  For further
// details, see the Creative Commons License "CC-BY-NC-ND-4.0".
//
// You should have received a copy of the CC-BY-NC-ND-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-nd/4.0/>.
//

#ifndef GRAPH_HH
#define GRAPH_HH

#include "panel.hh"
#include "subpanel.hh"
#include "color.hh"
#include "point.hh"
#include "multiwindow.hh"
#include "draw_text.hh"
#include "graph_data_base.hh"
#include "hotspot.hh"

class graph : public multiwindow
{
public:
  subpanel* graphpanel;  // A panel inside the main panel where graphs are done.  This is separate so the entire thing doesn't need to be redrawn with a data update.

  double* external_xmin;
  double* external_xmax;
  //  struct graph_config* gc;
  
  hotspot<graph>*  zoom_hotspot;
  hotspot<graph>*  xaxis_hotspot;
  hotspot<graph>*  yaxis_hotspot;


  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  //  Metadata
  //
  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  const static int MAX_GRAPHS = 13;
    
  const static color TITLE_COLOR      = WHITE;
  const static color AXIS_TITLE_COLOR = WHITE;
  const static color TIC_LABEL_COLOR  = GREY9;
  const static color AXIS_COLOR       = GREY8;
  const static color TIC_COLOR        = GREY9;
  const static color BG_COLOR         = GREY3;
  const static color PANEL_COLOR      = GREY2;
  const static color GRID_MINOR_COLOR = GREY5;
  const static color GRID_MAJOR_COLOR = GREY7;
  
  color* GRAPH_COLORS;
  
  char title[200];
  char x_axis_title[200];
  char y_axis_title[200];

  char* labels[MAX_GRAPHS];
  

  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  //  Data
  //
  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  graph_data_base*  gd;
  int               num_graphs;    // redundant with value in gd
  int               num_points;    // redundant with value in gd
  
  // These are the current values displayed on the graph
  double            xmin;
  double            xmax;
  double            ymin;
  double            ymax;

  // These are the values initially proposed, or saved whenever we enter the menu
  double            xmin_suggested;
  double            xmax_suggested;
  double            ymin_suggested;
  double            ymax_suggested;

  // These are the values seen in the provided data
  double            ymax_in_last_dataset;
  double            ymin_in_last_dataset;
  
  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  //  Calculated Tic Labels
  //
  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  const static int MAX_TIC_LABELS   = 22;
  const static int MAX_LABEL_LENGTH = 50;
  
  int    num_x_tic_labels;
  double x_tic_label_values[MAX_TIC_LABELS];
  bool   x_tic_label_drawn[MAX_TIC_LABELS];
  char   x_tic_label_text[MAX_TIC_LABELS][MAX_LABEL_LENGTH];
  double x_tic_unlabeled_start;
  double x_tic_unlabeled_step;
  int    x_tic_label_greatest_width;

  int    num_y_tic_labels;
  double y_tic_label_values[MAX_TIC_LABELS];
  bool   y_tic_label_drawn[MAX_TIC_LABELS];
  char   y_tic_label_text[MAX_TIC_LABELS][MAX_LABEL_LENGTH];
  double y_tic_unlabeled_start;
  double y_tic_unlabeled_step;
  int    y_tic_label_greatest_width;

  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  //  Geometry
  //
  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  // Width and height of the graph
  //int width;   // Must be the same as p->width, or geometry needs to be recalculated
  //int height;  // Must be the same as p->height, or geometry needs to be recalculated

  bool desire_title;
  bool desire_x_title;
  bool desire_y_title;

  bool draw_title;
  bool draw_x_title;
  bool draw_y_title;

  bool draw_x_tic_labels;
  bool draw_y_tic_labels;

  // font sizes
  int title_font_height;
  int axis_title_font_height;
  int axis_tic_label_font_height;

  // Size of drawn tics
  
  int minor_tic_length;
  int major_tic_length;
  
  // Widths, from left to right.  "width" or "margin" indicate how wide an item is.  "offset" indicates position from left edge of panel (0).

  int graph_left_margin;
  int y_title_width;           // The same value as axis_title_font_height
  int y_title_offset;          // Offset of y-axis title text, which is specified from a bottom center point, then rotated 90 degrees.
  int y_title_right_margin;
  int y_tic_label_width;       // Calculated carefully based on the length of the selected y-axis tic labels
  int y_tic_label_offset;      // Offset of y-axis tic text, which is specified from a right center point.
  int y_tic_label_margin;
  int graph_left_offset;

  // Widths, from right to left.

  int graph_right_margin;
  int graph_x_label_margin;
  int graph_right_offset;
  

  // Heights, from top to bottom.  "height" indicates how tall an item is.  "offset' indicates position from top (0)

  int graph_top_margin;
  int title_height;            // Same as title_font_height
  int title_offset;            // Offset of title text, specified from the bottom center point.
  int title_below_margin;
  int graph_top_offset;

  // Heights, from bottom to top.

  int graph_bottom_margin;
  int x_title_offset;          // Offset of x-axis title text, which is specified from a top center point.
  int x_title_height;          // Same as axis_title_font_height
  int x_title_above_margin;
  int x_tic_label_offset;
  int x_tic_label_height;      // Same as axis_tic_label_font_height
  int x_tic_above_margin;
  int graph_bottom_offset;

  // Centers
  int graph_center_x;
  int graph_center_y;
  


  int delayed_redraw_count;



  char error_text[200];  // if first character is non-null, error will be displayed instead of graph.
  
  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  //  Functions
  //
  /////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  float  graph_x_value_to_pixel(double value);
  float  graph_y_value_to_pixel(double value);
  int    graph_x_value_to_datapoint(double value);

  // force causes the nearest setting to be applied, if a setting is out of
  // bounds.  If force is false, an out-of-bounds setting won't be applied.
  // The return value is true if the setting is in bounds, false if out of bounds.
  bool set_xmin_xmax(double new_xmin,   double new_xmax,   bool force=true);
  bool set_ymin_ymax(double new_ymin=0, double new_ymax=0, bool force=true);
  
  void calculate_x_tic_labels();
  void calculate_y_tic_labels();

  virtual void   layout();
  virtual void   draw_dirty();
  virtual void   draw_dynamic();

  // Function for updating data indepently from drawing it
  virtual void   update_data();

  graph(graph_data_base*   gd,
	const char*        title,
	const char*        x_axis_title,
	const char*        y_axis_title,
	double             xmin,
	double             xmax,
	double             ymin,
	double             ymax);

  ~graph();

};


inline float graph::graph_x_value_to_pixel(double value)
{
  return (value-xmin) * (graphpanel->width-1) / (xmax-xmin);
}


inline float graph::graph_y_value_to_pixel(double value)
{
  return graphpanel->height-1 - (value-ymin) * (graphpanel->height-1) / (ymax-ymin);
}


inline int graph::graph_x_value_to_datapoint(double value)
{
  return int(floor( (value - gd->x_data_start) / gd->x_data_step));
}

#endif
