
// SPDX-License-Identifier: CC-BY-NC-ND-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial NoDerivatives 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-ND-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-ND-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  This work may be distributed in unmodified form,
// but derivatives of this work may not be distributed.  For further
// details, see the Creative Commons License "CC-BY-NC-ND-4.0".
//
// You should have received a copy of the CC-BY-NC-ND-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-nd/4.0/>.
//

#ifndef GRAPH_DATA_HH
#define GRAPH_DATA_HH

#include "graph.hh"
#include "base_data.hh"

inline int clog2(int size)
{
  if(size<=1)
    return 0;
  int r = 1;
  while(size>(1<<r))
    r++;
  return r;
}


//
// The types on this list also have forced instantiations at the end of graph_data.cc
//
enum graph_data_type
  {
    TYPE_SINT16 = 0,
    TYPE_DOUBLE = 1,
    TYPE_SINT32 = 2,
    TYPE_FLOAT  = 3,
  };

enum averaging_type
  {
   AVERAGING_NONE         = 0,
   AVERAGING_MINIMAL      = 1,
   AVERAGING_MODERATE     = 2,
   AVERAGING_LARGE        = 3,
   AVERAGING_HUGE         = 4,
   AVERAGING_IMMENSE      = 5,
   AVERAGING_PEAK_TRACK   = 6,
   AVERAGING_PEAK_HOLD    = 7,
  };


template <typename T> graph_data_type data_type_calc() { printf("Unknown Data type.\n"); abort(); }
template <> graph_data_type data_type_calc<int32_t>();
template <> graph_data_type data_type_calc<int16_t>();
template <> graph_data_type data_type_calc<float>();
template <> graph_data_type data_type_calc<double>();


//
// Generic class.  Subclasses actually either hold data or get it directly from the hardware.
//
template<typename T> class graph_data : public graph_data_base
{
public:
  base_data<T>**   y_data;
  
  graph_data_type  data_type;

  int no_trigger_count;

  
  // In base class
  // int       num_graphs;
  // int       num_points;  
  // double    x_data_start;
  // double    x_data_step;
  // double    y_scale;
  
  virtual void plot_data(class graph* g);
  virtual void find_ymin_ymax (double& ymin, double& ymax);

  virtual void update_data();   // Update data periodically independent of display rate (e.g. for averaging)
  virtual void finalize_data(); // Updata data prior to display (e.g. calculate dB values from average power)

  graph_data(graph_type source_type, int num_graphs, int num_points, double x_data_start, double x_data_step, double y_scale);
  virtual ~graph_data();
};


template <typename T> void graph_data<T>::update_data()
{
  for(int graph=0; graph<num_graphs; graph++)
    {
      y_data[graph]->update_data();
    }
}


template <typename T> void graph_data<T>::finalize_data()
{
  for(int graph=0; graph<num_graphs; graph++)
    {
      y_data[graph]->finalize_data();
    }
}


template <typename T> graph_data<T>::graph_data(graph_type source_type, int num_graphs, int num_points, double x_data_start, double x_data_step, double y_scale)
  : graph_data_base(source_type, num_graphs, num_points, x_data_start, x_data_step, y_scale)
{
  y_data = 0;
  data_type = data_type_calc<T>();
  no_trigger_count = 0;
}

template <typename T> graph_data<T>::~graph_data()
{
  if(this->y_data)
    {
      for(int g=0; g<this->num_graphs; g++)
	delete this->y_data[g];
      delete [] this->y_data;
    }
}


template <typename T> void graph_data<T>::find_ymin_ymax(double& ymin, double& ymax)
{
  double ymin_in_this_dataset = -1;  // Overridden below if there's any data
  double ymax_in_this_dataset =  1;
  bool first = true;
  
  if(y_data && num_graphs && num_points)
    {
      for(int graphitem=0; graphitem<num_graphs; graphitem++)
	{
	  double this_ymin = NAN;
	  double this_ymax = NAN;
	  y_data[graphitem]->find_ymin_ymax(this_ymin, this_ymax);

	  if(isfinite(this_ymin) && isfinite(this_ymax))
	    {
	      if(first)
		{
		  first = false;
		  ymin_in_this_dataset = this_ymin;
		  ymax_in_this_dataset = this_ymax;
		}
	      else
		{
		  ymin_in_this_dataset = (this_ymin<ymin_in_this_dataset) ? this_ymin : ymin_in_this_dataset;
		  ymax_in_this_dataset = (this_ymax>ymax_in_this_dataset) ? this_ymax : ymax_in_this_dataset;
		}
	    }
	}
    }

  ymin = ymin_in_this_dataset * y_scale;
  ymax = ymax_in_this_dataset * y_scale;
}



//
// Common instantiations of graph_data class, int16_t, int32_t, and double, are
// forced at the end of graph_data.cc
//


template <> void graph_data<int16_t>::plot_data(class graph* g);






#ifdef THIS_IS_OLD_CODE_NEWER_PLOTS_IN_DISPLAY_FOR_OPTIMIZATION_TO_DISPLAY_TYPE_USING_FUNCTION_ABOVE
template <typename T> void graph_data<T>::plot_data(class graph* g)
{
  subpanel* graphpanel = g->graphpanel;

  finalize_data();
  
  if(!y_data)
    return;

  if(source_type==TYPE_TIME)
    {
      if(!time_data_good)
	return;
      
      x_data_start = time_x_offset * x_data_step;
    }
  
  int firstx = g->graph_x_value_to_datapoint(g->xmin) - 1;
  int lastx  = g->graph_x_value_to_datapoint(g->xmax) + 1 + 8;   // 8 is because trigger variance can mean 8 samples of jitter in graph positioning

  if(firstx<0)
    firstx=0;
  if(lastx>=num_points)
    lastx=num_points-1;

  if(firstx>num_points-1 || lastx<0 || lastx<=firstx)
    return;

  
  int x_scale_to_pixel_num   = (graphpanel->width-1);
  int x_scale_to_pixel_denom = int((g->xmax - g->xmin)/x_data_step);
  int x_scale_to_pixel_add   = int((x_data_start - g->xmin)/x_data_step * (graphpanel->width-1));

  float y_scale_to_pixel_num       = -y_scale * (graphpanel->height-1);
  float y_scale_to_pixel_inv_denom = 1.0/(g->ymax - g->ymin);
  float y_scale_to_pixel_add       = (graphpanel->height-1)*(g->ymax - g->ymin) + g->ymin*(graphpanel->height-1);
  
  double ymin_in_dataset = 0;  // Zero guaranteed overridden later
  double ymax_in_dataset = 0;

  for(int graphitem=0; graphitem<num_graphs; graphitem++)
    {
      int16_t color = g->GRAPH_COLORS[graphitem];
      double  x0  = x_data_start + firstx * x_data_step;
      T       y0r = y_data[graphitem]->data[firstx];
      double  y0  = y0r * y_scale;
      float   x0i = g->graph_x_value_to_pixel(x0);
      float   y0i = g->graph_y_value_to_pixel(y0);
      if(graphitem==0 || y0r<ymin_in_dataset)
	ymin_in_dataset = y0r;
      if(graphitem==0 || y0r>ymax_in_dataset)
	ymax_in_dataset = y0r;
      for(int pnt=firstx+1; pnt<=lastx; pnt++)
	{
	  // Replace x translation with faster version
	  //double x1 = x_data_start + pnt * x_data_step;
	  //int x1i = g->graph_x_value_to_pixel(x1);
	  float x1i = ( pnt * x_scale_to_pixel_num + x_scale_to_pixel_add) / x_scale_to_pixel_denom;
	  
	  //double y1r = y_data[graphitem][pnt];
	  //int y1i = g->graph_y_value_to_pixel(y1r * y_scale);
	  T y1r = y_data[graphitem]->data[pnt];
	  float y1i = ( y1r * y_scale_to_pixel_num + y_scale_to_pixel_add) * y_scale_to_pixel_inv_denom;

	  //graphpanel->draw_line( point(x0i, y0i), point(x1i, y1i), color);

	  graphpanel->draw_line_known_background( pointf(x0i, y0i), pointf(x1i, y1i), color, g->PANEL_COLOR);

	  x0i = x1i;
	  y0i = y1i;

	  if(y1r<ymin_in_dataset)
	    ymin_in_dataset = y1r;
	  if(y1r>ymax_in_dataset)
	    ymax_in_dataset = y1r;
	}
    }

  g->ymin_in_last_dataset = ymin_in_dataset * y_scale;
  g->ymax_in_last_dataset = ymax_in_dataset * y_scale;
}
#endif


#endif
