
// SPDX-License-Identifier: CC-BY-NC-ND-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial NoDerivatives 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-ND-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-ND-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  This work may be distributed in unmodified form,
// but derivatives of this work may not be distributed.  For further
// details, see the Creative Commons License "CC-BY-NC-ND-4.0".
//
// You should have received a copy of the CC-BY-NC-ND-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-nd/4.0/>.
//

#ifndef GRAPH_DATA_BASE_HH
#define GRAPH_DATA_BASE_HH

//#define DEBUG_PLOTS


// Names that correspond to graph_type enums
extern const char* graph_type_names[];


//
// List of graph types.
//
// WARNING:  If this is changed, corresponding names must also be changed in graph_data_base.cc
//
#ifdef DEBUG_PLOTS
enum graph_type
  {
   TYPE_DEFAULT                     = 0,  // Used when we really don't want to use the graph_type feature
   TYPE_TIME                        = 0,
   TYPE_FREQ_Z1                     = 1,  // Nyquist Zone 1
   TYPE_Z1_TRANSFER_MAG             = 2,
   TYPE_Z1_TRANSFER_PHASE           = 3,
   TYPE_Z1_TRANSFER_GROUP_DELAY     = 4,
   TYPE_FREQ_Z2                     = 5,  // Nyquist Zone 2
   TYPE_Z2_TRANSFER_MAG             = 6,
   TYPE_Z2_TRANSFER_PHASE           = 7,
   TYPE_Z2_TRANSFER_GROUP_DELAY     = 8,
   TYPE_IMPULSE_RESPONSE            = 9,
   TYPE_REFLECTIONS                 = 10,

   TYPE_Z1_TRANSFER_PHASE_DEBUG     = 11,
   TYPE_Z2_TRANSFER_PHASE_DEBUG     = 12,
   TYPE_LAST                        = 13,
  };
#else
enum graph_type
  {
   TYPE_DEFAULT                     = 0,  // Used when we really don't want to use the graph_type feature
   TYPE_TIME                        = 0,
   TYPE_FREQ_Z1                     = 1,  // Nyquist Zone 1
   TYPE_Z1_TRANSFER_MAG             = 2,
   TYPE_Z1_TRANSFER_PHASE           = 3,
   TYPE_Z1_TRANSFER_GROUP_DELAY     = 4,
   TYPE_FREQ_Z2                     = 5,  // Nyquist Zone 2
   TYPE_Z2_TRANSFER_MAG             = 6,
   TYPE_Z2_TRANSFER_PHASE           = 7,
   TYPE_Z2_TRANSFER_GROUP_DELAY     = 8,
   TYPE_IMPULSE_RESPONSE            = 9,
   TYPE_REFLECTIONS                 = 10,

   TYPE_Z1_TRANSFER_PHASE_DEBUG     = -10,
   TYPE_Z2_TRANSFER_PHASE_DEBUG     = -9,
   TYPE_LAST                        = 11,
  };
#endif



//
// Generic class.  Subclasses actually either hold data or get it directly from the hardware.
//
class graph_data_base
{
public:

  graph_type       source_type;  // example: TYPE_TIME

  static bool      time_data_good;  // Only applies to graphs of type TYPE_TIME
  static float     time_x_offset;

  int              num_graphs;
  int              num_points;
  
  double           x_data_start;
  double           x_data_step;
  
  double           y_scale;

  
  virtual void update_data ()  {} // Function for updating data indepently from drawing it
  virtual void finalize_data() {} // Function for updating data just prior to drawing it

  virtual void plot_data      (class graph* g)             = 0;
  virtual void find_ymin_ymax (double& ymin, double& ymax) = 0;

  graph_data_base(graph_type source_type, int num_graphs, int num_points, double x_data_start, double x_data_step, double y_scale);
  virtual ~graph_data_base();
};



inline graph_data_base::graph_data_base(graph_type source_type, int num_graphs, int num_points, double x_data_start, double x_data_step, double y_scale)
{
  this->source_type   = source_type;
  this->num_graphs    = num_graphs;
  this->num_points    = num_points;
  this->x_data_start  = x_data_start;
  this->x_data_step   = x_data_step;
  this->y_scale       = y_scale;
}


inline graph_data_base::~graph_data_base()
{
}



#endif
