
// SPDX-License-Identifier: CC-BY-NC-ND-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial NoDerivatives 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-ND-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-ND-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  This work may be distributed in unmodified form,
// but derivatives of this work may not be distributed.  For further
// details, see the Creative Commons License "CC-BY-NC-ND-4.0".
//
// You should have received a copy of the CC-BY-NC-ND-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-nd/4.0/>.
//

#ifndef HOTSPOT_HH
#define HOTSPOT_HH

#include "window.hh"

//
// The hotspot is a transparent window, used by its parent window for implementing buttons
// that are drawn (or not drawn) by the parent.  For example, the parent may have some text,
// and cover that text with a hotspot for handling events of touching that text.
//
// Because the hotspot needs to manipulate its parent, it gets a pointer to that parent.
// The parent is some class derived from a multiwindow, but we don't know which class.
// So we 
//


extern bool draw_hotspots;
extern struct timeval unhandled_button_press_event_time;


template<class T> class hotspot : public window
{
public:
  
  typedef bool hotfunc(hotspot<T>* hotspot, my_event&me, void* data);

  T*        parent;
  hotfunc*  event_handling_function;
  void*     data;
  bool      draw_this_hotspot;
  
  virtual void   draw_static     (panel* p = 0);
  virtual void   draw_dynamic    ();
  virtual void   draw_dirty      ();

  virtual bool   handle_event(my_event& me);
  
  hotspot(T* parent, hotfunc* event_handling_function, void* data, bool draw_this_hotspot=true);
  virtual ~hotspot();
};


template<class T> hotspot<T>::hotspot(T* parent, hotfunc* event_handling_function, void* data, bool draw_this_hotspot)
  : window(BLACK)
{
  this->parent                   = parent;
  this->event_handling_function  = event_handling_function;
  this->data                     = data;
  this->draw_this_hotspot        = draw_this_hotspot;
  window_subject_to_layout = false;
}


template<class T> hotspot<T>::~hotspot()
{
}


template<class T> void hotspot<T>::draw_static(panel* p)
{
}


template<class T> void hotspot<T>::draw_dynamic()
{
  if(draw_hotspots && draw_this_hotspot)
    {
      point tl(       0,        0);
      point tr( width-1,        0);
      point bl(       0, height-1);
      point br( width-1, height-1);
      draw_line(tl, tr, GREEN);
      draw_line(tl, bl, GREEN);
      draw_line(tr, br, GREEN);
      draw_line(bl, br, GREEN);
    }
}


template<class T> void hotspot<T>::draw_dirty()
{
}


template<class T> bool hotspot<T>::handle_event(my_event& me)
{
  return event_handling_function(this, me, data);
}



#endif
