
// SPDX-License-Identifier: GPL-3.0-only
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, version 3.0 of the License.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <https://www.gnu.org/licenses/>.
//


#include <stdio.h>
#include <unistd.h>
#include "hwalloc.hh"

//
// From Zynq UltraScale+ Devices Register Reference, IOU_SLCR Module
//
#define IOU_SLCR_BASE_ADDRESS 0x00FF180000
#define NUM_MIO_REGISTERS     78

//
// From Zynq UltraScale+ Devices Register Reference, MIO_PIN_0
//
#define IOU_SLCR_L3_SEL_MASK          (7<<5)
#define IOU_SLCR_L3_SEL_GPIO          (0<<5)
#define IOU_SLCR_L3_SEL_CAN1          (1<<5)
#define IOU_SLCR_L3_SEL_I2C1          (2<<5)
#define IOU_SLCR_L3_SEL_PJTAG         (3<<5)
#define IOU_SLCR_L3_SEL_SPI0          (4<<5)
#define IOU_SLCR_L3_SEL_TTC3          (5<<5)
#define IOU_SLCR_L3_SEL_UART1         (6<<5)
#define IOU_SLCR_L3_SEL_TracePortClk  (7<<5)

#define IOU_SLCR_L2_SEL_MASK          (3<<3)
#define IOU_SLCR_L2_SEL_L3            (0<<3)
#define IOU_SLCR_L2_SEL_RESERVED_1    (1<<3)
#define IOU_SLCR_L2_SEL_Scan_Test     (2<<3)
#define IOU_SLCR_L2_SEL_RESERVED_3    (3<<3)

#define IOU_SLCR_L1_SEL_MASK          (1<<2)
#define IOU_SLCR_L1_SEL_L2            (0<<2)
#define IOU_SLCR_L1_SEL_RESERVED_1    (1<<2)

#define IOU_SLCR_L0_SEL_MASK          (1<<1)
#define IOU_SLCR_L0_SEL_L1            (0<<1)
#define IOU_SLCR_L0_SEL_QUAD_SPIO     (1<<1)

#define IOU_SLCR_RESERVED_MASK        (0xFFFFFF01)

#define IOU_SLCR_GPIO (IOU_SLCR_L3_SEL_GPIO | IOU_SLCR_L2_SEL_L3 | IOU_SLCR_L1_SEL_L2 | IOU_SLCR_L0_SEL_L1 )


//
//
//   GPIO
//
//

// Bank 0: 26-bit bank controlling MIO pins [0:25].
// Bank 1: 26-bit bank controlling MIO pins [26:51].
// Bank 2: 26-bit bank controlling MIO pins [52:77].
// Bank 3: 32-bit bank controlling EMIO signal sets [0:31].
// Bank 4: 32-bit bank controlling EMIO signal sets [32:63].
// Bank 5: 32-bit bank controlling EMIO signal sets [64:95].

#define GPIO_BASE_ADDRESS  0x00FF0A0000

//       Register Name	   Address	  Width	Type	Reset Value	Description
#define MASK_DATA_0_LSW	0x0000000000	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank0, MIO, Lower 16bits)
#define MASK_DATA_0_MSW	0x0000000004	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank0, MIO, Upper 10bits)
#define MASK_DATA_1_LSW	0x0000000008	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank1, MIO, Lower 16bits)
#define MASK_DATA_1_MSW	0x000000000C	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank1, MIO, Upper 10 bits)
#define MASK_DATA_2_LSW	0x0000000010	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank2, MIO, Lower 16 bits)
#define MASK_DATA_2_MSW	0x0000000014	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank2, MIO, Upper 10 bits)
#define MASK_DATA_3_LSW	0x0000000018	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank3, EMIO, Lower 16bits)
#define MASK_DATA_3_MSW	0x000000001C	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank3, EMIO, Upper 16bits)
#define MASK_DATA_4_LSW	0x0000000020	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank4, EMIO, Lower 16bits)
#define MASK_DATA_4_MSW	0x0000000024	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank4, EMIO, Upper 16bits)
#define MASK_DATA_5_LSW	0x0000000028	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank5, EMIO, Lower 16bits)
#define MASK_DATA_5_MSW	0x000000002C	//  32	mixed	0x00000000	Maskable Output Data (GPIO Bank5, EMIO, Upper 16bits)

#define DATA_0  	0x0000000040	//  32	mixed	0x00000000	Output Data (GPIO Bank0, MIO)
#define DATA_1  	0x0000000044	//  32	mixed	0x00000000	Output Data (GPIO Bank1, MIO)
#define DATA_2  	0x0000000048	//  32	mixed	0x00000000	Output Data (GPIO Bank2, MIO)
#define DATA_3  	0x000000004C	//  32	rw	0x00000000	Output Data (GPIO Bank3, EMIO)
#define DATA_4  	0x0000000050	//  32	rw	0x00000000	Output Data (GPIO Bank4, EMIO)
#define DATA_5  	0x0000000054	//  32	rw	0x00000000	Output Data (GPIO Bank5, EMIO)

#define DATA_0_RO	0x0000000060	//  32	mixed	0x00000000	Input Data (GPIO Bank0, MIO)
#define DATA_1_RO	0x0000000064	//  32	mixed	0x00000000	Input Data (GPIO Bank1, MIO)
#define DATA_2_RO	0x0000000068	//  32	mixed	0x00000000	Input Data (GPIO Bank2, MIO)
#define DATA_3_RO	0x000000006C	//  32	ro	0x00000000	Input Data (GPIO Bank3, EMIO)
#define DATA_4_RO	0x0000000070	//  32	ro	0x00000000	Input Data (GPIO Bank4, EMIO)
#define DATA_5_RO	0x0000000074	//  32	ro	0x00000000	Input Data (GPIO Bank5, EMIO)

#define DIRM_0  	0x0000000204	//  32	mixed	0x00000000	Direction mode (GPIO Bank0, MIO)
#define OEN_0   	0x0000000208	//  32	mixed	0x00000000	Output enable (GPIO Bank0, MIO)
#define INT_MASK_0	0x000000020C	//  32	mixed	0x03FFFFFF	Interrupt Mask Status (GPIO Bank0, MIO)
#define INT_EN_0	0x0000000210	//  32	mixed	0x00000000	Interrupt Enable/Unmask (GPIO Bank0, MIO)
#define INT_DIS_0	0x0000000214	//  32	mixed	0x00000000	Interrupt Disable/Mask (GPIO Bank0, MIO)
#define INT_STAT_0	0x0000000218	//  32	mixed	0x00000000	Interrupt Status (GPIO Bank0, MIO)
#define INT_TYPE_0	0x000000021C	//  32	mixed	0x03FFFFFF	Interrupt Type (GPIO Bank0, MIO)
#define INT_POLARITY_0	0x0000000220	//  32	mixed	0x00000000	Interrupt Polarity (GPIO Bank0, MIO)
#define INT_ANY_0	0x0000000224	//  32	mixed	0x00000000	Interrupt Any Edge Sensitive (GPIO Bank0, MIO)

#define DIRM_1   	0x0000000244	//  32	mixed	0x00000000	Direction mode (GPIO Bank1, MIO)
#define OEN_1   	0x0000000248	//  32	mixed	0x00000000	Output enable (GPIO Bank1, MIO)
#define INT_MASK_1	0x000000024C	//  32	mixed	0x03FFFFFF	Interrupt Mask Status (GPIO Bank1, MIO)
#define INT_EN_1	0x0000000250	//  32	mixed	0x00000000	Interrupt Enable/Unmask (GPIO Bank1, MIO)
#define INT_DIS_1	0x0000000254	//  32	mixed	0x00000000	Interrupt Disable/Mask (GPIO Bank1, MIO)
#define INT_STAT_1	0x0000000258	//  32	mixed	0x00000000	Interrupt Status (GPIO Bank1, MIO)
#define INT_TYPE_1	0x000000025C	//  32	mixed	0x03FFFFFF	Interrupt Type (GPIO Bank1, MIO)
#define INT_POLARITY_1	0x0000000260	//  32	mixed	0x00000000	Interrupt Polarity (GPIO Bank1, MIO)
#define INT_ANY_1	0x0000000264	//  32	mixed	0x00000000	Interrupt Any Edge Sensitive (GPIO Bank1, MIO)

#define DIRM_2   	0x0000000284	//  32	mixed	0x00000000	Direction mode (GPIO Bank2, MIO)
#define OEN_2   	0x0000000288	//  32	mixed	0x00000000	Output enable (GPIO Bank2, MIO)
#define INT_MASK_2	0x000000028C	//  32	mixed	0x03FFFFFF	Interrupt Mask Status (GPIO Bank2, MIO)
#define INT_EN_2	0x0000000290	//  32	mixed	0x00000000	Interrupt Enable/Unmask (GPIO Bank2, MIO)
#define INT_DIS_2	0x0000000294	//  32	mixed	0x00000000	Interrupt Disable/Mask (GPIO Bank2, MIO)
#define INT_STAT_2	0x0000000298	//  32	mixed	0x00000000	Interrupt Status (GPIO Bank2, MIO)
#define INT_TYPE_2	0x000000029C	//  32	mixed	0x03FFFFFF	Interrupt Type (GPIO Bank2, MIO)
#define INT_POLARITY_2	0x00000002A0	//  32	mixed	0x00000000	Interrupt Polarity (GPIO Bank2, MIO)
#define INT_ANY_2	0x00000002A4	//  32	mixed	0x00000000	Interrupt Any Edge Sensitive (GPIO Bank2, MIO)

#define DIRM_3   	0x00000002C4	//  32	rw	0x00000000	Direction mode (GPIO Bank3, EMIO Bank0)
#define OEN_3   	0x00000002C8	//  32	rw	0x00000000	Output enable (GPIO Bank3, EMIO Bank0)
#define INT_MASK_3	0x00000002CC	//  32	ro	0xFFFFFFFF	Interrupt Mask Status (GPIO Bank3, EMIO Bank0)
#define INT_EN_3	0x00000002D0	//  32	wo	0x00000000	Interrupt Enable/Unmask (GPIO Bank3, EMIO Bank0)
#define INT_DIS_3	0x00000002D4	//  32	wo	0x00000000	Interrupt Disable/Mask (GPIO Bank3, EMIO Bank0)
#define INT_STAT_3	0x00000002D8	//  32	wtc	0x00000000	Interrupt Status (GPIO Bank3, EMIO Bank0)
#define INT_TYPE_3	0x00000002DC	//  32	rw	0xFFFFFFFF	Interrupt Type (GPIO Bank3, EMIO Bank0)
#define INT_POLARITY_3	0x00000002E0	//  32	rw	0x00000000	Interrupt Polarity (GPIO Bank3, EMIO Bank0)
#define INT_ANY_3	0x00000002E4	//  32	rw	0x00000000	Interrupt Any Edge Sensitive (GPIO Bank3, EMIO Bank0)

#define DIRM_4   	0x0000000304	//  32	rw	0x00000000	Direction mode (GPIO Bank4, EMIO Bank1)
#define OEN_4   	0x0000000308	//  32	rw	0x00000000	Output enable (GPIO Bank4, EMIO Bank1)
#define INT_MASK_4	0x000000030C	//  32	ro	0xFFFFFFFF	Interrupt Mask Status (GPIO Bank4, EMIO Bank1)
#define INT_EN_4	0x0000000310	//  32	wo	0x00000000	Interrupt Enable/Unmask (GPIO Bank4, EMIO Bank1)
#define INT_DIS_4	0x0000000314	//  32	wo	0x00000000	Interrupt Disable/Mask (GPIO Bank4, EMIO Bank1)
#define INT_STAT_4	0x0000000318	//  32	wtc	0x00000000	Interrupt Status (GPIO Bank4, EMIO Bank1)
#define INT_TYPE_4	0x000000031C	//  32	rw	0xFFFFFFFF	Interrupt Type (GPIO Bank4, EMIO Bank1)
#define INT_POLARITY_4	0x0000000320	//  32	rw	0x00000000	Interrupt Polarity (GPIO Bank4, EMIO Bank1)
#define INT_ANY_4	0x0000000324	//  32	rw	0x00000000	Interrupt Any Edge Sensitive (GPIO Bank4, EMIO Bank1)

#define DIRM_5   	0x0000000344	//  32	rw	0x00000000	Direction mode (GPIO Bank5, EMIO Bank2)
#define OEN_5   	0x0000000348	//  32	rw	0x00000000	Output enable (GPIO Bank5, EMIO Bank2)
#define INT_MASK_5	0x000000034C	//  32	ro	0xFFFFFFFF	Interrupt Mask Status (GPIO Bank5, EMIO Bank2)
#define INT_EN_5	0x0000000350	//  32	wo	0x00000000	Interrupt Enable/Unmask (GPIO Bank5, EMIO Bank2)
#define INT_DIS_5	0x0000000354	//  32	wo	0x00000000	Interrupt Disable/Mask (GPIO Bank5, EMIO Bank2)
#define INT_STAT_5	0x0000000358	//  32	wtc	0x00000000	Interrupt Status (GPIO Bank5, EMIO Bank2)
#define INT_TYPE_5	0x000000035C	//  32	rw	0xFFFFFFFF	Interrupt Type (GPIO Bank5, EMIO Bank2)
#define INT_POLARITY_5	0x0000000360	//  32	rw	0x00000000	Interrupt Polarity (GPIO Bank5, EMIO Bank2)
#define INT_ANY_5	0x0000000364	//  32	rw	0x00000000	Interrupt Any Edge Sensitive (GPIO Bank5, EMIO Bank2)

#define NUM_GPIO_REGISTERS ((INT_ANY_5 + 4) / 4)


void clear_bit(volatile uint32_t& reg, int bit)
{
  reg = reg & ~((uint32_t)(1<<bit));
}

void set_bit(volatile uint32_t& reg, int bit)
{
  reg = reg | ((uint32_t)(1<<bit));
}

int get_bit(volatile uint32_t& reg, int bit)
{
  return (reg>>bit) & 1;
}

int get_MIO_value(volatile uint32_t* gpio,  int mio_number)
{
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;

  uint32_t data_reg = ( (bank_number==0) ? DATA_0_RO :
			(bank_number==1) ? DATA_1_RO :
			/**/               DATA_2_RO );
	  
  return get_bit(gpio[data_reg/4], bit_number);
}

void set_MIO_value(volatile uint32_t* gpio,  int mio_number, int value)
{
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;

  uint32_t data_reg = ( (bank_number==0) ? DATA_0 :
			(bank_number==1) ? DATA_1 :
			/**/               DATA_2 );

  if(value)
    set_bit(gpio[data_reg/4], bit_number);
 else
   clear_bit(gpio[data_reg/4], bit_number);
}


void print_MIO_selections()
{
  volatile uint32_t* iou = hwalloc<uint32_t>(IOU_SLCR_BASE_ADDRESS, NUM_MIO_REGISTERS);
  volatile uint32_t* gpio = hwalloc<uint32_t>(GPIO_BASE_ADDRESS, NUM_GPIO_REGISTERS);

  printf("\nMIO Register routing:\n");
  fflush(stdout);

  for(int mio_number=0; mio_number<NUM_MIO_REGISTERS; mio_number++)
    {
      uint32_t iou_value = iou[mio_number];
      bool GPIO = false;
      
      const char* iou_string = "UNKNOWN ERROR";
      if(iou_value & IOU_SLCR_RESERVED_MASK)
	iou_string = "RESERVED BITS BAD";
      else if( (iou_value&IOU_SLCR_L0_SEL_MASK) == IOU_SLCR_L0_SEL_QUAD_SPIO )
	iou_string = "QUAD_SPIO";
      else if( (iou_value&IOU_SLCR_L1_SEL_MASK) == IOU_SLCR_L1_SEL_RESERVED_1 )
	iou_string = "L1 Reserved 1";
      else if( (iou_value&IOU_SLCR_L2_SEL_MASK) == IOU_SLCR_L2_SEL_RESERVED_3 )
	iou_string = "L2 Reserved 3";
      else if( (iou_value&IOU_SLCR_L2_SEL_MASK) == IOU_SLCR_L2_SEL_Scan_Test )
	iou_string = "Scan Test";
      else if( (iou_value&IOU_SLCR_L2_SEL_MASK) == IOU_SLCR_L2_SEL_RESERVED_1 )
	iou_string = "L2 Reserved 1";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_TracePortClk )
	iou_string = "Trace Port Clk";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_UART1 )
	iou_string = "UART1";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_TTC3 )
	iou_string = "TTC3";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_SPI0 )
	iou_string = "SPI0";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_PJTAG )
	iou_string = "PJTAG";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_I2C1 )
	iou_string = "I2C1";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_CAN1 )
	iou_string = "CAN1";
      else if( (iou_value&IOU_SLCR_L3_SEL_MASK) == IOU_SLCR_L3_SEL_GPIO )
	{
	  GPIO = true;
	  iou_string = "GPIO";      
	}
      
      printf("  MIO_%d%s: %20s (0x%08x)", mio_number, (mio_number<10)?" ":"", iou_string, iou_value);

      if(GPIO)
	{
	  int bank_number = mio_number/26;
	  int bit_number = mio_number%26;

	  if(bank_number>=3 || bank_number<0)
	    {
	      printf("ERROR: Not a valid MIO %d in set_MIO_to_GPIO_input()\n", mio_number);
	      exit(20);
	    }

	  uint32_t output_direction_reg = ( (bank_number==0) ? DIRM_0 :
					    (bank_number==1) ? DIRM_1 :
					    /**/               DIRM_2 );
	  
	  uint32_t output_enable_reg = ( (bank_number==0) ? OEN_0 :
					 (bank_number==1) ? OEN_1 :
					 /**/               OEN_2 );
	  
	  uint32_t interrupt_mask_reg = ( (bank_number==0) ? INT_MASK_0 :
					  (bank_number==1) ? INT_MASK_1 :
					  /**/               INT_MASK_2 );

	  printf("  Value=%d, ", get_MIO_value(gpio, mio_number));
	  
	  if(get_bit(gpio[output_direction_reg/4], bit_number))
	    {
	      printf(" OUTPUT, ");
	      if(get_bit(gpio[output_enable_reg/4], bit_number))
		printf("ENABLED ");
	      else
		printf("DISABLED ");
	    }
	  else
	    {
	      printf(" INPUT, ");
	     if(get_bit(gpio[interrupt_mask_reg/4], bit_number))
		printf("INTERRUPTS DISABLED ");
	      else
		printf("INTERRUPTS ENABLED ");
	    }
	}
      
      printf("\n");      
      fflush(stdout);
    }

  hwfree(iou, NUM_MIO_REGISTERS);
  hwfree(gpio, NUM_GPIO_REGISTERS);  
}

void set_MIO_to_GPIO_input(int mio_number)
{
  // Set the GPIO to an input
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;

  if(bank_number>=3 || bank_number<0)
    {
      printf("ERROR: Not a valid MIO %d in set_MIO_to_GPIO_input()\n", mio_number);
      exit(20);
    }

  uint32_t output_direction_reg = ( (bank_number==0) ? DIRM_0 :
				    (bank_number==1) ? DIRM_1 :
				    /**/               DIRM_2 );
  
  uint32_t output_enable_reg = ( (bank_number==0) ? OEN_0 :
				 (bank_number==1) ? OEN_1 :
				 /**/               OEN_2 );

  uint32_t interrupt_disable_reg = ( (bank_number==0) ? INT_DIS_0 :
				     (bank_number==1) ? INT_DIS_1 :
				     /**/               INT_DIS_2 );
  
  volatile uint32_t* gpio = hwalloc<uint32_t>(GPIO_BASE_ADDRESS, NUM_GPIO_REGISTERS);
  clear_bit(gpio[output_direction_reg/4], bit_number);  // This pin is not selected as an output.
  clear_bit(gpio[output_enable_reg/4], bit_number);     // This output is not enabled.
  gpio[interrupt_disable_reg/4] = (1<<bit_number);
  hwfree(gpio, NUM_GPIO_REGISTERS);  

  
  // Set the pin to use the GPIO
  volatile uint32_t* iou = hwalloc<uint32_t>(IOU_SLCR_BASE_ADDRESS, NUM_MIO_REGISTERS);
  iou[mio_number] = IOU_SLCR_GPIO;
  hwfree(iou, NUM_MIO_REGISTERS);  
}


void set_MIO_to_GPIO_output(int mio_number)
{
  // Set the GPIO to an input
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;

  if(bank_number>=3 || bank_number<0)
    {
      printf("ERROR: Not a valid MIO %d in set_MIO_to_GPIO_input()\n", mio_number);
      exit(20);
    }

  uint32_t output_direction_reg = ( (bank_number==0) ? DIRM_0 :
				    (bank_number==1) ? DIRM_1 :
				    /**/               DIRM_2 );
  
  uint32_t output_enable_reg = ( (bank_number==0) ? OEN_0 :
				 (bank_number==1) ? OEN_1 :
				 /**/               OEN_2 );

  uint32_t interrupt_disable_reg = ( (bank_number==0) ? INT_DIS_0 :
				     (bank_number==1) ? INT_DIS_1 :
				     /**/               INT_DIS_2 );
  
  volatile uint32_t* gpio = hwalloc<uint32_t>(GPIO_BASE_ADDRESS, NUM_GPIO_REGISTERS);
  set_bit(gpio[output_direction_reg/4], bit_number);  // This pin is selected as an output.
  set_bit(gpio[output_enable_reg/4], bit_number);     // This output is enabled.
  gpio[interrupt_disable_reg/4] = (1<<bit_number);
  hwfree(gpio, NUM_GPIO_REGISTERS);  

  
  // Set the pin to use the GPIO
  volatile uint32_t* iou = hwalloc<uint32_t>(IOU_SLCR_BASE_ADDRESS, NUM_MIO_REGISTERS);
  iou[mio_number] = IOU_SLCR_GPIO;
  hwfree(iou, NUM_MIO_REGISTERS);  

}


int main(void)
{
  volatile uint32_t* iou = hwalloc<uint32_t>(IOU_SLCR_BASE_ADDRESS, NUM_MIO_REGISTERS);
  volatile uint32_t* gpio = hwalloc<uint32_t>(GPIO_BASE_ADDRESS, NUM_GPIO_REGISTERS);

  print_MIO_selections();
  
  printf("Changing MIO 21 to SPI0\n");

  uint32_t iou_value = iou[21];

  iou[21] = IOU_SLCR_L3_SEL_I2C1 | IOU_SLCR_L2_SEL_L3 | IOU_SLCR_L1_SEL_L2 | IOU_SLCR_L0_SEL_L1;

  print_MIO_selections();

  printf("Changing MIO 21 to back\n");

  iou[21] = iou_value;

  
  print_MIO_selections();


  // On the ZCU111, MIO 23 is the LED, MIO 22 is the button.

  //iou[22] = IOU_SLCR_GPIO;
  //iou[23] = IOU_SLCR_GPIO;

  set_MIO_to_GPIO_input(22);
  set_MIO_to_GPIO_output(23);

  int counts_this_pass = 0;
  int last_button_value = 0;
  
  for(;;)
    {
      counts_this_pass = (counts_this_pass+1)%3;

      for(int count=0; count<counts_this_pass+1; count++)
	{
	  // Issue:  The OS also writes to this MIO.  So we need to write more.
	  for(int i=0; i<10000; i++)
	    {
	      set_MIO_value(gpio, 23, 1);
	      usleep(5);
	    }
	  for(int i=0; i<10000; i++)
	    {
	      set_MIO_value(gpio, 23, 0);
	      usleep(5);
	    }
	}
      
      int v = get_MIO_value(gpio, 22);

      if(v!=last_button_value)
	{
	  printf("Button value changed from %d to %d\n", last_button_value, v);
	  last_button_value=v;
	}

      for(int i=0; i<50000; i++)
	{
	  set_MIO_value(gpio, 23, 0);
	  usleep(5);
	}
    }


  hwfree(iou, NUM_MIO_REGISTERS);  
  hwfree(gpio, NUM_GPIO_REGISTERS);  
}
