

// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#ifndef PHYSICAL_MEMORY_H
#define PHYSICAL_MEMORY_H

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

template <typename T> class physical_memory
{
  volatile T*  memptr;
  uint32_t            byte_address;
  uint32_t            length;

 public:

  physical_memory(uint32_t byte_address, uint32_t length=0);
  ~physical_memory();

  bool address_valid        (uint32_t address);
  void address_check_abort  (uint32_t address);
  bool offset_valid         (uint32_t offset);
  void offset_check_abort   (uint32_t offset);

  T     read_address  (uint32_t address);
  void  write_address (uint32_t address, T value);

  T     read_offset   (uint32_t offset);
  void   write_offset (uint32_t offset, T value);
};


template<typename T> physical_memory<T>::physical_memory(uint32_t byte_address, uint32_t length)
{
  if(length==0)
    length = getpagesize()/sizeof(T);

  this->byte_address   = byte_address;
  this->length         = length;

  if((byte_address&3)!=0)
    {
      printf("ERROR: byte address 0x%08x is misaligned.\n", byte_address);
      abort();
    }
  
  int fd = open("/dev/mem",O_RDWR|O_SYNC);
  if(fd<0)
    {
      printf("ERROR: Can't open /dev/mem\n");
      exit(20);
    }
  
  memptr = (T*) mmap(0, length * sizeof(T), PROT_READ|PROT_WRITE, MAP_SHARED, fd, byte_address);
  if(!memptr)
    {
      printf("ERROR: Can't map memory to access physical addresses\n");
      exit(20);
    }

  close(fd);
}

template<typename T> physical_memory<T>::~physical_memory()
{
  munmap((void*)memptr, length * sizeof(T));
}


template<typename T> bool physical_memory<T>::address_valid(uint32_t address)
{
  return (address&3)==0 && offset_valid( (address-byte_address)/sizeof(T) );
}

template<typename T> void physical_memory<T>::address_check_abort(uint32_t address)
{
  if(!address_valid(address))
    {
      if((address&3)!=0)
	printf("ERROR: physical memory access to address 0x%08X is misaligned.\n", address);
      else
	printf("ERROR: physical memory access to address 0x%08X out of range 0x%08X to 0x%08lx\n", address, byte_address, byte_address+sizeof(T)*length);
      abort();
    }
}

template<typename T> bool physical_memory<T>::offset_valid(uint32_t offset)
{
  return (offset>=0 && offset<=length);
}

template<typename T> void physical_memory<T>::offset_check_abort(uint32_t offset)
{
  if(!offset_valid(offset))
    {
      uint32_t address = byte_address + sizeof(T) * offset;
      printf("ERROR: physical memory access to offset %d (address 0x%08X) is out of range 0 to %d (addresses 0x%08X to 0x%08lx).\n", offset, address, length, byte_address, byte_address+sizeof(T)*length);
      abort();
    }
}

template<typename T> T physical_memory<T>::read_offset(uint32_t offset)
{
  offset_check_abort(offset);
  T data = memptr[offset];
  return data;
}

template<typename T> void physical_memory<T>::write_offset(uint32_t offset, T data)
{
  offset_check_abort(offset);
  memptr[offset] = data;
}

template<typename T> T physical_memory<T>::read_address(uint32_t address)
{
  address_check_abort(address);
  uint32_t offset = address - byte_address;
  T data   = memptr[offset];
  return data;
}

template<typename T> void physical_memory<T>::write_address(uint32_t address, T data)
{
  address_check_abort(address);
  uint32_t offset = address - byte_address;
  memptr[offset] = data;
}


#endif
