
// SPDX-License-Identifier: GPL-3.0-only
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, version 3.0 of the License.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <https://www.gnu.org/licenses/>.
//

#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

#include "hwalloc.hh"


class rfdc
{
public:
  bool turn_off_when_deleted;
  
  uint32_t rfdc_base_address;

  const static int NUM_TILES = 8;
  volatile struct rfdc_common_registers*  common_registers;  
  volatile struct rfdc_tile_registers*    tile_registers[NUM_TILES];

  bool all_running(bool verbose);
  bool none_running(bool verbose);
  void num_running(bool verbose, int& num_run, int& num_present);

  void all_on(bool verbose);
  void all_off(bool verbose);
  
  rfdc(bool verbose, bool turn_off_when_deleted = false, uint32_t base_address = 0);
  ~rfdc();
};


// BASE + 0x0000
struct rfdc_common_registers
{
  /* 0x0000 */ uint32_t version_information;    // 31:24=Major, 23:16=Minor, 15:8=Revision, 7:0=Reserved
  /* 0x0004 */ uint32_t master_reset_register;  // bit 0 is reset, write a 1 to perform reset.  Other bits all 0

  /* 0x0008 */ uint8_t  reserved_0008[0x0100-0x008];

  /* 0x0100 */ uint32_t common_interrupt_status_register;  // clear on read
  /* 0x0104 */ uint32_t common_interrupt_enable_register;
};


//
// Base + 0x04000: RF-DAC Tile 0
// Base + 0x08000: RF-DAC Tile 1
// Base + 0x0C000: RF-DAC Tile 2
// Base + 0x10000: RF-DAC Tile 3
// Base + 0x14000: RF-ADC Tile 0
// Base + 0x18000: RF-ADC Tile 1
// Base + 0x1C000: RF-ADC Tile 2
// Base + 0x20000: RF-ADC Tile 3
//
// Converter states:
//     0-2=Device power up
//     3-5=Power supply adjustment
//    6-10=Clock configuration
//   11-13=(ADC only) Converter calibration
//      14=Waiting for AXI4-Stream reset to be released
//      15=Done with power-up sequence
//
struct rfdc_tile_registers
{
  /* 0x0000 */ uint32_t reserved_0000;
  /* 0x0004 */ uint32_t restart_power_on_state_machine_register;  // Write 1 to change state.  Read 0 before anything else is written
  /* 0x0008 */ uint32_t restart_state_register; // Desired state: 0xF to enable converters, 0x3 to disable.  Set then command state change.
  /* 0x000C */ uint32_t current_state_register; // Current state, described in table above
  /* 0x0010 */ uint32_t reserved_0010;
  /* 0x0014 */ uint32_t reserved_0014;
  /* 0x0018 */ uint32_t reserved_0018;
  /* 0x001C */ uint32_t reserved_001C;
  /* 0x0020 */ uint32_t reserved_0020;
  /* 0x0024 */ uint32_t reserved_0024;
  /* 0x0028 */ uint32_t reserved_0028;
  /* 0x002C */ uint32_t reserved_002C;
  /* 0x0030 */ uint32_t reserved_0030;
  /* 0x0034 */ uint32_t reserved_0034;
  /* 0x0038 */ uint32_t reset_count_register;  // Count of resets due to errors like PLL lock.  Reset clears.  Saturates at 255.
  /* 0x003C */ uint32_t reserved_003C;
  
  /* 0x0040 */ uint8_t  reserved_0040[0x0100-0x0040];

  /* 0x0100 */ uint32_t post_implementation_simulation_speedup_register; // never use on hardware.

  /* 0x0104 */ uint8_t  reserved_0104[0x0200-0x0104];

  /* 0x0200 */ uint32_t interrupt_status_register;
  /* 0x0204 */ uint32_t interrupt_enable_register;
  /* 0x0208 */ uint32_t converter_0_interrupt_register;
  /* 0x020C */ uint32_t converter_0_interrupt_enable_register;
  /* 0x0210 */ uint32_t converter_1_interrupt_register;
  /* 0x0214 */ uint32_t converter_1_interrupt_enable_register;
  /* 0x0218 */ uint32_t converter_2_interrupt_register;
  /* 0x021C */ uint32_t converter_2_interrupt_enable_register;
  /* 0x0220 */ uint32_t converter_3_interrupt_register;
  /* 0x0224 */ uint32_t converter_3_interrupt_enable_register;
  /* 0x0228 */ uint32_t tile_common_status_register; // bits 3=pll locked, 2=operating, 1=voltage_supplies_stable, 0=clock present
  /* 0x022C */ uint32_t reserved_022C;
  /* 0x0230 */ uint32_t tile_fifo_disable_register;  // 1 disables fifo, 0 enables
};










inline rfdc::rfdc(bool verbose, bool turn_off_when_deleted, uint32_t base_address)
{
  this->turn_off_when_deleted = turn_off_when_deleted;


  if(base_address)
    {
      rfdc_base_address = base_address;
    }
  else
    {
      ///////////////////////////////////////////////////////////////////////////
      // Find the base address of the rfdc
      ///////////////////////////////////////////////////////////////////////////

      rfdc_base_address = 0;
      const char* devdir = "/sys/bus/platform/devices/";
      DIR* dp = opendir(devdir);
      if(!dp)
	{
	  printf("ERROR:  Can't search directory \"%s\".\n", devdir);
	  exit(20);
	}
      for(;;)
	{
	  dirent *de = readdir(dp);
	  
	  if(!de)
	    break;
	  
	  if(!strcmp(".usp_rf_data_converter", &de->d_name[8]))
	    {
	      sscanf(de->d_name, "%x", &rfdc_base_address);
	      break;
	    }
	}
      closedir(dp);
      
      if(rfdc_base_address==0)
	{
	  printf("ERROR:  Can't find RFDC device \"*.usp_rf_data_converter\" in directory \"%s\".\n", devdir);
	  exit(20);      
	}
    }

  
  if(verbose)
    printf("Found RFDC device at base address 0x%08X\n", rfdc_base_address);

  ///////////////////////////////////////////////////////////////////////////
  // Map the common registers, print the version info
  ///////////////////////////////////////////////////////////////////////////

  common_registers = hwalloc<rfdc_common_registers>(rfdc_base_address);

  ///////////////////////////////////////////////////////////////////////////
  // Map the tile registers
  ///////////////////////////////////////////////////////////////////////////
  
  for(int i=0; i<NUM_TILES; i++)
    tile_registers[i] = hwalloc<rfdc_tile_registers>(rfdc_base_address + 0x4000 * (i+1));
  
  ///////////////////////////////////////////////////////////////////////////
  // Reset the RFDC
  ///////////////////////////////////////////////////////////////////////////
  
  common_registers->common_interrupt_enable_register = 0;
  common_registers->master_reset_register = 1;

  ///////////////////////////////////////////////////////////////////////////
  // Wait for all RFDC tiles to complete initialization
  ///////////////////////////////////////////////////////////////////////////

  int wait_count = 30; // 3 seconds max wait

  for(int i=0; i<NUM_TILES; i++)
    {      
      while(tile_registers[i]->restart_power_on_state_machine_register)
	{
	  if(!wait_count)
	    break;
	  wait_count--;
	  usleep(100000);
	}
    }

  // Disable interrupts one more time, for extra safety.  It's not clear that the first
  // write to this register took effect, since the core possibly wasn't reset.
  common_registers->common_interrupt_enable_register = 0;

  ///////////////////////////////////////////////////////////////////////////
  // Report status
  ///////////////////////////////////////////////////////////////////////////

  if(!all_running(verbose))
    {
      printf("ERROR: not all RFDC tiles are running after initialization.\n");
      //  exit(20);
    }

  printf("Turning on the ADCs and DACs.\n");
  all_on(verbose);
}


inline bool rfdc::all_running(bool verbose)
{
  int num_run;
  int num_present;
  num_running(verbose, num_run, num_present);
  if(verbose)
    printf("ADC and DAC tile check:  %d total present, %d running.\n", num_present, num_run);
  return (num_run==num_present);
}

inline bool rfdc::none_running(bool verbose)
{
  int num_run;
  int num_present;
  num_running(verbose, num_run, num_present);
  return (num_run==0);
}


inline void rfdc::num_running(bool verbose, int& num_run, int& num_present)
{
  if(verbose)
    {
      printf("RFDC Device at base address 0x%08X has Version=%d.%d.%d\n",
	     rfdc_base_address,
	     (common_registers->version_information >> 24 ) & 0xFF,
	     (common_registers->version_information >> 16 ) & 0xFF,
	     (common_registers->version_information >>  8 ) & 0xFF);
    }

  num_present = 0;
  num_run = 0;

  for(int i=0; i<NUM_TILES; i++)
    {
      int run_state = tile_registers[i]->current_state_register;
      int status_bits = tile_registers[i]->tile_common_status_register;
      if(run_state==0)
	{
	  if(verbose)
	    printf("Tile %d isn't physically present.\n", i);
	}
      else
	{
	  num_present++;
	  if(verbose)
	    {
	      printf("Tile %d State is %s (%d), %s, %s, %s, %s\n",
		     i, 
		     ( (run_state ==  0) ? "Device Power-up"                  :
		       (run_state ==  1) ? "Device Power-up"                  :
		       (run_state ==  2) ? "Device Power-up"                  :
		       (run_state ==  3) ? "Power Supply Adjustment"          :
		       (run_state ==  4) ? "Power Supply Adjustment"          :
		       (run_state ==  5) ? "Power Supply Adjustment"          :
		       (run_state ==  6) ? "Clock Configuration"              :
		       (run_state ==  7) ? "Clock Configuration"              :
		       (run_state ==  8) ? "Clock Configuration"              :
		       (run_state ==  9) ? "Clock Configuration"              :
		       (run_state == 10) ? "Clock Configuration"              :
		       (run_state == 11) ? "ADC Calibrationtion"              :
		       (run_state == 12) ? "ADC Calibrationtion"              :
		       (run_state == 13) ? "ADC Calibrationtion"              :
		       (run_state == 14) ? "Wait for AXI4S Reset Release"     :
		       (run_state == 15) ? "Normal Operation"                 :
		       /**/                                         "Unknown and bad state"            ),
		     run_state,
		     (status_bits & 0x08) ? "PLL locked"     : "NO PLL LOCK",
		     (status_bits & 0x04) ? "Operating"      : "NOT OPERATING",
		     (status_bits & 0x02) ? "Voltage Stable" : "VOLTAGE UNSTABLE",
		     (status_bits & 0x01) ? "Clock Present"  : "NO CLOCK");
	    }
	}

      if(run_state==15)
	num_run++;
    }
}


inline rfdc::~rfdc()
{
  if(turn_off_when_deleted)
    {
      all_off(false);
      printf("The ADCs and DACs are shut down.\n");
    }

  hwfree(common_registers);

  for(int i=0; i<NUM_TILES; i++)
    hwfree(tile_registers[i]);
}


void rfdc::all_off(bool verbose)
{
  for(int i=0; i<NUM_TILES; i++)
    {
      tile_registers[i]->restart_state_register = 3;
      tile_registers[i]->restart_power_on_state_machine_register = 1;
    }
  
  int wait_count = 30; // 3 seconds max wait
  for(int i=0; i<NUM_TILES; i++)
    {      
      while(tile_registers[i]->restart_power_on_state_machine_register)
	{
	  if(!wait_count)
	    break;
	  wait_count--;
	  usleep(100000);
	}
    }

  if(!none_running(verbose))
    {
      printf("ERROR: not all RFDC tiles could be turned off.\n");
      exit(20);
    }
}


void rfdc::all_on(bool verbose)
{
  for(int i=0; i<NUM_TILES; i++)
    {
      tile_registers[i]->restart_state_register = 0xF;
      tile_registers[i]->restart_power_on_state_machine_register = 1;
    }
  
  int wait_count = 30; // 3 seconds max wait
  for(int i=0; i<NUM_TILES; i++)
    {      
      while(tile_registers[i]->restart_power_on_state_machine_register)
	{
	  if(!wait_count)
	    break;
	  wait_count--;
	  usleep(100000);
	}
    }

  if(!all_running(verbose))
    {
      printf("ERROR: not all RFDC tiles could be turned on.\n");
      //exit(20);
    }
}
