
// SPDX-License-Identifier: GPL-3.0-only
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, version 3.0 of the License.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <https://www.gnu.org/licenses/>.
//

//
// Note:  Some of this code originally started with Patrick Allison's
// libunivrfdc.  There's not so much left of what he was trying to do,
// but I owe him a debt for getting me started.
//


/* EVERYTHING GOES IN HERE! */
#ifndef METAL_SYS_H
#define METAL_SYS_H

#include <stdint.h>
#include <unistd.h>
#include <string.h>

#ifndef XPAR_XRFDC_NUM_INSTANCES
#define XPAR_XRFDC_NUM_INSTANCES 1
#endif

#define __BAREMETAL__

#define METAL_IRQ_HANDLED 0


typedef uint64_t metal_phys_addr_t;
typedef uint32_t __u32;
typedef uint16_t __u16;
typedef uint8_t __u8;
typedef int32_t __s32;
typedef int16_t __s16;
typedef int8_t __s8;

// Work around a conflicting definition by defining this way
typedef unsigned long long __u64;
typedef long long __s64;


typedef uint32_t (*read_function_t)(void *dev, uint32_t addr);
typedef void (*write_function_t)(void *dev, uint32_t addr, uint32_t value, uint32_t mask);

struct metal_io_region {
  int dummy;
};

#define metal_device_io_region metal_io_region

struct metal_device {
  char name[16];
};


#define METAL_INIT_DEFAULTS { 42 }

struct metal_init_params
{
  int dummy;
};


#define METAL_LOG_ERROR 0
#define METAL_LOG_WARNING 1
#define METAL_LOG_INFO 2
#define METAL_LOG_DEBUG 3

#define metal_sleep_usec usleep

int metal_init(struct metal_init_params* mip);
  

int metal_device_open(const char *bus_name,
			     const char *dev_name,
			     struct metal_device **device);

void metal_device_close(struct metal_device *device);

struct metal_io_region *metal_device_io_region( struct metal_device *dev,
						unsigned int index );

int metal_linux_get_device_property(struct metal_device *dev,
				    const char *property_name,
				    void *output,
				    int len);

void metal_log( int verbosity,
		       const char *format,
		       ...);

void metal_io_write16( struct metal_io_region *io,
		       unsigned long offset,
		       uint16_t value );

void metal_io_write32( struct metal_io_region *io,
		       unsigned long offset,
		       uint32_t value );

uint16_t metal_io_read16( struct metal_io_region *io,
			  unsigned long offset );

uint32_t metal_io_read32( struct metal_io_region *io,
			  unsigned long offset );

void metal_io_set_device( void *dev,
			  read_function_t read_function,
			  write_function_t write_function);




#ifdef __BAREMETAL__
//
// These are specifically for XRFdc
//
#define Xil_AssertNonvoid(Expression)
#define Xil_AssertVoid(Expression)
#define Xil_AssertVoidAlways()

typedef uint64_t u64;
typedef uint32_t u32;
typedef uint16_t u16;
typedef uint8_t u8;

typedef int64_t s64;
typedef int32_t s32;
typedef int16_t s16;
typedef int8_t  s8;

void* metal_allocate_memory(u32 length);

struct metal_io_ops;

void
metal_io_init(struct metal_io_region *io, void *virt,
	      const metal_phys_addr_t *physmap, size_t size,
	      unsigned int page_shift, unsigned int mem_flags,
	      const struct metal_io_ops *ops);


s32 metal_register_generic_device(struct metal_device *);

//
// End of stuff specifically for XRFdc
//
#endif

#endif
