
`timescale 1ns / 1ps

`define w assign
`define raxis always@(posedge AXIS_aclk)
`define raxi  always@(posedge S_AXI_ACLK)

//
// Only works for power-of-2 samples per clock.  Only works if AXI_SAMPLES_PER_WORD
// evenly divides into AXIS_SAMPLES_PER_CLOCK
//
module spectral_capture_96000R_12
  #(
    // Capture parameters
    parameter AXIS_SAMPLES_PER_CLOCK       = 8,
    parameter AXIS_BITS_PER_SAMPLE         = 16,
    parameter AXIS_BUS_WIDTH               = AXIS_BITS_PER_SAMPLE * AXIS_SAMPLES_PER_CLOCK,

    // ARM interface localparams
    parameter AXI_BITS_PER_WORD            = 32,
    parameter AXI_BYTES_PER_WORD           = AXI_BITS_PER_WORD/8,
    parameter AXI_BYTE_ADDR_WIDTH_PLUS     = 22,   // Bigger than it needs to be.

    parameter EXTERNAL_COLLECT             = 0,
    parameter EXTERNAL_CAPTURE_DELAY       = 0,

    parameter READ_DELAYS                  = 4,  // 0 is OK for higher FPGA DSP clocks
    
    parameter STATIC_SHIFTS                = 64'hAAAAAAAAA
    )
   (
    // Data collection and passthrough

    (* X_INTERFACE_INFO = "xilinx.com:signal:clock:1.0 AXIS_aclk CLK" *)
    (* X_INTERFACE_PARAMETER = "ASSOCIATED_BUSIF EARLY_AXIS:LATE_AXIS, ASSOCIATED_RESET AXIS_aresetn" *)
    input                                AXIS_aclk,

    (* X_INTERFACE_INFO = "xilinx.com:signal:reset:1.0 AXIS_aresetn RST" *)
    input                                AXIS_aresetn,

    input [AXIS_BUS_WIDTH-1:0]           EARLY_AXIS_tdata,
    output wire                          EARLY_AXIS_tready,
    input                                EARLY_AXIS_tvalid,
    input                                EARLY_AXIS_tlast,

    input [AXIS_BUS_WIDTH-1:0]           LATE_AXIS_tdata,
    output wire                          LATE_AXIS_tready,
    input                                LATE_AXIS_tvalid,
    input                                LATE_AXIS_tlast,

    // Set collect to 1 to begin collection.  Set collect to 0 to abort or reset collection.
    // Collection stops automatically and then sets first early_spectrum_done_o and then
    // late_spectrum_done_o.  To start a new collection, bring collect low and then high again.
    //
    // Note that collect_i is ORed with a value from an internal register.  So tie it to
    // zero to get the internal-only effect.
    //
    input                                collect_i,

    output                               early_spectrum_done_o,
    output                               late_spectrum_done_o,

    // This is the delay between the beginning of the early capture and the beginning of the
    // late capture.  It is added to a value from an internal register, so tie it to zero to
    // get the internal-only effect.
    input [31:0]                         late_capture_delay_i,

    //
    // This is the interface to the ARM processor
    //
    input                                     S_AXI_ACLK,
    input                                     S_AXI_ARESETN,
    input [AXI_BYTE_ADDR_WIDTH_PLUS-1 : 0]    S_AXI_AWADDR,
    input [2 : 0]                             S_AXI_AWPROT,
    input                                     S_AXI_AWVALID,
    output reg                                S_AXI_AWREADY,
    input [AXI_BITS_PER_WORD-1 : 0]           S_AXI_WDATA,
    input [AXI_BYTES_PER_WORD-1 : 0]          S_AXI_WSTRB,
    input                                     S_AXI_WVALID,
    output reg                                S_AXI_WREADY,
    output reg [1 : 0]                        S_AXI_BRESP,
    output reg                                S_AXI_BVALID,
    input                                     S_AXI_BREADY,
    input [AXI_BYTE_ADDR_WIDTH_PLUS-1 : 0]    S_AXI_ARADDR,
    input [2 : 0]                             S_AXI_ARPROT,
    input                                     S_AXI_ARVALID,
    output reg                                S_AXI_ARREADY,
    output reg [AXI_BITS_PER_WORD-1 : 0]      S_AXI_RDATA,
    output reg [1 : 0]                        S_AXI_RRESP,
    output reg                                S_AXI_RVALID,
    input                                     S_AXI_RREADY

);
   genvar                                  i, j;

   // Layout of capture memory
   localparam MEM_BITS_PER_REAL_SAMPLE     = 32;
   localparam MEM_BYTES_PER_REAL_SAMPLE    = (MEM_BITS_PER_REAL_SAMPLE+7)/8;
   localparam REALS_PER_COMPLEX            = 2;
   localparam MEM_SAMPLES_PER_SECTION      = 48000 / 6;
   localparam MEM_NUM_SECTIONS             = 6;
   localparam NUM_STREAMS_EARLY            = 1;
   localparam NUM_STREAMS_LATE             = 1;
   localparam NUM_STREAMS                  = NUM_STREAMS_EARLY + NUM_STREAMS_LATE;

   localparam AXIS_BYTES_PER_CLOCK         = (AXIS_BUS_WIDTH+7)/8;

   // Address bits are broken down into groups.  The first group is ADDR_BITS_BYTES_PER_REAL_SAMPLE.
   // There are currently AXI_BYTES_PER_REAL_SAMPLE=4 bytes per 32-bit sample, so 
   // ADDR_BITS_BYTES_PER_REAL_SAMPLE is $clog2(4) = 2.  Take off these 2 bits, and you're addressing 
   // real samples.
   //
   // The next section is COMPLEX values.  Actual outputs are in two (REAL,IMAGINARY) output values.
   // REAL is in the low address and IMAG is in the high address.  1 bit is used to distinguish them.
   // ADDR_BITS_IN_COMPLEX=1.
   //
   // The next group is sections of the FFT data.  The length of each section is 
   // AXI_SAMPLES_PER_SECTION=COMPLEX_FFT_SIZE/COMPLEX_PPC.  So the number of bits to address the 
   // section is $clog2(AXI_SAMPLES_PER_SECTION).  Note that if AXI_SAMPLES_PER_SECTION is not
   // a power of 2, then there will be a gap in the address space between the sections.
   //
   // The next group is the sections.  There are AXI_NUM_SECTIONS=COMPLEX_PPC sections.  The
   // address bits are $clog2(AXI_NUM_SECTIONS).
   //
   localparam ADDR_BITS_BYTES_PER_REAL_SAMPLE    = $clog2(MEM_BYTES_PER_REAL_SAMPLE);
   localparam ADDR_BITS_IN_COMPLEX               = $clog2(REALS_PER_COMPLEX);
   localparam ADDR_BITS_SAMPLES_PER_SECTION      = $clog2(MEM_SAMPLES_PER_SECTION);
   localparam ADDR_BITS_INSIDE_SECTION           = ADDR_BITS_SAMPLES_PER_SECTION;
   localparam ADDR_BITS_SECTION                  = ADDR_BITS_IN_COMPLEX + $clog2(MEM_NUM_SECTIONS);
   localparam ADDR_BITS_STREAMS                  = $clog2(NUM_STREAMS);
   
   localparam ADDR_WIDTH                         = ADDR_BITS_STREAMS + ADDR_BITS_SECTION + ADDR_BITS_INSIDE_SECTION + 1;  // +1 is to also cover CAPTURE_ADDR_OFFSET
   
   localparam ADDR_OFFSET                        = ADDR_BITS_BYTES_PER_REAL_SAMPLE;
   
   localparam ADDR_INSIDE_SECTION_OFFSET         = 0;  // ADDR_BITS_BYTES_PER_REAL_SAMPLE already removed
   localparam ADDR_SECTION_OFFSET                = ADDR_INSIDE_SECTION_OFFSET + ADDR_BITS_INSIDE_SECTION;
   localparam ADDR_STREAM_OFFSET                 = ADDR_SECTION_OFFSET + ADDR_BITS_SECTION;
       
   // ARM interface localparams
   localparam AXI_BYTE_ADDR_WIDTH                = ADDR_WIDTH + ADDR_BITS_BYTES_PER_REAL_SAMPLE;

   // This is a power of 2, so the read of the internal data can ignore it, and be overridden here at the top
   // level without doing an address subtract.
   localparam CAPTURE_ADDR_OFFSET                = (1<<(AXI_BYTE_ADDR_WIDTH-1));

   localparam REAL_WIDTH        = 27;
   localparam NUM_SYNC_READ     = 1;

   localparam AXIS_SHIFT_RIGHT  = 4;
   localparam TID_WIDTH         = 0;
   localparam TDEST_WIDTH       = 0;
   localparam TID_DUMMY_WIDTH   = 1;
   localparam TDEST_DUMMY_WIDTH = 1;
   localparam RESET_ADVANCEMENT = 16;
   localparam NUM_SYNC          = 1;
   localparam REAL_PPC          = AXIS_SAMPLES_PER_CLOCK;

   generate
      if(AXI_BYTE_ADDR_WIDTH>AXI_BYTE_ADDR_WIDTH_PLUS)
	error_module  fix_this("Address bits aren't sufficient at top level");
   endgenerate

   //
   // Signals to convert AXIS signalling to BxBFFT syncs.
   //
   wire [NUM_SYNC-1:0] 			   early_in_sync_w;
   wire signed [REAL_WIDTH-1:0] 	   early_in_data_w[0:REAL_PPC-1];
   wire signed [REAL_WIDTH-1:0] 	   early_in_data_i[0:0][0:REAL_PPC-1];
   wire [NUM_SYNC-1:0] 			   late_in_sync_w;
   wire signed [REAL_WIDTH-1:0] 	   late_in_data_w[0:REAL_PPC-1];
   wire signed [REAL_WIDTH-1:0] 	   late_in_data_i[0:0][0:REAL_PPC-1];
   reg [TID_DUMMY_WIDTH-1:0] 		   EARLY_AXIS_tid                      = 0;
   reg [TDEST_DUMMY_WIDTH-1:0] 		   EARLY_AXIS_tdest                    = 0;
   reg [TID_DUMMY_WIDTH-1:0] 		   LATE_AXIS_tid                       = 0;
   reg [TDEST_DUMMY_WIDTH-1:0] 		   LATE_AXIS_tdest                     = 0;
   reg [AXIS_BYTES_PER_CLOCK-1:0] 	   EARLY_AXIS_tstrb                    = 0;
   reg [AXIS_BYTES_PER_CLOCK-1:0] 	   EARLY_AXIS_tkeep                    = 0;
   reg [AXIS_BYTES_PER_CLOCK-1:0] 	   LATE_AXIS_tstrb                     = 0;
   reg [AXIS_BYTES_PER_CLOCK-1:0] 	   LATE_AXIS_tkeep                     = 0;

   genvar 				   m;
   generate
      for(m=0; m<REAL_PPC; m=m+1)
	begin
	   `w early_in_data_i[0][m] 	   = early_in_data_w[m];
	   `w late_in_data_i[0][m]  = late_in_data_w[m];
	end
   endgenerate

   reg                                                                                      collect_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg                                   collect_arr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *) reg         collect_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg 		                    collect_srr;

   reg [31:0]                                                                               late_delay_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg [31:0]                            late_delay_arr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *) reg [31:0]  late_delay_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg [31:0] 			    late_delay_srr;
   reg [31:0] 										    late_delay_value_r;

   `raxi  collect_arr    <= collect_ar;
   `raxis collect_sr     <= collect_arr;
   `raxis collect_srr    <= collect_sr;

   `raxi  late_delay_arr  <= late_delay_ar;
   `raxis late_delay_sr   <= late_delay_arr;
   `raxis late_delay_srr  <= late_delay_sr;

   generate
      if(EXTERNAL_CAPTURE_DELAY)
        `raxis late_delay_value_r <= late_delay_srr + late_capture_delay_i;
      else
        `raxis late_delay_value_r <= late_delay_srr;
   endgenerate



   axis_to_bxb  #( .POINTS_PER_CLOCK     ( AXIS_SAMPLES_PER_CLOCK/2 ),
                   .S_AXIS_SAMPLE_WIDTH  ( AXIS_BITS_PER_SAMPLE     ),
                   .S_AXIS_SHIFT_RIGHT   ( AXIS_SHIFT_RIGHT         ),
                   .REAL_WIDTH           ( REAL_WIDTH               ),
                   .TID_WIDTH            ( TID_WIDTH                ),
                   .TDEST_WIDTH          ( TDEST_WIDTH              ),
                   .RESET_ADVANCEMENT    ( RESET_ADVANCEMENT        ))
   a2b_early     ( .clk                  ( AXIS_aclk                ),
		   .resetn               ( AXIS_aresetn             ),
		   
                   .s_axis_tvalid        ( EARLY_AXIS_tvalid        ),
                   .s_axis_tready        ( EARLY_AXIS_tready        ),
                   .s_axis_tdata         ( EARLY_AXIS_tdata         ),
                   .s_axis_tstrb         ( EARLY_AXIS_tstrb         ),
                   .s_axis_tkeep         ( EARLY_AXIS_tkeep         ),
                   .s_axis_tlast         ( EARLY_AXIS_tlast         ),
                   .s_axis_tid           ( EARLY_AXIS_tid           ),
                   .s_axis_tdest         ( EARLY_AXIS_tdest         ),
		   
                   .bxb_sync_o           ( early_in_sync_w          ),
                   .bxb_data_o           ( early_in_data_w          ));


   axis_to_bxb  #( .POINTS_PER_CLOCK     ( AXIS_SAMPLES_PER_CLOCK/2 ),
                   .S_AXIS_SAMPLE_WIDTH  ( AXIS_BITS_PER_SAMPLE     ),
                   .S_AXIS_SHIFT_RIGHT   ( AXIS_SHIFT_RIGHT         ),
                   .REAL_WIDTH           ( REAL_WIDTH               ),
                   .TID_WIDTH            ( TID_WIDTH                ),
                   .TDEST_WIDTH          ( TDEST_WIDTH              ),
                   .RESET_ADVANCEMENT    ( RESET_ADVANCEMENT        ))
   a2b_late      ( .clk                  ( AXIS_aclk                ),
		   .resetn               ( AXIS_aresetn             ),
		   
                   .s_axis_tvalid        ( LATE_AXIS_tvalid         ),
                   .s_axis_tready        ( LATE_AXIS_tready         ),
                   .s_axis_tdata         ( LATE_AXIS_tdata          ),
                   .s_axis_tstrb         ( LATE_AXIS_tstrb          ),
                   .s_axis_tkeep         ( LATE_AXIS_tkeep          ),
                   .s_axis_tlast         ( LATE_AXIS_tlast          ),
                   .s_axis_tid           ( LATE_AXIS_tid            ),
                   .s_axis_tdest         ( LATE_AXIS_tdest          ),
		   
                   .bxb_sync_o           ( late_in_sync_w           ),
                   .bxb_data_o           ( late_in_data_w           ));


   wire [ADDR_BITS_STREAMS-1:0] 	   read_stream_i;
   wire [ADDR_BITS_INSIDE_SECTION-1:0] 	   read_address_i;
   wire [ADDR_BITS_SECTION-1:0] 	   read_ppc_i;
   wire signed [REAL_WIDTH-1:0] 	   read_data_o;
   
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                           reg signed [31:0] read_data_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *) reg signed [31:0] read_data_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                           reg signed [31:0] read_data_arr;
   
   wire [NUM_SYNC_READ-1:0] 		   read_sync_in_i;
   wire [NUM_SYNC_READ-1:0] 		   read_sync_out_o;
   wire [1:0] 				   spectrum_done_o;
   wire                                    collect_w;

   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                              reg [NUM_STREAMS-1:0]  spectrum_done_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)    reg [NUM_STREAMS-1:0]  spectrum_done_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                              reg [NUM_STREAMS-1:0]  spectrum_done_arr;

   `raxis spectrum_done_sr  <= spectrum_done_o;
   `raxi  spectrum_done_ar  <= spectrum_done_sr;
   `raxi  spectrum_done_arr <= spectrum_done_ar;
   
   generate
      if(EXTERNAL_COLLECT)
        `w collect_w = collect_i || collect_srr;
      else
        `w collect_w = collect_srr;
   endgenerate

   spectrum_pfb  #( .REAL_WIDTH            ( REAL_WIDTH                  ),
                    .STATIC_SHIFTS         ( STATIC_SHIFTS               ),
                    .NUM_STREAMS_EARLY     ( NUM_STREAMS_EARLY           ),
                    .NUM_STREAMS_LATE      ( NUM_STREAMS_LATE            ),
                    .NUM_SYNC_READ         ( NUM_SYNC_READ               ),
		    .RESET_ADVANCEMENT     ( RESET_ADVANCEMENT           ))
   spect          ( .clk                   ( AXIS_aclk                   ),
                    .resetn                ( AXIS_aresetn                ),
                    .run_i                 ( collect_w                   ),
                    .in_data_early_i       ( early_in_data_i             ),
                    .in_data_late_i        ( late_in_data_i              ),
                    .late_capture_delay_i  ( late_delay_value_r          ),
                    .spectrum_done_o       ( spectrum_done_o             ),
                    .read_stream_i         ( read_stream_i               ),
                    .read_address_i        ( read_address_i              ),
                    .read_ppc_i            ( read_ppc_i                  ),
                    .read_data_o           ( read_data_o                 ),
                    .read_sync_in_i        ( read_sync_in_i              ),
                    .read_sync_out_o       ( read_sync_out_o             ));


   `w early_spectrum_done_o = spectrum_done_o[NUM_STREAMS_EARLY-1];
   `w late_spectrum_done_o  = spectrum_done_o[NUM_STREAMS-1];


   `raxis read_data_sr  <= read_data_o;
   `raxi  read_data_ar  <= read_data_sr;  // Sign extends, as well as a clock transition
   `raxi  read_data_arr <= read_data_ar;  // Second register to guarantee stability of clock transition
   
   //
   // Handle READs/WRITEs from the processor on the AXI bus.
   //    

   reg [ADDR_WIDTH-1 : 0]                  write_address;
   reg [AXI_BITS_PER_WORD-1:0]             write_data;
   wire                                    write_enable;
   wire                                    write_enable_w;
   reg                                     write_enable_r;
   wire                                    write_done;
   
   reg                                     read_enable;

   reg [ADDR_WIDTH-1 : 0]                  read_address;
   wire [AXI_BITS_PER_WORD-1:0]            read_data;
   wire                                    read_done;
   
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                           reg [ADDR_WIDTH-1:0] read_address_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *) reg [ADDR_WIDTH-1:0] read_address_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                           reg [ADDR_WIDTH-1:0] read_address_srr;

   //
   // The write interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_AWREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_AWREADY && S_AXI_AWVALID)   ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_AWREADY  ;

   `raxi S_AXI_WREADY  <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_WREADY && S_AXI_WVALID)     ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_WREADY   ;

   `raxi S_AXI_BVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          write_done                         ?  1'b1           :
                          (S_AXI_BREADY && S_AXI_BVALID)     ?  1'b0           :
                          /* default */                         S_AXI_BVALID   ;

   `raxi S_AXI_BRESP   <= 2'b0;

   //
   // The read interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_ARREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_ARREADY && S_AXI_ARVALID)   ?  1'b0           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b1           :
                          /* default */                         S_AXI_ARREADY  ;
   

   `raxi S_AXI_RVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          read_done                          ?  1'b1           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b0           :
                          /* default */                         S_AXI_RVALID   ;

   `raxi S_AXI_RRESP   <= 2'b0;
   
   `raxi S_AXI_RDATA   <= (!S_AXI_ARESETN)                   ?  0              : 
                          read_done                          ?  read_data      :
                          /* default */                         S_AXI_RDATA    ;


   //
   // Define write addresses, data, and enables
   //
   `w write_enable_w          = !S_AXI_WREADY && !S_AXI_AWREADY;
   `raxi write_enable_r       = (!S_AXI_ARESETN) ? 0 : write_enable_w;
   `w write_enable            = write_enable_w && !write_enable_r;   // Only pulse write enable for a single clock, regardless of BREADY
   
   `raxi write_address       <= (!S_AXI_ARESETN)                  ?  0                                                            :
                                (S_AXI_AWREADY && S_AXI_AWVALID)  ?  S_AXI_AWADDR[ADDR_BITS_BYTES_PER_REAL_SAMPLE +: ADDR_WIDTH]  :
                                /* default */                        write_address                                                ;
   
   `raxi write_data          <= (!S_AXI_ARESETN)                  ?  0                :
                                (S_AXI_WREADY && S_AXI_WVALID)    ?  S_AXI_WDATA      :
                                /* default */                        write_data       ;

   `w write_done              = write_enable;  // Write done immediately after address and data are both present
   

   //
   // Define read addresses, data, and enables
   //
   `raxi read_address        <= (!S_AXI_ARESETN)               ?  0                                                            : 
                                S_AXI_ARREADY & S_AXI_ARVALID  ?  S_AXI_ARADDR[ADDR_BITS_BYTES_PER_REAL_SAMPLE +: ADDR_WIDTH]  :
                                /* default */                     read_address                                                 ;

   `raxi  read_address_ar    <= read_address;
   `raxis read_address_sr    <= read_address_ar;
   `raxis read_address_srr   <= read_address_sr;
   
   `raxi read_enable         <= S_AXI_ARREADY & S_AXI_ARVALID;


   reg 					   read_enable_r;
   reg 					   read_done_pulse_r;

   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                           reg read_done_toggle_r = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *) reg read_done_toggle_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                           reg read_done_toggle_srr;
   
   `raxi read_enable_r       <= read_enable;
   
   `raxi read_done_pulse_r   <= read_enable && !read_enable_r;                                 // Pulse on leading edge of read_enable
   `raxi read_done_toggle_r  <= read_done_pulse_r ? !read_done_toggle_r : read_done_toggle_r;  // Turn pulse into a toggle

   reg 					   read_done_toggle_srrr;
   reg 					   read_done_toggle_srrrr;
   reg 					   read_done_toggle_srrrrr;
   reg 					   read_done_toggle_srrrrrr;
   reg 					   read_done_toggle_srrrrrrr;
   reg 					   read_done_toggle_srrrrrrrr;
   reg 					   read_done_toggle_srrrrrrrrr;
   reg 					   read_done_toggle_srrrrrrrrrr;
   
   `raxis read_done_toggle_sr          <= read_done_toggle_r;
   `raxis read_done_toggle_srr         <= read_done_toggle_sr;

   `w read_sync_in_i                    = read_done_toggle_srr;
   
   `raxis read_done_toggle_srrr        <= read_sync_out_o;
   `raxis read_done_toggle_srrrr       <= read_done_toggle_srrr;
   `raxis read_done_toggle_srrrrr      <= read_done_toggle_srrrr;
   `raxis read_done_toggle_srrrrrr     <= read_done_toggle_srrrrr;
   `raxis read_done_toggle_srrrrrrr    <= read_done_toggle_srrrrrr;
   `raxis read_done_toggle_srrrrrrrr   <= read_done_toggle_srrrrrrr;
   `raxis read_done_toggle_srrrrrrrrr  <= read_done_toggle_srrrrrrrr;
   `raxis read_done_toggle_srrrrrrrrrr <= read_done_toggle_srrrrrrrrr;

   // Read data has a bit of delay after address is present.  It goes through
   // a clock transition and back.  Must have adequate delay.

   wire 				   read_done_toggle_sd;
   
   `w read_done_toggle_sd     = (READ_DELAYS==0) ? read_done_toggle_srrr         :
				(READ_DELAYS==1) ? read_done_toggle_srrrr        :
				(READ_DELAYS==2) ? read_done_toggle_srrrrr       :
				(READ_DELAYS==3) ? read_done_toggle_srrrrrr      :
				(READ_DELAYS==4) ? read_done_toggle_srrrrrrr     :
				(READ_DELAYS==5) ? read_done_toggle_srrrrrrrr    :
				(READ_DELAYS==6) ? read_done_toggle_srrrrrrrrr   :
				/**/               read_done_toggle_srrrrrrrrrr  ;

   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg read_done_toggle_sdr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg read_done_toggle_dr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg read_done_toggle_drr;
   reg read_done_toggle_drrr;

   `raxis read_done_toggle_sdr  <= read_done_toggle_sd;
   `raxi  read_done_toggle_dr   <= read_done_toggle_sdr;
   `raxi  read_done_toggle_drr  <= read_done_toggle_dr;
   `raxi  read_done_toggle_drrr <= read_done_toggle_drr;
   
   reg read_done_pulse_dr;
   
   `raxi read_done_pulse_dr   <= (read_done_toggle_drr != read_done_toggle_drrr);

   `w read_done = read_done_pulse_dr;
   
   `w  read_address_i         = read_address_srr >> ADDR_INSIDE_SECTION_OFFSET;
   `w  read_ppc_i             = read_address_srr >> ADDR_SECTION_OFFSET;   
   `w  read_stream_i          = read_address_srr >> ADDR_STREAM_OFFSET;

   //
   // Handle read data
   //   
   `w read_data = (read_address==0)  ?  collect_ar                       :
                  (read_address==1)  ?  late_delay_ar                    :
                  (read_address==2)  ?  spectrum_done_arr                :
                  (read_address==3)  ?  CAPTURE_ADDR_OFFSET              :
                  (read_address==4)  ?  MEM_SAMPLES_PER_SECTION          :
                  (read_address==5)  ?  MEM_NUM_SECTIONS                 :
                  (read_address==6)  ?  NUM_STREAMS_EARLY                :
                  (read_address==7)  ?  NUM_STREAMS_LATE                 :
                  /* default */         read_data_arr                    ;

   
   //
   // Handle write data.  Registers write on the AXI clock.
   // None of the other registers are supposed to be written unless reg_trigger_arm_ar is 0.
   // Thus they won't be written at a critical time and thus metastability isn't an issue.
   //
   
   `raxi collect_ar             <=  (!S_AXI_ARESETN)                       ?  0                      :
                                    write_enable && (write_address==0)     ?  write_data             :
                                    /* default */                             collect_ar             ;

   `raxi late_delay_ar          <=  (!S_AXI_ARESETN)                       ?  0                      :
                                    write_enable && (write_address==1)     ?  write_data             :
                                    /* default */                             late_delay_ar          ;
  
     
endmodule
