
//`timescale 1ns / 1ps

`define w assign
`define raxis always@(posedge IN_AXIS_aclk)
`define raxi  always@(posedge S_AXI_ACLK)

//
// Only works for power-of-2 samples per clock.  Only works if AXI_SAMPLES_PER_WORD
// evenly divides into AXIS_SAMPLES_PER_CLOCK
//
module spectrum_capture_control
  #(
    // Capture parameters
    parameter AXIS_SAMPLES_PER_CLOCK       = 8,
    parameter AXIS_BITS_PER_SAMPLE         = 16,
    parameter AXIS_BUS_WIDTH               = AXIS_BITS_PER_SAMPLE * AXIS_SAMPLES_PER_CLOCK,
    
    // ARM interface parameters
    parameter AXI_BITS_PER_WORD            = 32,
    parameter AXI_BYTES_PER_WORD           = AXI_BITS_PER_WORD/8,
    parameter NUM_REGISTERS                = 16,  // Must be greater than 1
    parameter AXI_BYTE_ADDR_WIDTH          = $clog2( NUM_REGISTERS * AXI_BYTES_PER_WORD ),

    parameter REGISTER_THE_PASSTHROUGH     = 0
    )
   (
    // Data collection options, muxed to output

    (* X_INTERFACE_INFO = "xilinx.com:signal:clock:1.0 IN_AXIS_aclk CLK" *)
    (* X_INTERFACE_PARAMETER = "ASSOCIATED_BUSIF IN_0_AXIS:IN_1_AXIS:IN_2_AXIS:IN_3_AXIS:IN_4_AXIS:IN_5_AXIS:IN_6_AXIS:IN_7_AXIS:IN_8_AXIS:IN_9_AXIS:IN_10_AXIS:IN_11_AXIS:IN_12_AXIS:IN_13_AXIS:IN_14_AXIS:IN_15_AXIS:PASS_0_AXIS:PASS_1_AXIS:PASS_2_AXIS:PASS_3_AXIS:PASS_4_AXIS:PASS_5_AXIS:PASS_6_AXIS:PASS_7_AXIS:PASS_8_AXIS:PASS_9_AXIS:PASS_10_AXIS:PASS_11_AXIS:PASS_12_AXIS:PASS_13_AXIS:PASS_14_AXIS:PASS_15_AXIS:OUT_AXIS, ASSOCIATED_RESET IN_AXIS_aresetn" *)
    input                                IN_AXIS_aclk,

    (* X_INTERFACE_INFO = "xilinx.com:signal:reset:1.0 IN_AXIS_aresetn RST" *)
    input                                IN_AXIS_aresetn,

    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_0_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_0_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_0_AXIS TREADY" *)  output wire                          IN_0_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_0_AXIS TVALID" *)  input                                IN_0_AXIS_tvalid,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_1_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_1_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_1_AXIS TREADY" *)  output wire                          IN_1_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_1_AXIS TVALID" *)  input                                IN_1_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_2_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_2_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_2_AXIS TREADY" *)  output wire                          IN_2_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_2_AXIS TVALID" *)  input                                IN_2_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_3_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_3_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_3_AXIS TREADY" *)  output wire                          IN_3_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_3_AXIS TVALID" *)  input                                IN_3_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_4_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_4_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_4_AXIS TREADY" *)  output wire                          IN_4_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_4_AXIS TVALID" *)  input                                IN_4_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_5_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_5_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_5_AXIS TREADY" *)  output wire                          IN_5_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_5_AXIS TVALID" *)  input                                IN_5_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_6_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_6_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_6_AXIS TREADY" *)  output wire                          IN_6_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_6_AXIS TVALID" *)  input                                IN_6_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_7_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_7_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_7_AXIS TREADY" *)  output wire                          IN_7_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_7_AXIS TVALID" *)  input                                IN_7_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_8_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_8_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_8_AXIS TREADY" *)  output wire                          IN_8_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_8_AXIS TVALID" *)  input                                IN_8_AXIS_tvalid,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_9_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]           IN_9_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_9_AXIS TREADY" *)  output wire                          IN_9_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_9_AXIS TVALID" *)  input                                IN_9_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_10_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]          IN_10_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_10_AXIS TREADY" *)  output wire                         IN_10_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_10_AXIS TVALID" *)  input                               IN_10_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_11_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]          IN_11_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_11_AXIS TREADY" *)  output wire                         IN_11_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_11_AXIS TVALID" *)  input                               IN_11_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_12_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]          IN_12_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_12_AXIS TREADY" *)  output wire                         IN_12_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_12_AXIS TVALID" *)  input                               IN_12_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_13_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]          IN_13_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_13_AXIS TREADY" *)  output wire                         IN_13_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_13_AXIS TVALID" *)  input                               IN_13_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_14_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]          IN_14_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_14_AXIS TREADY" *)  output wire                         IN_14_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_14_AXIS TVALID" *)  input                               IN_14_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_15_AXIS TDATA" *)   input [AXIS_BUS_WIDTH-1:0]          IN_15_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_15_AXIS TREADY" *)  output wire                         IN_15_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 IN_15_AXIS TVALID" *)  input                               IN_15_AXIS_tvalid,


    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_0_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_0_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_0_AXIS TREADY" *)  input                              PASS_0_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_0_AXIS TVALID" *)  output                             PASS_0_AXIS_tvalid,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_1_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_1_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_1_AXIS TREADY" *)  input                              PASS_1_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_1_AXIS TVALID" *)  output                             PASS_1_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_2_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_2_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_2_AXIS TREADY" *)  input                              PASS_2_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_2_AXIS TVALID" *)  output                             PASS_2_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_3_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_3_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_3_AXIS TREADY" *)  input                              PASS_3_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_3_AXIS TVALID" *)  output                             PASS_3_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_4_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_4_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_4_AXIS TREADY" *)  input                              PASS_4_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_4_AXIS TVALID" *)  output                             PASS_4_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_5_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_5_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_5_AXIS TREADY" *)  input                              PASS_5_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_5_AXIS TVALID" *)  output                             PASS_5_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_6_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_6_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_6_AXIS TREADY" *)  input                              PASS_6_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_6_AXIS TVALID" *)  output                             PASS_6_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_7_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_7_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_7_AXIS TREADY" *)  input                              PASS_7_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_7_AXIS TVALID" *)  output                             PASS_7_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_8_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_8_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_8_AXIS TREADY" *)  input                              PASS_8_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_8_AXIS TVALID" *)  output                             PASS_8_AXIS_tvalid,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_9_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]        PASS_9_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_9_AXIS TREADY" *)  input                              PASS_9_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_9_AXIS TVALID" *)  output                             PASS_9_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_10_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]       PASS_10_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_10_AXIS TREADY" *)  input                             PASS_10_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_10_AXIS TVALID" *)  output                            PASS_10_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_11_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]       PASS_11_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_11_AXIS TREADY" *)  input                             PASS_11_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_11_AXIS TVALID" *)  output                            PASS_11_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_12_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]       PASS_12_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_12_AXIS TREADY" *)  input                             PASS_12_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_12_AXIS TVALID" *)  output                            PASS_12_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_13_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]       PASS_13_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_13_AXIS TREADY" *)  input                             PASS_13_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_13_AXIS TVALID" *)  output                            PASS_13_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_14_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]       PASS_14_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_14_AXIS TREADY" *)  input                             PASS_14_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_14_AXIS TVALID" *)  output                            PASS_14_AXIS_tvalid,

    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_15_AXIS TDATA" *)   output [AXIS_BUS_WIDTH-1:0]       PASS_15_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_15_AXIS TREADY" *)  input                             PASS_15_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 PASS_15_AXIS TVALID" *)  output                            PASS_15_AXIS_tvalid,

    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_AXIS TREADY" *)  input                                 OUT_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_AXIS TVALID" *)  output reg                            OUT_AXIS_tvalid,
    

    // Data is continuously collected while collect is high.
    // Bringing collect low stops collection immediately
    // so the data can be read, unless collection is synced
    // with tlast (an option on the collecting module), in 
    // which case collection ends at the next tlast.
    //
    // Intended operation is to stop collection, then
    // increment the mux to the next selection for capture.
    // Incrementing the mux changes the data going to the
    // collection buffer eventually, but since this data
    // goes through the PFB/FFT with some latency, it will
    // not affect the current capture.
    //
    // Care must be taken to avoid capturing more data
    // too soon after change of the mux.  If data is
    // captured too soon, it will be corrupted with data
    // from before the mux change.
    //
    output                               collect,

    
    //
    // This is the interface to the ARM processor
    //
    input                                S_AXI_ACLK,
    input                                S_AXI_ARESETN,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_AWADDR,
    input [2 : 0]                        S_AXI_AWPROT,
    input                                S_AXI_AWVALID,
    output reg                           S_AXI_AWREADY,
    input [AXI_BITS_PER_WORD-1 : 0]      S_AXI_WDATA,
    input [AXI_BYTES_PER_WORD-1 : 0]     S_AXI_WSTRB,
    input                                S_AXI_WVALID,
    output reg                           S_AXI_WREADY,
    output reg [1 : 0]                   S_AXI_BRESP,
    output reg                           S_AXI_BVALID,
    input                                S_AXI_BREADY,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_ARADDR,
    input [2 : 0]                        S_AXI_ARPROT,
    input                                S_AXI_ARVALID,
    output reg                           S_AXI_ARREADY,
    output reg [AXI_BITS_PER_WORD-1 : 0] S_AXI_RDATA,
    output reg [1 : 0]                   S_AXI_RRESP,
    output reg                           S_AXI_RVALID,
    input                                S_AXI_RREADY
);
   

   genvar                                i, j;

   reg [AXIS_BUS_WIDTH-1:0]              PASS_0_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_1_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_2_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_3_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_4_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_5_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_6_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_7_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_8_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_9_AXIS_tdata_r    = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_10_AXIS_tdata_r   = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_11_AXIS_tdata_r   = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_12_AXIS_tdata_r   = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_13_AXIS_tdata_r   = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_14_AXIS_tdata_r   = 0;
   reg [AXIS_BUS_WIDTH-1:0]              PASS_15_AXIS_tdata_r   = 0;

   reg                                   PASS_0_AXIS_tvalid_r   = 0;
   reg                                   PASS_1_AXIS_tvalid_r   = 0;
   reg                                   PASS_2_AXIS_tvalid_r   = 0;
   reg                                   PASS_3_AXIS_tvalid_r   = 0;
   reg                                   PASS_4_AXIS_tvalid_r   = 0;
   reg                                   PASS_5_AXIS_tvalid_r   = 0;
   reg                                   PASS_6_AXIS_tvalid_r   = 0;
   reg                                   PASS_7_AXIS_tvalid_r   = 0;
   reg                                   PASS_8_AXIS_tvalid_r   = 0;
   reg                                   PASS_9_AXIS_tvalid_r   = 0;
   reg                                   PASS_10_AXIS_tvalid_r  = 0;
   reg                                   PASS_11_AXIS_tvalid_r  = 0;
   reg                                   PASS_12_AXIS_tvalid_r  = 0;
   reg                                   PASS_13_AXIS_tvalid_r  = 0;
   reg                                   PASS_14_AXIS_tvalid_r  = 0;
   reg                                   PASS_15_AXIS_tvalid_r  = 0;

   generate
      if(REGISTER_THE_PASSTHROUGH)
        begin

           `raxis PASS_0_AXIS_tdata_r     <= IN_0_AXIS_tdata;
           `raxis PASS_1_AXIS_tdata_r     <= IN_1_AXIS_tdata;
           `raxis PASS_2_AXIS_tdata_r     <= IN_2_AXIS_tdata;
           `raxis PASS_3_AXIS_tdata_r     <= IN_3_AXIS_tdata;
           `raxis PASS_4_AXIS_tdata_r     <= IN_4_AXIS_tdata;
           `raxis PASS_5_AXIS_tdata_r     <= IN_5_AXIS_tdata;
           `raxis PASS_6_AXIS_tdata_r     <= IN_6_AXIS_tdata;
           `raxis PASS_7_AXIS_tdata_r     <= IN_7_AXIS_tdata;
           `raxis PASS_8_AXIS_tdata_r     <= IN_8_AXIS_tdata;
           `raxis PASS_9_AXIS_tdata_r     <= IN_9_AXIS_tdata;
           `raxis PASS_10_AXIS_tdata_r    <= IN_10_AXIS_tdata;
           `raxis PASS_11_AXIS_tdata_r    <= IN_11_AXIS_tdata;
           `raxis PASS_12_AXIS_tdata_r    <= IN_12_AXIS_tdata;
           `raxis PASS_13_AXIS_tdata_r    <= IN_13_AXIS_tdata;
           `raxis PASS_14_AXIS_tdata_r    <= IN_14_AXIS_tdata;
           `raxis PASS_15_AXIS_tdata_r    <= IN_15_AXIS_tdata;
           
           `raxis PASS_0_AXIS_tvalid_r    <= IN_0_AXIS_tvalid;
           `raxis PASS_1_AXIS_tvalid_r    <= IN_1_AXIS_tvalid;
           `raxis PASS_2_AXIS_tvalid_r    <= IN_2_AXIS_tvalid;
           `raxis PASS_3_AXIS_tvalid_r    <= IN_3_AXIS_tvalid;
           `raxis PASS_4_AXIS_tvalid_r    <= IN_4_AXIS_tvalid;
           `raxis PASS_5_AXIS_tvalid_r    <= IN_5_AXIS_tvalid;
           `raxis PASS_6_AXIS_tvalid_r    <= IN_6_AXIS_tvalid;
           `raxis PASS_7_AXIS_tvalid_r    <= IN_7_AXIS_tvalid;
           `raxis PASS_8_AXIS_tvalid_r    <= IN_8_AXIS_tvalid;
           `raxis PASS_9_AXIS_tvalid_r    <= IN_9_AXIS_tvalid;
           `raxis PASS_10_AXIS_tvalid_r   <= IN_10_AXIS_tvalid;
           `raxis PASS_11_AXIS_tvalid_r   <= IN_11_AXIS_tvalid;
           `raxis PASS_12_AXIS_tvalid_r   <= IN_12_AXIS_tvalid;
           `raxis PASS_13_AXIS_tvalid_r   <= IN_13_AXIS_tvalid;
           `raxis PASS_14_AXIS_tvalid_r   <= IN_14_AXIS_tvalid;
           `raxis PASS_15_AXIS_tvalid_r   <= IN_15_AXIS_tvalid;
           
           `w PASS_0_AXIS_tdata   = PASS_0_AXIS_tdata_r;
           `w PASS_1_AXIS_tdata   = PASS_1_AXIS_tdata_r;
           `w PASS_2_AXIS_tdata   = PASS_2_AXIS_tdata_r;
           `w PASS_3_AXIS_tdata   = PASS_3_AXIS_tdata_r;
           `w PASS_4_AXIS_tdata   = PASS_4_AXIS_tdata_r;
           `w PASS_5_AXIS_tdata   = PASS_5_AXIS_tdata_r;
           `w PASS_6_AXIS_tdata   = PASS_6_AXIS_tdata_r;
           `w PASS_7_AXIS_tdata   = PASS_7_AXIS_tdata_r;
           `w PASS_8_AXIS_tdata   = PASS_8_AXIS_tdata_r;
           `w PASS_9_AXIS_tdata   = PASS_9_AXIS_tdata_r;
           `w PASS_10_AXIS_tdata  = PASS_10_AXIS_tdata_r;
           `w PASS_11_AXIS_tdata  = PASS_11_AXIS_tdata_r;
           `w PASS_12_AXIS_tdata  = PASS_12_AXIS_tdata_r;
           `w PASS_13_AXIS_tdata  = PASS_13_AXIS_tdata_r;
           `w PASS_14_AXIS_tdata  = PASS_14_AXIS_tdata_r;
           `w PASS_15_AXIS_tdata  = PASS_15_AXIS_tdata_r;

           `w PASS_0_AXIS_tvalid  = PASS_0_AXIS_tvalid_r;
           `w PASS_1_AXIS_tvalid  = PASS_1_AXIS_tvalid_r;
           `w PASS_2_AXIS_tvalid  = PASS_2_AXIS_tvalid_r;
           `w PASS_3_AXIS_tvalid  = PASS_3_AXIS_tvalid_r;
           `w PASS_4_AXIS_tvalid  = PASS_4_AXIS_tvalid_r;
           `w PASS_5_AXIS_tvalid  = PASS_5_AXIS_tvalid_r;
           `w PASS_6_AXIS_tvalid  = PASS_6_AXIS_tvalid_r;
           `w PASS_7_AXIS_tvalid  = PASS_7_AXIS_tvalid_r;
           `w PASS_8_AXIS_tvalid  = PASS_8_AXIS_tvalid_r;
           `w PASS_9_AXIS_tvalid  = PASS_9_AXIS_tvalid_r;
           `w PASS_10_AXIS_tvalid = PASS_10_AXIS_tvalid_r;
           `w PASS_11_AXIS_tvalid = PASS_11_AXIS_tvalid_r;
           `w PASS_12_AXIS_tvalid = PASS_12_AXIS_tvalid_r;
           `w PASS_13_AXIS_tvalid = PASS_13_AXIS_tvalid_r;
           `w PASS_14_AXIS_tvalid = PASS_14_AXIS_tvalid_r;
           `w PASS_15_AXIS_tvalid = PASS_15_AXIS_tvalid_r;
           
        end // if (REGISTER_THE_PASSTHROUGH)
      else
        begin

           `w PASS_0_AXIS_tdata   = IN_0_AXIS_tdata;
           `w PASS_1_AXIS_tdata   = IN_1_AXIS_tdata;
           `w PASS_2_AXIS_tdata   = IN_2_AXIS_tdata;
           `w PASS_3_AXIS_tdata   = IN_3_AXIS_tdata;
           `w PASS_4_AXIS_tdata   = IN_4_AXIS_tdata;
           `w PASS_5_AXIS_tdata   = IN_5_AXIS_tdata;
           `w PASS_6_AXIS_tdata   = IN_6_AXIS_tdata;
           `w PASS_7_AXIS_tdata   = IN_7_AXIS_tdata;
           `w PASS_8_AXIS_tdata   = IN_8_AXIS_tdata;
           `w PASS_9_AXIS_tdata   = IN_9_AXIS_tdata;
           `w PASS_10_AXIS_tdata  = IN_10_AXIS_tdata;
           `w PASS_11_AXIS_tdata  = IN_11_AXIS_tdata;
           `w PASS_12_AXIS_tdata  = IN_12_AXIS_tdata;
           `w PASS_13_AXIS_tdata  = IN_13_AXIS_tdata;
           `w PASS_14_AXIS_tdata  = IN_14_AXIS_tdata;
           `w PASS_15_AXIS_tdata  = IN_15_AXIS_tdata;

           `w PASS_0_AXIS_tvalid  = IN_0_AXIS_tvalid;
           `w PASS_1_AXIS_tvalid  = IN_1_AXIS_tvalid;
           `w PASS_2_AXIS_tvalid  = IN_2_AXIS_tvalid;
           `w PASS_3_AXIS_tvalid  = IN_3_AXIS_tvalid;
           `w PASS_4_AXIS_tvalid  = IN_4_AXIS_tvalid;
           `w PASS_5_AXIS_tvalid  = IN_5_AXIS_tvalid;
           `w PASS_6_AXIS_tvalid  = IN_6_AXIS_tvalid;
           `w PASS_7_AXIS_tvalid  = IN_7_AXIS_tvalid;
           `w PASS_8_AXIS_tvalid  = IN_8_AXIS_tvalid;
           `w PASS_9_AXIS_tvalid  = IN_9_AXIS_tvalid;
           `w PASS_10_AXIS_tvalid = IN_10_AXIS_tvalid;
           `w PASS_11_AXIS_tvalid = IN_11_AXIS_tvalid;
           `w PASS_12_AXIS_tvalid = IN_12_AXIS_tvalid;
           `w PASS_13_AXIS_tvalid = IN_13_AXIS_tvalid;
           `w PASS_14_AXIS_tvalid = IN_14_AXIS_tvalid;
           `w PASS_15_AXIS_tvalid = IN_15_AXIS_tvalid;

        end
   endgenerate
     
   //
   // Registers coming from other clock domain
   //
   // Normally only two registers are needed to avoid metastability.  Here we have a third register, to make
   // sure that the register can be replicated if necessary to meet timing.  (probably unnecessary.)
   
   /**/                                                                          reg [3:0]  sel_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [3:0]  sel_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [3:0]  sel_srr;
   (* max_fanout = 10 *)                                                         reg [3:0]  sel_srrr;

   /**/                                                                          reg        collect_ar;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg        collect_sr;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg        collect_srr;
   /**/                                                                          reg        collect_srrr;

   `raxis  collect_sr    <=  collect_ar;      // collect_ar is on the AXI clock.
   `raxis  collect_srr   <=  collect_sr;
   `raxis  collect_srrr  <=  collect_srr;
   `w      collect        =  collect_srrr;

   `raxis  sel_sr        <=  sel_ar;          // sel_ar is on the AXI clock
   `raxis  sel_srr       <=  sel_sr;
   `raxis  sel_srrr      <=  sel_srr;
   
   // Never assert (or support) flow control.
   `w IN_0_AXIS_tready = 1'b1;
   `w IN_1_AXIS_tready = 1'b1;
   `w IN_2_AXIS_tready = 1'b1;
   `w IN_3_AXIS_tready = 1'b1;
   `w IN_4_AXIS_tready = 1'b1;
   `w IN_5_AXIS_tready = 1'b1;
   `w IN_6_AXIS_tready = 1'b1;
   `w IN_7_AXIS_tready = 1'b1;

   `raxis OUT_AXIS_tdata   <=  (sel_srrr==0)   ?   IN_0_AXIS_tdata   :   
                               (sel_srrr==1)   ?   IN_1_AXIS_tdata   :
                               (sel_srrr==2)   ?   IN_2_AXIS_tdata   :
                               (sel_srrr==3)   ?   IN_3_AXIS_tdata   :
                               (sel_srrr==4)   ?   IN_4_AXIS_tdata   :
                               (sel_srrr==5)   ?   IN_5_AXIS_tdata   :
                               (sel_srrr==6)   ?   IN_6_AXIS_tdata   :
                               (sel_srrr==7)   ?   IN_7_AXIS_tdata   :
                               (sel_srrr==8)   ?   IN_8_AXIS_tdata   :   
                               (sel_srrr==9)   ?   IN_9_AXIS_tdata   :
                               (sel_srrr==10)  ?  IN_10_AXIS_tdata   :
                               (sel_srrr==11)  ?  IN_11_AXIS_tdata   :
                               (sel_srrr==12)  ?  IN_12_AXIS_tdata   :
                               (sel_srrr==13)  ?  IN_13_AXIS_tdata   :
                               (sel_srrr==14)  ?  IN_14_AXIS_tdata   :
                               /**/               IN_15_AXIS_tdata   ;


   `raxis OUT_AXIS_tvalid   <=  (sel_srrr==0)   ?   IN_0_AXIS_tvalid   :   
                                (sel_srrr==1)   ?   IN_1_AXIS_tvalid   :
                                (sel_srrr==2)   ?   IN_2_AXIS_tvalid   :
                                (sel_srrr==3)   ?   IN_3_AXIS_tvalid   :
                                (sel_srrr==4)   ?   IN_4_AXIS_tvalid   :
                                (sel_srrr==5)   ?   IN_5_AXIS_tvalid   :
                                (sel_srrr==6)   ?   IN_6_AXIS_tvalid   :
                                (sel_srrr==7)   ?   IN_7_AXIS_tvalid   :
                                (sel_srrr==8)   ?   IN_8_AXIS_tvalid   :   
                                (sel_srrr==9)   ?   IN_9_AXIS_tvalid   :
                                (sel_srrr==10)  ?  IN_10_AXIS_tvalid   :
                                (sel_srrr==11)  ?  IN_11_AXIS_tvalid   :
                                (sel_srrr==12)  ?  IN_12_AXIS_tvalid   :
                                (sel_srrr==13)  ?  IN_13_AXIS_tvalid   :
                                (sel_srrr==14)  ?  IN_14_AXIS_tvalid   :
                                /**/               IN_15_AXIS_tvalid   ;

   
   reg [31:0]                              reg_mux_select_ar;              // Register 0
   reg [31:0]                              reg_capture_enable_ar;          // Register 1


   //
   // Send registers to AXIS clock domain
   //
   `raxi  sel_ar      <= reg_mux_select_ar[3:0];
   `raxi  collect_ar  <= reg_capture_enable_ar[0];

   //
   // Handle READs/WRITEs from the processor on the AXI bus.
   //    

   localparam                              WORD_ADDR_BITS         = $clog2(NUM_REGISTERS);   
   localparam                              BYTE_ADDR_BITS_IN_WORD = $clog2(AXI_BYTES_PER_WORD);
   
   reg [WORD_ADDR_BITS-1 : 0]              write_address;
   reg [AXI_BITS_PER_WORD-1:0]             write_data;
   wire                                    write_enable;
   wire                                    write_enable_w;
   reg                                     write_enable_r;
   wire                                    write_done;
   
   reg                                     read_enable;
   reg [WORD_ADDR_BITS-1 : 0]              read_address;
   wire [AXI_BITS_PER_WORD-1:0]            read_data;
   wire                                    read_done;

   //
   // The write interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_AWREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_AWREADY && S_AXI_AWVALID)   ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_AWREADY  ;

   `raxi S_AXI_WREADY  <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_WREADY && S_AXI_WVALID)     ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_WREADY   ;

   `raxi S_AXI_BVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          write_done                         ?  1'b1           :
                          (S_AXI_BREADY && S_AXI_BVALID)     ?  1'b0           :
                          /* default */                         S_AXI_BVALID   ;

   `raxi S_AXI_BRESP   <= 2'b0;

   //
   // The read interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_ARREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_ARREADY && S_AXI_ARVALID)   ?  1'b0           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b1           :
                          /* default */                         S_AXI_ARREADY  ;
   

   `raxi S_AXI_RVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          read_done                          ?  1'b1           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b0           :
                          /* default */                         S_AXI_RVALID   ;

   `raxi S_AXI_RRESP   <= 2'b0;
   
   `raxi S_AXI_RDATA   <= (!S_AXI_ARESETN)                   ?  0              : 
                          read_done                          ?  read_data      :
                          /* default */                         S_AXI_RDATA    ;


   //
   // Define write addresses, data, and enables
   //
   `w write_enable_w          = !S_AXI_WREADY && !S_AXI_AWREADY;
   `raxi write_enable_r       = (!S_AXI_ARESETN) ? 0 : write_enable_w;
   `w write_enable            = write_enable_w && !write_enable_r;   // Only pulse write enable for a single clock, regardless of BREADY
   
   `raxi write_address       <= (!S_AXI_ARESETN)                  ?  0                                                              :
                                (S_AXI_AWREADY && S_AXI_AWVALID)  ?  S_AXI_AWADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                        write_address                                                  ;
   
   `raxi write_data          <= (!S_AXI_ARESETN)                  ?  0                :
                                (S_AXI_WREADY && S_AXI_WVALID)    ?  S_AXI_WDATA      :
                                /* default */                        write_data       ;

   `w write_done              = write_enable;  // Write done immediately after address and data are both present
   

   //
   // Define read addresses, data, and enables
   //
   `raxi read_address        <= (!S_AXI_ARESETN)               ?  0                                                              : 
                                S_AXI_ARREADY & S_AXI_ARVALID  ?  S_AXI_ARADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                     read_address                                                   ;
   
   `raxi read_enable         <= S_AXI_ARREADY & S_AXI_ARVALID;

   `w read_done               = read_enable;  // Read data available immediately after address is present.
   
   // read_data  is assigned a value below



   //
   // Handle read data.  For this module we don't have any data to read on the AXIS clock, so we don't need to
   // handle any clock transitions here.
   //
      
   `w read_data = (read_address==0)  ?  reg_mux_select_ar             :
                  (read_address==1)  ?  reg_capture_enable_ar         :
                  /* default */         0                             ;

   
   //
   // Handle write data.  Registers write on the AXI clock.
   // None of the other registers are supposed to be written unless reg_trigger_arm_ar is 0.
   // Thus they won't be written at a critical time and thus metastability isn't an issue.
   //
   
   `raxi reg_mux_select_ar              <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==0)     ?  write_data                     :
                                            /* default */                             reg_mux_select_ar              ;

   `raxi reg_capture_enable_ar          <=  (!S_AXI_ARESETN)                       ?  0                              :
                                            write_enable && (write_address==1)     ?  write_data                     :
                                            /* default */                             reg_capture_enable_ar          ;
  
   

endmodule
