

const char* copyright_info =
  "\n"
  "Copyright (C) 2025 Bit By Bit Signal Processing LLC\n"
  "\n"
  "* This program is free software; you can redistribute it and/or\n"
  "* modify it under the terms of the GNU General Public License as\n"
  "* published by the Free Software Foundation; either version 2 of\n"
  "* the License, or (at your option) any later version.\n"
  "*\n"
  "* This program is distributed in the hope that it will be useful,\n"
  "* but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
  "* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
  "* GNU General Public License for more details.\n"
  "*\n"
  "* You should have received a copy of the GNU General Public License\n"
  "* along with this program; if not, write to the Free Software\n"
  "* Foundation, Inc., http://www.fsf.org/about/contact/\n"
  "\n";

typedef unsigned int u32;

void Xil_Out32(u32 addr, u32 value)
{
  *(u32*)( ((u8*)0)+addr) = value;
}

u32 Xil_In32(u32 addr)
{
  return *(u32*)( ((u8*)0)+addr);
}

u32 mask_read(u32 add, u32 mask);
void PSU_Mask_Write(unsigned long offset, unsigned long mask, unsigned long val);

struct mio_ctrl
{
  u32 iou;
  u32 gpio;
};

typedef struct mio_ctrl mio_ctrl;


int  mio_ctrl_get_MIO_value(mio_ctrl* mio, int mio_number);
void mio_ctrl_set_MIO_value(mio_ctrl* mio, int mio_number, int value);

void mio_ctrl_set_MIO_to_GPIO_input(mio_ctrl* mio, int mio_number);
void mio_ctrl_set_MIO_to_GPIO_output(mio_ctrl* mio, int mio_number);

void mio_ctrl_set_MIO_config(mio_ctrl* mio, int mio_number, u32 value);
u32  mio_ctrl_get_MIO_config(mio_ctrl* mio, int mio_bumber);


void mio_ctrl_set_MIO_config(mio_ctrl* mio, int mio_number, u32 value)
{
  Xil_Out32(mio->iou+4*mio_number, value);
}

u32  mio_ctrl_get_MIO_config(mio_ctrl* mio, int mio_number)
{
  u32 data = Xil_In32(mio->iou+4*mio_number);
  return data;
}

void mio_ctrl_new(mio_ctrl* mio);


typedef unsigned char u8;


//const static int true  = 1;
//const static int false = 0;


struct newhaven_display
{
  struct mio_ctrl* mio;

  char line_1[16];
  char line_2[16];
    
  int MIO_CLK;
  int MIO_CSB;
  int MIO_DATA_OUT;
  int MIO_DATA_IN;

  u32 MIO_CLK_CTRL_REG_VALUE;
  u32 MIO_CSB_CTRL_REG_VALUE;
  u32 MIO_DATA_OUT_CTRL_REG_VALUE;
  u32 MIO_DATA_IN_CTRL_REG_VALUE;
};


typedef struct newhaven_display newhaven_display;


void newhaven_display_CSB_HIGH(newhaven_display* nhd)       { mio_ctrl_set_MIO_value(nhd->mio, nhd->MIO_CSB, 1);           }
void newhaven_display_CSB_LOW(newhaven_display* nhd)        { mio_ctrl_set_MIO_value(nhd->mio, nhd->MIO_CSB, 0);           }

void newhaven_display_CLK_HIGH(newhaven_display* nhd)       { mio_ctrl_set_MIO_value(nhd->mio, nhd->MIO_CLK, 1);           }
void newhaven_display_CLK_LOW(newhaven_display* nhd)        { mio_ctrl_set_MIO_value(nhd->mio, nhd->MIO_CLK, 0);           }

void newhaven_display_DATA_OUT_HIGH(newhaven_display* nhd)  { mio_ctrl_set_MIO_value(nhd->mio, nhd->MIO_DATA_OUT, 1);      }
void newhaven_display_DATA_OUT_LOW(newhaven_display* nhd)   { mio_ctrl_set_MIO_value(nhd->mio, nhd->MIO_DATA_OUT, 0);      }

int  newhaven_display_DATA_BIT_IN(newhaven_display* nhd)    { return mio_ctrl_get_MIO_value(nhd->mio, nhd->MIO_DATA_IN);   }

void empty_function(void);

void newhaven_display_DELAY(newhaven_display* nhd)          { for(int i=0; i<10000; i++) empty_function(); }

void empty_function() { }


void newhaven_display_set_line_1(newhaven_display* nhd, const char* txt, int centered);
void newhaven_display_set_line_2(newhaven_display* nhd, const char* txt, int centered);
void newhaven_display_draw(newhaven_display* nhd);
void newhaven_display_clear(newhaven_display* nhd);
  
void newhaven_display_home_1(newhaven_display* nhd);
void newhaven_display_home_2(newhaven_display* nhd);

void newhaven_display_write (newhaven_display* nhd, u8 data, int command);
u8   newhaven_display_read  (newhaven_display* nhd, int command);




void newhaven_display_write_data(newhaven_display* nhd, u8 data)
{
  newhaven_display_write(nhd, data, false);
}

void newhaven_display_write_command(newhaven_display* nhd, u8 data)
{
  newhaven_display_write(nhd, data, true);
}

u8 newhaven_display_read_data(newhaven_display* nhd)
{
  return newhaven_display_read(nhd, false);
}

u8 newhaven_display_read_command(newhaven_display* nhd)
{
  return newhaven_display_read(nhd, true);
}

void newhaven_display_delete(newhaven_display* nhd)
{
  mio_ctrl_set_MIO_config ( nhd->mio, nhd->MIO_CLK,      nhd->MIO_CLK_CTRL_REG_VALUE );
  mio_ctrl_set_MIO_config ( nhd->mio, nhd->MIO_CSB,      nhd->MIO_CSB_CTRL_REG_VALUE );
  mio_ctrl_set_MIO_config ( nhd->mio, nhd->MIO_DATA_OUT, nhd->MIO_DATA_OUT_CTRL_REG_VALUE );
  mio_ctrl_set_MIO_config ( nhd->mio, nhd->MIO_DATA_IN,  nhd->MIO_DATA_IN_CTRL_REG_VALUE );
}

void newhaven_display_new(newhaven_display* nhd,
			  mio_ctrl*         mio,
			  int               MIO_CLK,
			  int               MIO_CSB,
			  int               MIO_DATA_OUT,
			  int               MIO_DATA_IN)
{
  nhd->mio             = mio;
  nhd->MIO_CLK         = MIO_CLK;
  nhd->MIO_CSB         = MIO_CSB;
  nhd->MIO_DATA_OUT    = MIO_DATA_OUT;
  nhd->MIO_DATA_IN     = MIO_DATA_IN;
  
  newhaven_display_CSB_HIGH(nhd);
  newhaven_display_CLK_LOW(nhd);
  newhaven_display_DATA_OUT_LOW(nhd);
  
  nhd->MIO_CLK_CTRL_REG_VALUE       = mio_ctrl_get_MIO_config ( mio, MIO_CLK          );
  nhd->MIO_CSB_CTRL_REG_VALUE       = mio_ctrl_get_MIO_config ( mio, MIO_CSB          );
  nhd->MIO_DATA_OUT_CTRL_REG_VALUE  = mio_ctrl_get_MIO_config ( mio, MIO_DATA_OUT     );
  nhd->MIO_DATA_IN_CTRL_REG_VALUE   = mio_ctrl_get_MIO_config ( mio, MIO_DATA_IN      );

  mio_ctrl_set_MIO_to_GPIO_output ( mio, MIO_CLK          );
  mio_ctrl_set_MIO_to_GPIO_output ( mio, MIO_CSB          );
  mio_ctrl_set_MIO_to_GPIO_output ( mio, MIO_DATA_OUT     );
  mio_ctrl_set_MIO_to_GPIO_input  ( mio, MIO_DATA_IN      );
  
  newhaven_display_write_command(nhd, 0x2A);
  newhaven_display_write_command(nhd, 0x71);
  newhaven_display_write_data(nhd, 0x00);
  newhaven_display_write_command(nhd, 0x28);
  newhaven_display_write_command(nhd, 0x08);
  newhaven_display_write_command(nhd, 0x2A);
  newhaven_display_write_command(nhd, 0x79);
  newhaven_display_write_command(nhd, 0xD5);
  newhaven_display_write_command(nhd, 0x70);
  newhaven_display_write_command(nhd, 0x78);
  newhaven_display_write_command(nhd, 0x08);
  newhaven_display_write_command(nhd, 0x06);
  newhaven_display_write_command(nhd, 0x72);
  newhaven_display_write_data(nhd, 0x00);
  newhaven_display_write_command(nhd, 0x2A);
  newhaven_display_write_command(nhd, 0x79);
  newhaven_display_write_command(nhd, 0xDA);
  newhaven_display_write_command(nhd, 0x00);
  newhaven_display_write_command(nhd, 0xDC);
  newhaven_display_write_command(nhd, 0x00);
  newhaven_display_write_command(nhd, 0x81);
  newhaven_display_write_command(nhd, 0x7F);
  newhaven_display_write_command(nhd, 0xD9);
  newhaven_display_write_command(nhd, 0xF1);
  newhaven_display_write_command(nhd, 0xDB);
  newhaven_display_write_command(nhd, 0x40);
  newhaven_display_write_command(nhd, 0x78);
  newhaven_display_write_command(nhd, 0x28);
  newhaven_display_write_command(nhd, 0x01);
  newhaven_display_write_command(nhd, 0x80);
  newhaven_display_write_command(nhd, 0x0C);

  for(int i=0; i<16; i++)
    {
      nhd->line_1[i] = ' ';
      nhd->line_2[i] = ' ';
    }
}


void newhaven_display_write(newhaven_display* nhd, u8 data, int command)
{  
  u32 word = ( (((u32)0x1F)<<19)     +
	       0 +                    
	       ((command?0:1) << 17) +
	       0 +                    
	       (((data>>0)&1)<<15) +  
	       (((data>>1)&1)<<14) +
	       (((data>>2)&1)<<13) +
	       (((data>>3)&1)<<12) +
	       0 +                    
	       (((data>>4)&1)<<7)  +  
	       (((data>>5)&1)<<6)  +
	       (((data>>6)&1)<<5)  +
	       (((data>>7)&1)<<4)  +
	       0);                    

  newhaven_display_CLK_LOW(nhd);
  newhaven_display_DELAY(nhd);
  newhaven_display_CSB_LOW(nhd);
  newhaven_display_DELAY(nhd);

  for(int i=23; i>=0; i--)
    {
      int bit = (word>>i) & 1;

      if(bit)
	newhaven_display_DATA_OUT_HIGH(nhd);
      else
	newhaven_display_DATA_OUT_LOW(nhd);

      newhaven_display_DELAY(nhd);
      newhaven_display_CLK_HIGH(nhd);
      newhaven_display_DELAY(nhd);
      newhaven_display_CLK_LOW(nhd);
    }

  newhaven_display_DELAY(nhd);
  newhaven_display_CSB_HIGH(nhd);
  newhaven_display_DELAY(nhd);
}




u8 newhaven_display_read(newhaven_display* nhd, int command)
{
  int preamble = command ? 0xFC : 0xFE;

  u8 data = 0;
  
  newhaven_display_CLK_LOW(nhd);
  newhaven_display_DELAY(nhd);
  newhaven_display_CSB_LOW(nhd);
  newhaven_display_DELAY(nhd);

  for(int i=7; i>=0; i--)
    {
      int bit = (preamble>>i) & 1;

      if(bit)
	newhaven_display_DATA_OUT_HIGH(nhd);
      else
	newhaven_display_DATA_OUT_LOW(nhd);

      newhaven_display_DELAY(nhd);
      newhaven_display_CLK_HIGH(nhd);
      newhaven_display_DELAY(nhd);
      newhaven_display_CLK_LOW(nhd);
    }

  for(int i=0; i<8; i++)
    {
      newhaven_display_DELAY(nhd);
      
      int bit = newhaven_display_DATA_BIT_IN(nhd);
      data += (bit<<i);

      newhaven_display_CLK_HIGH(nhd);
      newhaven_display_DELAY(nhd);
      newhaven_display_CLK_LOW(nhd);
    }

  newhaven_display_DELAY(nhd);
  newhaven_display_CSB_HIGH(nhd);
  newhaven_display_DELAY(nhd);
  
  return data;
}


void newhaven_display_home_1(newhaven_display* nhd)
{
  newhaven_display_write_command(nhd, 0x80 + 00);
}

void newhaven_display_home_2(newhaven_display* nhd)
{
  newhaven_display_write_command(nhd, 0x80 + 40);
}

void newhaven_display_draw(newhaven_display* nhd)
{
  newhaven_display_home_1(nhd);
  for(int i=0;i<16;i++)
    {
      newhaven_display_write_data(nhd, nhd->line_1[i]);
    }

  newhaven_display_home_2(nhd);
  for(int i=0;i<16;i++)
    {
      newhaven_display_write_data(nhd, nhd->line_2[i]);
    }
}

void newhaven_display_clear(newhaven_display* nhd)
{
  newhaven_display_write_command(nhd, 0x01);
}

void newhaven_display_set_line_1(newhaven_display* nhd, const char* txt, int centered)
{
  int len = 0;
  while(txt[len] && len<16)
    len++;
  
  if(len>16)
    len=16;
  
  int extra_left  = centered ? (16-len)/2           : 0;
  int extra_right = centered ? 16-len-extra_left    : 16 - len;

  int j = 0;
  for(int i=0; i<extra_left; i++)
    nhd->line_1[j++] = ' ';

  for(int i=0; i<len; i++)
    nhd->line_1[j++] = txt[i];

  for(int i=0; i<extra_right; i++)
    nhd->line_1[j++] = ' ';
}


void newhaven_display_set_line_2(newhaven_display* nhd, const char* txt, int centered)
{
  int len = 0;
  while(txt[len] && len<16)
    len++;
  
  if(len>16)
    len=16;
  
  int extra_left  = centered ? (16-len)/2           : 0;
  int extra_right = centered ? 16-len-extra_left    : 16 - len;

  int j = 0;
  for(int i=0; i<extra_left; i++)
    nhd->line_2[j++] = ' ';

  for(int i=0; i<len; i++)
    nhd->line_2[j++] = txt[i];

  for(int i=0; i<extra_right; i++)
    nhd->line_2[j++] = ' ';
}


#define IOU_SLCR_BASE_ADDRESS 0x00FF180000
#define NUM_MIO_REGISTERS     78

#define IOU_SLCR_L3_SEL_MASK          (7<<5)
#define IOU_SLCR_L3_SEL_GPIO          (0<<5)
#define IOU_SLCR_L3_SEL_CAN1          (1<<5)
#define IOU_SLCR_L3_SEL_I2C1          (2<<5)
#define IOU_SLCR_L3_SEL_PJTAG         (3<<5)
#define IOU_SLCR_L3_SEL_SPI0          (4<<5)
#define IOU_SLCR_L3_SEL_TTC3          (5<<5)
#define IOU_SLCR_L3_SEL_UART1         (6<<5)
#define IOU_SLCR_L3_SEL_TracePortClk  (7<<5)

#define IOU_SLCR_L2_SEL_MASK          (3<<3)
#define IOU_SLCR_L2_SEL_L3            (0<<3)
#define IOU_SLCR_L2_SEL_RESERVED_1    (1<<3)
#define IOU_SLCR_L2_SEL_Scan_Test     (2<<3)
#define IOU_SLCR_L2_SEL_RESERVED_3    (3<<3)

#define IOU_SLCR_L1_SEL_MASK          (1<<2)
#define IOU_SLCR_L1_SEL_L2            (0<<2)
#define IOU_SLCR_L1_SEL_RESERVED_1    (1<<2)

#define IOU_SLCR_L0_SEL_MASK          (1<<1)
#define IOU_SLCR_L0_SEL_L1            (0<<1)
#define IOU_SLCR_L0_SEL_QUAD_SPIO     (1<<1)

#define IOU_SLCR_RESERVED_MASK        (0xFFFFFF01)

#define IOU_SLCR_GPIO (IOU_SLCR_L3_SEL_GPIO | IOU_SLCR_L2_SEL_L3 | IOU_SLCR_L1_SEL_L2 | IOU_SLCR_L0_SEL_L1 )


#define GPIO_BASE_ADDRESS  0x00FF0A0000


#define MASK_DATA_0_LSW	0x0000000000	
#define MASK_DATA_0_MSW	0x0000000004	
#define MASK_DATA_1_LSW	0x0000000008	
#define MASK_DATA_1_MSW	0x000000000C	
#define MASK_DATA_2_LSW	0x0000000010	
#define MASK_DATA_2_MSW	0x0000000014	
#define MASK_DATA_3_LSW	0x0000000018	
#define MASK_DATA_3_MSW	0x000000001C	
#define MASK_DATA_4_LSW	0x0000000020	
#define MASK_DATA_4_MSW	0x0000000024	
#define MASK_DATA_5_LSW	0x0000000028	
#define MASK_DATA_5_MSW	0x000000002C	

#define DATA_0  	0x0000000040	
#define DATA_1  	0x0000000044	
#define DATA_2  	0x0000000048	
#define DATA_3  	0x000000004C	
#define DATA_4  	0x0000000050	
#define DATA_5  	0x0000000054	

#define DATA_0_RO	0x0000000060	
#define DATA_1_RO	0x0000000064	
#define DATA_2_RO	0x0000000068	
#define DATA_3_RO	0x000000006C	
#define DATA_4_RO	0x0000000070	
#define DATA_5_RO	0x0000000074	

#define DIRM_0  	0x0000000204	
#define OEN_0   	0x0000000208	
#define INT_MASK_0	0x000000020C	
#define INT_EN_0	0x0000000210	
#define INT_DIS_0	0x0000000214	
#define INT_STAT_0	0x0000000218	
#define INT_TYPE_0	0x000000021C	
#define INT_POLARITY_0	0x0000000220	
#define INT_ANY_0	0x0000000224	

#define DIRM_1   	0x0000000244	
#define OEN_1   	0x0000000248	
#define INT_MASK_1	0x000000024C	
#define INT_EN_1	0x0000000250	
#define INT_DIS_1	0x0000000254	
#define INT_STAT_1	0x0000000258	
#define INT_TYPE_1	0x000000025C	
#define INT_POLARITY_1	0x0000000260	
#define INT_ANY_1	0x0000000264	

#define DIRM_2   	0x0000000284	
#define OEN_2   	0x0000000288	
#define INT_MASK_2	0x000000028C	
#define INT_EN_2	0x0000000290	
#define INT_DIS_2	0x0000000294	
#define INT_STAT_2	0x0000000298	
#define INT_TYPE_2	0x000000029C	
#define INT_POLARITY_2	0x00000002A0	
#define INT_ANY_2	0x00000002A4	

#define DIRM_3   	0x00000002C4	
#define OEN_3   	0x00000002C8	
#define INT_MASK_3	0x00000002CC	
#define INT_EN_3	0x00000002D0	
#define INT_DIS_3	0x00000002D4	
#define INT_STAT_3	0x00000002D8	
#define INT_TYPE_3	0x00000002DC	
#define INT_POLARITY_3	0x00000002E0	
#define INT_ANY_3	0x00000002E4	

#define DIRM_4   	0x0000000304	
#define OEN_4   	0x0000000308	
#define INT_MASK_4	0x000000030C	
#define INT_EN_4	0x0000000310	
#define INT_DIS_4	0x0000000314	
#define INT_STAT_4	0x0000000318	
#define INT_TYPE_4	0x000000031C	
#define INT_POLARITY_4	0x0000000320	
#define INT_ANY_4	0x0000000324	

#define DIRM_5   	0x0000000344	
#define OEN_5   	0x0000000348	
#define INT_MASK_5	0x000000034C	
#define INT_EN_5	0x0000000350	
#define INT_DIS_5	0x0000000354	
#define INT_STAT_5	0x0000000358	
#define INT_TYPE_5	0x000000035C	
#define INT_POLARITY_5	0x0000000360	
#define INT_ANY_5	0x0000000364	

#define NUM_GPIO_REGISTERS ((INT_ANY_5 + 4) / 4)





void util_clear_bit(u32 regaddr, int bit)
{
  Xil_Out32(regaddr, Xil_In32(regaddr) &~ (((u32)1)<<bit));
}

void util_set_bit(u32 regaddr, int bit)
{
  Xil_Out32(regaddr, Xil_In32(regaddr) | (((u32)1)<<bit));
}

int util_get_bit(u32 regaddr, int bit)
{
  return (Xil_In32(regaddr) >> bit ) & 1;
}








void mio_ctrl_new(mio_ctrl* mio)
{
  mio->iou  = IOU_SLCR_BASE_ADDRESS; 
  mio->gpio = GPIO_BASE_ADDRESS;     
}


int mio_ctrl_get_MIO_value(mio_ctrl* mio, int mio_number)
{
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;

  u32 data_reg = ( (bank_number==0) ? DATA_0_RO :
		   (bank_number==1) ? DATA_1_RO :
		   /**/               DATA_2_RO );
	  
  return util_get_bit(mio->gpio+data_reg, bit_number);
}


void mio_ctrl_set_MIO_value(mio_ctrl* mio, int mio_number, int value)
{
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;

  u32 data_reg = ( (bank_number==0) ? DATA_0 :
		    (bank_number==1) ? DATA_1 :
		    /**/               DATA_2 );
  
  if(value)
    util_set_bit(mio->gpio+data_reg, bit_number);
  else
    util_clear_bit(mio->gpio+data_reg, bit_number);
}


void mio_ctrl_set_MIO_to_GPIO_input(mio_ctrl* mio, int mio_number)
{
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;


  u32 output_direction_reg = ( (bank_number==0) ? DIRM_0 :
			        (bank_number==1) ? DIRM_1 :
			        /**/               DIRM_2 );
  
  u32 output_enable_reg = ( (bank_number==0) ? OEN_0 :
			     (bank_number==1) ? OEN_1 :
			     /**/               OEN_2 );

  u32 interrupt_disable_reg = ( (bank_number==0) ? INT_DIS_0 :
	 			 (bank_number==1) ? INT_DIS_1 :
				 /**/               INT_DIS_2 );
  
  util_clear_bit(mio->gpio+output_direction_reg, bit_number);
  util_clear_bit(mio->gpio+output_enable_reg, bit_number);

  Xil_Out32(mio->gpio+interrupt_disable_reg, (1<<bit_number));
  
  Xil_Out32(mio->iou+4*mio_number, IOU_SLCR_GPIO);
}


void mio_ctrl_set_MIO_to_GPIO_output(mio_ctrl* mio, int mio_number)
{
  int bank_number = mio_number/26;
  int bit_number = mio_number%26;

  u32 output_direction_reg = ( (bank_number==0) ? DIRM_0 :
			       (bank_number==1) ? DIRM_1 :
			       /**/               DIRM_2 );
  
  u32 output_enable_reg = ( (bank_number==0) ? OEN_0 :
			    (bank_number==1) ? OEN_1 :
			    /**/               OEN_2 );
  
  u32 interrupt_disable_reg = ( (bank_number==0) ? INT_DIS_0 :
				(bank_number==1) ? INT_DIS_1 :
				/**/               INT_DIS_2 );
  
  util_set_bit(mio->gpio+output_direction_reg, bit_number);
  util_set_bit(mio->gpio+output_enable_reg, bit_number);

  Xil_Out32(mio->gpio+interrupt_disable_reg, (1<<bit_number));
  
  Xil_Out32(mio->iou+4*mio_number, IOU_SLCR_GPIO);
}


int show_board_info(void)
{
  printf("\n\r"
	 "*******************************************************************************\n\r"
	 "**\n\r"
	 "**      UBOOT for STYNQ RFSoC4x2.\n\r"
	 "**\n\r"
	 "**      Available at https://STYNQ.com\n\r"
	 "**\n\r"
	 "**      Packaged for you by Bit by Bit Signal Processing LLC\n\r"
	 "**      https://bxbsp.com\n\r"
	 "**\n\r"
	 "*******************************************************************************\n\r"
	 "\n\r");
  
  int MIO_CLK      = 6;
  int MIO_CSB      = 9;
  int MIO_DATA_IN  = 10;
  int MIO_DATA_OUT = 11;
  
  mio_ctrl         mio;
  newhaven_display nhd;
  
  mio_ctrl_new(&mio);
  newhaven_display_new(&nhd, &mio, MIO_CLK, MIO_CSB, MIO_DATA_OUT, MIO_DATA_IN);
  
  newhaven_display_set_line_1(&nhd, "  STYNQ UBOOT   ", false);
  newhaven_display_set_line_2(&nhd, "http://STYNQ.com", false);
  
  newhaven_display_draw(&nhd);
  
  //newhaven_display_delete(&nhd);
  
  return checkboard();
}


void board_quiesce_devices(void)
{
  int MIO_CLK      = 6;
  int MIO_CSB      = 9;
  int MIO_DATA_IN  = 10;
  int MIO_DATA_OUT = 11;
  
  mio_ctrl         mio;
  newhaven_display nhd;
  
  mio_ctrl_new(&mio);
  newhaven_display_new(&nhd, &mio, MIO_CLK, MIO_CSB, MIO_DATA_OUT, MIO_DATA_IN);
  
  newhaven_display_set_line_1(&nhd, " Starting Linux ", false);
  newhaven_display_set_line_2(&nhd, "http://STYNQ.com", false);
  
  newhaven_display_draw(&nhd);
}
