#!/bin/bash

DTB="$1"

OUTDIR=out

mkdir -p "$OUTDIR"

DESIGN=`basename "$DTB" | sed 's/[.].*//'`
UBOOT="$OUTDIR/${DESIGN}_U_BOOT.elf"

if [ "$UBOOT" -nt "$DTB" ]; then
    echo ""
    echo "  U-BOOT outputs for $DESIGN are newer than inputs.  Skipping"
    echo "  rebuild.  To force rebuild, run veryclean.sh first."
    echo ""
    exit 0
fi

if [ ! -f "$DTB" ]; then

    DTB_PATH=../device_tree_gen/out
    
    DTB="$DTB_PATH/$DTB.dtb"

    if [ ! -f "$DTB" ]; then
    
	echo ""
	echo "  File \"$DTB\" doesn't exist to support board \"$DESIGN\"."
	echo ""
	echo "Instead try one of these boards:"
	echo ""
	for file in "$DTB_PATH"/*.dtb; do
	    
	    name=`echo $file | sed 's@.*/@@; s/[.]dtb//'`
	    
	    echo "   $name"
	    
	done
	echo ""
	exit 20
    fi
fi

    
rm -rf u-boot-xlnx
rm -f u-boot/${BOARD}-u-boot.elf

#sudo apt install uuid-dev
#sudo apt-get install libgnutls28-dev
#sudo apt install device-tree-overlay



COMPILER=`which aarch64-linux-gnu-gcc`

if [ ! -f "$COMPILER" ]; then
    echo "Need to install compiler aarch64-linux-gnu-gcc"
    sudo apt install gcc-aarch64-linux-gnu
fi




if [ -f u-boot-xlnx.tgz ]; then
    echo "Extracting pre-existing u-boot-xlnx.tgz."
    tar xzf u-boot-xlnx.tgz
else
    echo "Fetching u-boot-xlnx."
    git clone https://github.com/Xilinx/u-boot-xlnx.git
    tar czf u-boot-xlnx.tgz u-boot-xlnx
fi


if [ -f /proc/cpuinfo -a -f /proc/loadavg ]; then
    CPUs_minus_1=`cat /proc/cpuinfo | grep processor | tail -1 | sed 's/.*://'`
else
    CPUs_minus_1=0
fi

CPUs=$((CPUs_minus_1 + 1))

MAXJOBS=$(( 3 * CPUs / 4 - 1 ))

if [ $MAXJOBS -lt 1 ]; then
   MAXJOBS=1
fi

echo "Will run parallel build with $MAXJOBS threads."


export CROSS_COMPILE=aarch64-linux-gnu-
export ARCH=aarch64

#
# The build needs the device tree of the board.  Put it in
# the expected place, and then configure for it.
#
cp "$DTB" u-boot-xlnx/arch/arm/dts/
export DEVICE_TREE="$DESIGN"

cp inputs/my_zynqmp_defconfig u-boot-xlnx/configs/

if [ -f "inputs/${DESIGN}_UBOOT_Splash.c" ]; then
    echo "WARNING:"
    echo "Tainting GPL code with non-GPL modifications."
    echo "This uboot binary can't be redistributed."
    echo "Distribution is only in source code form"
    echo "with the differently-licensed parts separated."
    echo "Delete inputs/*_Splash.c and rebuild if you"
    echo "want an untainted build."
    cat "inputs/${DESIGN}_UBOOT_Splash.c" >>u-boot-xlnx/common/main.c
fi


PATH="$PWD/exe:$PATH"

(cd u-boot-xlnx; make -j$MAXJOBS my_zynqmp_defconfig)
(cd u-boot-xlnx; make -j$MAXJOBS)


#mkdir -p exe
#cp u-boot-xlnx/tools/mkimage    exe/
#cp u-boot-xlnx/tools/dumpimage  exe/
#cp u-boot-xlnx/scripts/dtc/dtc  exe/

cp u-boot-xlnx/u-boot.elf  out/${DESIGN}_U_BOOT.elf

${CROSS_COMPILE}strip out/${DESIGN}_U_BOOT.elf


echo ""
echo ""
echo "To delete temporary files, run clean.sh."
echo ""
echo "Executable outputs are in the exe directory."

if [ -f "out/${DESIGN}_U_BOOT.elf" ]; then
    echo ""
    echo "new u-boot.elf is in out/${DESIGN}_U_BOOT.elf"
    echo ""
else
    echo ""
    echo "ERROR: new out/${DESIGN}_U_BOOT.elf is missing."
    echo ""
fi    

