
// SPDX-License-Identifier: CC-BY-NC-ND-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial NoDerivatives 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-ND-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-ND-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  This work may be distributed in unmodified form,
// but derivatives of this work may not be distributed.  For further
// details, see the Creative Commons License "CC-BY-NC-ND-4.0".
//
// You should have received a copy of the CC-BY-NC-ND-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-nd/4.0/>.
//


#include "xml.hh"
#include "parse_support.hh"
#include <time.h>

void usage()
{
  printf("Usage:\n"
	 "\n"
	 "   xml_test infile.xml outfile.dts\n"
	 "\n"
	 "This program reads in the XML file and writes it out as a device tree.\n"
	 "\n");

  exit(20);
}

void write_device_tree(xml* x, writestream& w);


int main(int argc, const char* const argv[])
{
  if(argc<3)
    usage();

  const char* infile  = argv[1];
  const char* outfile = argv[2];

  auto x = new xml(infile);
  writestream w(outfile);
  
  //x->print_with_indent(w);

  write_device_tree(x, w);

  delete x;
  
  return 0;
}


const char* get_xsa_parameter_value(xml* x, String name, bool debug=false);

//
// Searching this way through the XSA for every parameter is highly inefficient.
// It doesn't matter; this isn't real-time and it's fast enough anyway.
//
const char* get_xsa_parameter_value(xml* x, String name, bool debug)
{
  xml_entry* EDKSYSTEM = x->find_entry("EDKSYSTEM");

  if(!EDKSYSTEM)
    {
      printf("ERROR:  Can't find EDKSYSTEM in the input XML file.\n");
      exit(20);
    }
  
  xml_entry* MODULES = EDKSYSTEM->find_child("MODULES");
  
  if(!MODULES)
    {
      printf("ERROR:  Can't find MODULES in EDKSYSTEM in the input XML file.\n");
      exit(20);
    }
  
  int module_count = 0;
  xml_entry* MODULE = MODULES->find_child("MODULE");

  while(MODULE)
    {
      //if(debug)
      //printf("Found %s number %d\n", (const char*)MODULE->name, module_count);
      
      xml_entry* PARAMETERS = MODULE->find_child("PARAMETERS");
      
      if(PARAMETERS)
	{
	  //if(debug)
	  //	printf("Found %s\n", (const char*)PARAMETERS->name);
	      
	  int parameter_count = 0;
	  xml_entry* PARAMETER = PARAMETERS->find_child("PARAMETER");
	  
	  while(PARAMETER)
	    {
	      //if(debug)
	      //printf("Found %s number %d\n", (const char*)PARAMETER->name, parameter_count);
	      
	      xml_attribute* NAME  = PARAMETER->find_attribute("NAME");
	      xml_attribute* VALUE = PARAMETER->find_attribute("VALUE");

	      if(debug)
		{
		  if(NAME && VALUE)
		    printf("%s = %s\n", (const char*)NAME->value, (const char*)VALUE->value);
		  else if(NAME)
		    printf("%s    (This has no value)\n", (const char*)NAME->value);
		}

	      if(NAME->value.equals(name))
		{
		  if(!VALUE)
		    return 0;
		  return VALUE->value;
		}

	      PARAMETER = PARAMETERS->find_child_after("PARAMETER", PARAMETER);
	      parameter_count++;
	    }
	}
      
      MODULE = MODULES->find_child_after("MODULE", MODULE);
      module_count++;	      
    }
  
  return 0;
}


int get_gpio_pin_from_name(xml* x, const char* pin_name)
{
  const char* pin_assignments = get_xsa_parameter_value(x, "PSU_MIO_TREE_SIGNALS");
  if(!pin_assignments)
    {
      printf("Can't get MIO pin assignments.\n");
      exit(20);
    }

  //printf("pin assignments are %s\n", (const char*)pin_assignments);

  int pin_count = 0;
  int i=0;
  while(pin_assignments[i])
    {
      int match = stringMatchToLengthOfFirst(pin_name, &pin_assignments[i]);
      if(match)
	{
	  //printf("Pin \"%s\" is assigned to MIO_%d\n", pin_name, pin_count);
	  return pin_count;
	}
      
      while(pin_assignments[i] && pin_assignments[i]!='#')
	i++;

      if(pin_assignments[i])
	i++;
      pin_count++;
    }
  
  //printf("Can't find pin assignment for MIO pin \"%s\"\n", pin_name);
  //exit(20);
  return -1;
}


void get_addresses_from_map(const char* PSU__PROTECTION__SLAVES, const char* key, unsigned long long& start, unsigned long long& end, bool& enabled)
{
     
  const char* s = PSU__PROTECTION__SLAVES;
  
  while(*s && !equalToLengthOfFirst(key, s))
    {
      s++;
    }

  if(!*s)
    {
      printf("ERROR:  Key %s not found in string.\n", key);
      exit(20);
    }

  s += strlen(key);

  if(*s!=';')
    {
      printf("ERROR:  Key %s missing following semicolon in string.\n", key);
      exit(20);
    }

  s++;
  
  int num = sscanf(s, "%llx", &start);

  if(num<=0)
    {
      printf("ERROR:  Can't read key %s's start address.\n", key);
      exit(20);
    }

  while(*s != ';')
    {
      if(!*s)
	{
	  printf("Unexpected end of string parsing string for %s\n", key);
	  exit(20);
	}
      s++;
    }

  s++;
  
  num = sscanf(s, "%llx", &end);

  if(num<=0)
    {
      printf("ERROR:  Can't read %s's end address.\n", key);
      exit(20);
    }

    while(*s != ';')
    {
      if(!*s)
	{
	  printf("Unexpected end of string parsing for %s\n", key);
	  exit(20);
	}
      s++;
    }

  s++;
  
  long long unsigned is_enabled = -1;
  num = sscanf(s, "%llx", &is_enabled);

  if(num<=0)
    {
      printf("ERROR:  Can't read DDR_LOW's enablement.\n");
      exit(20);
    }

  enabled = (is_enabled!=0);
}

void write_device_tree(xml* x, writestream& w)
{
  printf("\n");

  w <<
    "// Automatically generated by write_device_tree.cc\n"
    "/dts-v1/;\n"
    "#include \"zynqmp.dtsi\"\n"
    "#include \"zynqmp-clk-ccf.dtsi\"\n"
    "// #include \"pcw.dtsi\"\n"
    "/ {\n"
    "     chosen {\n"
    "        bootargs    =  \"earlycon console=ttyPS0,115200 clk_ignore_unused rootwait root=/dev/mmcblk0p2 rw earlyprintk net.ifnames=0\";\n"
    "        stdout-path =  \"serial0:115200n8\";\n"
    "     };\n";

  w <<
    "     aliases {\n";

  int ethernet_number = 0;
  int i2c_number      = 0;
  int serial_number   = 0;
  int spi_number      = 0;
  int mmc_number      = 0;
  int usb_number      = 0;

  String one = "1";

  for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__ENET%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  printf("Ethernet gem%d is enabled.\n", i);
	  w << "        ethernet" << ethernet_number++ << "   =  &gem" << i << ";\n";
	}
    }

  for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__I2C%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  printf("I2C bus i2c%d is enabled.\n", i);
	  w << "        i2c" << i2c_number++ << "        =  &i2c" << i << ";\n";
	}
    }

  for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__UART%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  printf("UART%d is enabled.\n", i);
	  w << "        serial" << serial_number++ << "     =  &uart" << i << ";\n";
	}
    }
  
  // These seems to be in many device trees, with no corresponding entry in the XSA
  w << "        serial" << serial_number++ << "     =  &dcc;\n";
  w << "        rtc0        =  &rtc;\n";
  
  for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__SPI%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  printf("SPI bus spi%d is enabled.\n", i);
	  w << "        spi" << spi_number++ << "        =  &spi" << i << ";\n";
	}
    }


  for(int i=0; i<=1; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__SD%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  w << "        mmc" << mmc_number++ << "        =  &sdhci" << i << ";\n";
	}
    }

  for(int i=0; i<=1; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__USB%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  printf("USB%d is enabled.\n", i);
	  w << "        usb" << usb_number++ << "        =  &usb" << i << ";\n";
	}
    }
    
  if ( one.equals(get_xsa_parameter_value(x, "PSU__NAND__PERIPHERAL__ENABLE")))
    {
      printf("Flash memory nvmem0 is enabled.\n");
      w << "        nvmem0" << "        =  &eeprom;\n";
    }
  
  w <<
    "     };\n";

  const char* PSU__PROTECTION__SLAVES = get_xsa_parameter_value(x, "PSU__PROTECTION__SLAVES");

  if(!PSU__PROTECTION__SLAVES)
    {
      printf("ERROR:  Key PSU__PROTECTION__SLAVES not found in XML.\n");
      exit(20);
    }
  
  //w << PSU__PROTECTION__SLAVES << "\n";
  printf("\n");

  unsigned long long low_start;
  unsigned long long low_end;
  bool               low_enabled;

  get_addresses_from_map(PSU__PROTECTION__SLAVES, "DDR_LOW", low_start, low_end, low_enabled);
  printf("DDR_LOW  is %s address 0x%09llX to address 0x%09llX.\n", low_enabled?"is enabled from":"disabled but would be from", low_start, low_end);

  unsigned long long high_start;
  unsigned long long high_end;
  bool               high_enabled;

  get_addresses_from_map(PSU__PROTECTION__SLAVES, "DDR_HIGH", high_start, high_end, high_enabled);
  printf("DDR_HIGH is %s address 0x%09llX to address 0x%09llX.\n", high_enabled?"is enabled from":"disabled but would be from", high_start, high_end);

  if(!low_enabled && !high_enabled)
    {
      printf("ERROR:  No SDRAM is enabled in the input XML.\n");
      exit(20);
    }

  unsigned long long low_length  = low_enabled   ?   low_end -  low_start + 1  : 0;
  unsigned long long high_length = high_enabled  ?  high_end - high_start + 1  : 0;

  printf("Total SDRAM: %.2fGiB\n\n", (low_length+high_length) / float(1ll<<30));
  
  
  w <<
    "     memory@0 {\n"
    "        device_type = \"memory\";\n";


  {
    char tmp[1000];
    sprintf(tmp, 
	    "        reg = <0x%llX 0x%llX 0x%llX 0x%llX>, <0x%llX 0x%llX 0x%llX 0x%llX>;\n",
	    low_start>>32,   low_start&0xFFFFFFFF,  low_length>>32,  low_length&0xFFFFFFFF,   
	    high_start>>32, high_start&0xFFFFFFFF, high_length>>32, high_length&0xFFFFFFFF);

    w << tmp;
  }
  
  
  w <<
    "     };\n"
    "};\n"
    "\n";

//
// Don't normally define the GEM in the device tree, since
// it can't be fully specified anyway.  The PHY address is
// set externally to the ZYNQ, in typical configurations,
// and isn't included in the Vivado ZYNQ block diagram
// configuration.  So it's not in the HWH file inside the
// XSA file.  We have no way to know the PHY address here.
// So it must be added manually anyway.  Thus do the
// entire GEM configuration manually by steps external
// to this program
//
#ifdef DEFINE_THE_GEM
  srand48((long int) time(NULL));

  for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__ENET%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  //printf("Ethernet gem%d is enabled.\n", i);

	  sprintf(tmp, "PSU__ENET%d__GRP_MDIO__ENABLE", i);
	  bool uses_mdio = one.equals(get_xsa_parameter_value(x, tmp));
	  
	  w <<
	    "\n"
	    "&gem" << i <<  " {\n";
	  
	  //
	  // For STYNQ, we override the MAC address anyway with a random value seeded by the hostname.
	  // However, in case that fails, have a random address to try to keep two STYNQ boxes from
	  // having the same ethernet address, and conflicting.
	  //
	  char tmp[200];
    
	  sprintf(tmp, "%02lX %02lX %02lX %02lX %02lX %02lX",
		  lrand48()&0xFF, lrand48()&0xFF, lrand48()&0xFF,
		  lrand48()&0xFF, lrand48()&0xFF, lrand48()&0xFF);
	  
	  w <<
	    "   local-mac-address = ["  << tmp << "];\n"
	    "\n"
	    "   phy-mode = \"rgmii-id\";\n"
	    "   status = \"okay\";\n"
	    "   xlnx,ptp-enet-clock = <0x0>;\n";


	  //
	  // ZCU111 uses phy reg c.  RFSoC4x2 uses phy reg f
	  // These values come from the PHY chip (83867) external
	  // to the ZYNQ.  They are set by strapping resistors.
	  // There is no entry in the Vivado ZYNQ block design
	  // configuration to set this, so it must be set in the
	  // device tree by some other method.
	  //
	  
	  if(uses_mdio)
	    {
	      w <<
		"   phy-handle = <&phy0>;\n"
		"   mdio: mdio {\n"
		"      #address-cells = <1>;\n"
		"      #size-cells = <0>;\n"
		"      phy0: ethernet-phy@f {\n"
		"         #phy-cells = <1>;\n"
		"         compatible = \"ethernet-phy-id2000.a231\";\n"
		"         reg = <0xf>;\n"
		"         ti,rx-internal-delay = <0x8>;\n"
		"         ti,tx-internal-delay = <0xa>;\n"
		"         ti,fifo-depth = <0x1>;\n"
		"         ti,dp83867-rxctrl-strap-quirk;\n"
		"         //reset-gpios = <&tca6416_u22 6 GPIO_ACTIVE_LOW>;\n"
		"      };\n"
		"   };\n";
	    }
	  
	  w <<
	    "};\n"
	    "\n";

	  printf("Configured gem%d to random ethernet address [%s].  (Overridden in Linux later).\n\n", i, tmp);
	}
      
    }
#endif // DEFINE_THE_GEM
  

  //
  // No known source for this in the XSA's HWH file:
  //
  w <<
    "\n"
    "&gic {\n"
    "   num_cpus = <2>;\n"
    "   num_interrupts = <96>;\n"
    "};\n"
    "&lpd_dma_chan1 {\n"
    "   status = \"okay\";\n"
    "};\n"
    "&lpd_dma_chan2 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&lpd_dma_chan3 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&lpd_dma_chan4 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&lpd_dma_chan5 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&lpd_dma_chan6 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&lpd_dma_chan7 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&lpd_dma_chan8 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&xilinx_ams {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&cci {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "\n";

  bool DP_enable = one.equals(get_xsa_parameter_value(x, "PSU__DISPLAYPORT__PERIPHERAL__ENABLE"));

  if(DP_enable)
    {
      bool DP_lane0_enable = one.equals(get_xsa_parameter_value(x, "PSU__DISPLAYPORT__LANE0__ENABLE"));
      bool DP_lane1_enable = one.equals(get_xsa_parameter_value(x, "PSU__DISPLAYPORT__LANE1__ENABLE"));

      if(!DP_lane0_enable && !DP_lane1_enable)
	{
	  printf("DisplayPort enabled but no lanes enabled.\n");
	  exit(20);
	}
      
      String DP_ref_clk_string = get_xsa_parameter_value(x, "PSU__DP__REF_CLK_SEL");

      int DP_ref_clk;
      
      if(DP_ref_clk_string.equals("Ref Clk0"))
	{
	  DP_ref_clk = 0;
	}
      else if(DP_ref_clk_string.equals("Ref Clk1"))
	{
	  DP_ref_clk = 1;
	}
      else if(DP_ref_clk_string.equals("Ref Clk2"))
	{
	  DP_ref_clk = 2;
	}
      else if(DP_ref_clk_string.equals("Ref Clk3"))
	{
	  DP_ref_clk = 3;
	}
      else
	{
	  printf("Invalid entry in HWH XML for PUS__DP_REF_CLK_SEL\n");
	  exit(20);
	}
	
      String DP_lane0_io_string = get_xsa_parameter_value(x, "PSU__DISPLAYPORT__LANE0__IO");
      String DP_lane1_io_string = get_xsa_parameter_value(x, "PSU__DISPLAYPORT__LANE1__IO");

      int DP_lane0_io = 100;

      if(DP_lane0_enable)
	{
	  if(DP_lane0_io_string.equals("GT Lane0"))
	    {
	      DP_lane0_io = 0;
	    }
	  else if(DP_lane0_io_string.equals("GT Lane1"))
	    {
	      DP_lane0_io = 1;
	    }
	  else if(DP_lane0_io_string.equals("GT Lane2"))
	    {
	      DP_lane0_io = 2;
	    }
	  else if(DP_lane0_io_string.equals("GT Lane3"))
	    {
	      DP_lane0_io = 3;
	    }
	  else
	    {
	      printf("Read bad value for DisplayPort Lane0 I/O\n");
	      exit(20);
	    }
	}
      
      int DP_lane1_io = 100;
      
      if(DP_lane1_enable)
	{
	  if(DP_lane1_io_string.equals("GT Lane0"))
	    {
	      DP_lane1_io = 0;
	    }
	  else if(DP_lane1_io_string.equals("GT Lane1"))
	    {
	      DP_lane1_io = 1;
	    }
	  else if(DP_lane1_io_string.equals("GT Lane2"))
	    {
	      DP_lane1_io = 2;
	    }
	  else if(DP_lane1_io_string.equals("GT Lane3"))
	    {
	      DP_lane1_io = 3;
	    }
	  else
	    {
	      printf("Read bad value for DisplayPort Lane1 I/O\n");
	      exit(20);
	    }
	}

      // An attempt to match an old device tree entry to get sound working.  Didn't work.
#ifdef NOTDEF
      w <<
	" / {\n"
	"        zynqmp_dpaud_setting: dp-aud@fd4ac000 {\n"
	"           compatible = \"xlnx,zynqmp-dpaud-setting\", \"syscon\";\n"
	"           reg = <0x0 0xfd4ac000 0x0 0x1000>;\n"
	"           };\n"
	"};\n";
#endif
      
      w <<
	"&zynqmp_dpsub {\n";

      //
      // Note that the phys are <&psgtr  GTR_LANE  PHY_TYPE_DP  PHY_INSTANCE  REFCLK_NUMBER>
      //
      //  GTR_LANE is 0 to 3
      //  PHY_TYPE_DP is always 6   (for DisplayPort)
      //  PHY_INSTANCE is 0 or 1    (for DIsplayPort)
      //  REFCLK_NUMBER is 0 to 3
      //

      if(DP_lane0_enable && DP_lane1_enable)
	{
	  w << "   phy-names = \"dp-phy0\",\"dp-phy1\";\n";
	  w << "   phys = <&psgtr " << DP_lane0_io << " 6 0 " << DP_ref_clk << ">, <&psgtr " << DP_lane1_io << " 6 1 " << DP_ref_clk << ">;\n";
	}
      else if(DP_lane0_enable)
	{
	  w << "   phy-names = \"dp-phy1\";\n";
	  w << "   phys = <&psgtr " << DP_lane0_io << " 6 0 " << DP_ref_clk << ">;\n";
	}
      else // lane1 must be enabled
	{
	  w << "   phy-names = \"dp-phy0\";\n";
	  w << "   phys = <&psgtr " << DP_lane1_io << " 6 1 " << DP_ref_clk << ">;\n";
	}
      
      w <<
	"   status = \"okay\";\n"
	"   xlnx,max-lanes = <2>;\n";


      // An attempt to match an old device tree entry to get sound working.  Didn't work.
#ifdef NOTDEF
      w <<
	"   xlnx,dpaud-reg = <&zynqmp_dpaud_setting>;";
#endif
      
      // This was an attempt to match the device tree distributed with the kernel.
      // It didn't work to bring up DisplayPort audio.
      //#ifdef NOTDEF
      w <<
	"   dma-names = \"vid0\", \"vid1\", \"vid2\", \"gfx0\",\n"
	"               \"aud0\", \"aud1\";\n"
	"   dmas = <&zynqmp_dpdma 0>,\n"
	"          <&zynqmp_dpdma 1>,\n"
	"          <&zynqmp_dpdma 2>,\n"
	"          <&zynqmp_dpdma 3>,\n"
	"          <&zynqmp_dpdma 4>,\n"
	"          <&zynqmp_dpdma 5>;\n";
      //#endif
      
      // If ports are added like below, DisplayPort doesn't work.
#ifdef NOTDEF
      w <<
	"   ports {\n"
	"      #address-cells = <1>;\n"
	"      #size-cells = <0>;\n"
	"      port@0 {\n"
	"         reg = <0>;\n"
	"         };\n"
	"      port@1 {\n"
	"         reg = <1>;\n"
	"         };\n"
	"      port@2 {\n"
	"         reg = <2>;\n"
	"         };\n"
	"      port@3 {\n"
	"         reg = <3>;\n"
	"         };\n"
	"      port@4 {\n"
	"         reg = <4>;\n"
	"         };\n"
	"      port@5 {\n"
	"         reg = <5>;\n"
	"         };\n"
	"      };\n";
#endif
      w <<
	"   };\n";

      w <<
	"&zynqmp_dpdma {\n"
	"   status = \"okay\";\n"
	"   };\n";

      
      w <<
        "&zynqmp_dp_snd_pcm0 {\n"
	"   status = \"okay\";\n"
	"   };\n"
	"&zynqmp_dp_snd_pcm1 {\n"
	"   status = \"okay\";\n"
	"   };\n"
	"&zynqmp_dp_snd_card0 {\n"
	"   status = \"okay\";\n"
	"   };\n"
	"&zynqmp_dp_snd_codec0 {\n"
	"   status = \"okay\";\n"
	"   };\n"
	"   \n";


    }
      

  w <<
    "&fpd_dma_chan1 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&fpd_dma_chan2 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&fpd_dma_chan3 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&fpd_dma_chan4 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&fpd_dma_chan5 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&fpd_dma_chan6 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&fpd_dma_chan7 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&fpd_dma_chan8 {\n"
    "   status = \"okay\";\n"
    "   };\n"
    "&gpio {\n"
    "   emio-gpio-width = <32>;\n"
    "   gpio-mask-high = <0x0>;\n"
    "   gpio-mask-low = <0x5600>;\n"
    "   status = \"okay\";\n"
    "   };\n";

  for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__I2C%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  w <<
	    "&i2c" << i << " {\n"
	    "   status = \"okay\";\n"
	    "   };\n";
	}
    }

  w <<
    "&rtc {\n"
    "   status = \"okay\";\n"
    "   };\n";

  for(int i=0; i<=1; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__SD%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  char tmp[200];
	  sprintf(tmp, "PSU__CRL_APB__SDIO%d_REF_CTRL__ACT_FREQMHZ", i);
	  const char* freq_MHz = get_xsa_parameter_value(x, tmp);
	  if(!freq_MHz)
	    {
	      printf("Can't get SD card %d clock frequency.\n", i);
	      exit(20);
	    }

	  const char* pin_name = (i==0) ? "sdio0_clk_out" : "sdio1_clk_out";

	  int pin_number = get_gpio_pin_from_name(x, pin_name);

	  int bank = 4;
	  
	  if(pin_number<0)
	    {
	      printf("WARNING:  Can't get pin number for sdio%d\n", i);
	    }
	  else if(pin_number<=25)
	    {
	      bank = 0;
	    }
	  else if(pin_number<=51)
	    {
	      bank = 1;
	    }
	  else if(pin_number<=51)
	    {
	      bank = 2;
	    }
	  else
	    {
	      bank = 3;
	    }

	  const char* wp_name = (i==0) ? "sdio0_wp" : "sdio1_wp";
  
	  int wp_pin = get_gpio_pin_from_name(x, wp_name);

	  const char* disable_wp = (wp_pin<0) ? "   disable-wp;\n" : "";
	  
	  long freq_Hz = atof(freq_MHz) * 1000000l;

	  printf("SD card sdhci%d is enabled with clock frequency %sMHz.\n", i, freq_MHz);

	  w <<
	    "&sdhci" << i << " {\n"
	    "   clock-frequency = <" << freq_Hz << ">;\n"
	    "   status = \"okay\";\n"
	    "   xlnx,mio-bank = <" << bank << ">;\n"
	    "   no-1-8-v;\n" <<
	    disable_wp <<
	    "   };\n";
	}
    }

  for(int i=0; i<=1; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__USB%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  //printf("USB%d is enabled.\n", i);
	  w <<
	    "&usb" << i << " {\n"
	    "   status = \"okay\";\n"
	    "   xlnx,tz-nonsecure = <0x1>;\n"
	    "   xlnx,usb-polarity = <0x0>;\n"
	    "   xlnx,usb-reset-mode = <0x0>;\n"
	    "   };\n";
	  
	  w <<
	    "&dwc3_" << i << " {\n"
	    "   status = \"okay\";\n"
	    "   };\n";
	}
    }

  bool use_psgtr = false;
  long ref_clk_freq_Hz[4] = { 0, 0, 0, 0 };
  const char* psgtr_devices[] = { "USB0", "USB1", "DP", "SATA", "PCIE", "SGMII" };
  int psgtr_devices_number = sizeof(psgtr_devices) / sizeof(psgtr_devices[0]);
  for(int i=0; i<psgtr_devices_number; i++)
    {
      char sel_string[200];
      sprintf( sel_string, "PSU__%s__REF_CLK_SEL",  psgtr_devices[i]);
      String sel_s      = get_xsa_parameter_value(x,  sel_string);

      int sel;
      if(sel_s.equals("Ref Clk0"))
	sel=0;
      else if(sel_s.equals("Ref Clk1"))
	sel=1;
      else if(sel_s.equals("Ref Clk2"))
	sel=2;
      else if(sel_s.equals("Ref Clk3"))
	sel=3;
      else
	continue;  // Device isn't using a ref clock.  Probably the device is off.

      char freq_string[200];
      sprintf(freq_string, "PSU__%s__REF_CLK_FREQ", psgtr_devices[i]);
      String freq_s_MHz = get_xsa_parameter_value(x, freq_string);

      long freq_Hz = atol(freq_s_MHz) * 1000000l;

      printf("%5s uses PSGTR Ref Clk%d at frequency %.2fMHz\n", psgtr_devices[i], sel, freq_Hz/1000000.0);
      
      if(ref_clk_freq_Hz[sel]==0)
	ref_clk_freq_Hz[sel] = freq_Hz;

      if(ref_clk_freq_Hz[sel]!=freq_Hz)
	{
	  printf("ERROR: Multiple devices are set up to use PSGTR Ref Clk %d at different frequencies:  %ldHz vs %ldHz.\n",
		 sel,  ref_clk_freq_Hz[sel], freq_Hz);
	  exit(20);
	}
      use_psgtr = true;
    }

  if(use_psgtr)
    {
      w << "/ {\n";
      
      for(int i=0; i<4; i++)
	{
	  if(!ref_clk_freq_Hz[i])
	    continue;

	  w <<
	    "   psgtr_refclk_" << i << ": refclk_" << i << " {\n"
	    "	   compatible = \"fixed-clock\";\n"
	    "	   #clock-cells = <0x00>;\n"
	    "	   clock-frequency = <" << ref_clk_freq_Hz[i] << ">;\n"
	    "   };\n";
	}

      w << "};\n";
  
      w <<
	"&psgtr {\n"
	"   status = \"okay\";\n";
      
      w << "   clocks = ";
      const char* sep = "";
      for(int i=0; i<4; i++)
	{
	  if(!ref_clk_freq_Hz[i])
	    continue;
	  
	  w << sep << "<&psgtr_refclk_" << i << ">";
	  sep = ", ";
	}
      
      w <<
	";\n";



      w << "   clock-names = ";
      sep = "";
      for(int i=0; i<4; i++)
	{
	  if(!ref_clk_freq_Hz[i])
	    continue;
	  
	  w << sep << "\"ref" << i << "\"";
	  sep = ", ";
	}
      
      w <<
	";\n";

      w <<
	"};\n";

    }

  
  for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__SPI%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  int num_cs = 0;
	  for(int j=0; j<3; j++)
	    {	      
	      sprintf(tmp, "PSU__SPI%d__GRP_SS%d__ENABLE", i, j);
	      if(one.equals(get_xsa_parameter_value(x, tmp)))
		num_cs++;
	    }
	  
	  w <<
	    "&spi" << i << " {\n"
	    "   is-decoded-cs = <0>;\n"
	    "   num-cs = <" << num_cs << ">;\n"
	    "   status = \"okay\";\n"
	    "   };\n";
	}
    }


    for(int i=0; i<=3; i++)
    {
      char tmp[200];
      sprintf(tmp, "PSU__UART%d__PERIPHERAL__ENABLE", i);
      if ( one.equals(get_xsa_parameter_value(x, tmp)))
	{
	  w << 
	    "&uart" << i << " {\n"
	    "   cts-override ;\n"
	    "   device_type = \"serial\";\n"
	    "   port-number = <0>;\n"
	    "   status = \"okay\";\n"
	    "   u-boot,dm-pre-reloc ;\n"
	    "   };\n";
	}
    }
    
    const char* pss_freq = get_xsa_parameter_value(x, "PSU__PSS_REF_CLK__FREQMHZ");
    if(pss_freq)
      {
	long freq_Hz = atol(pss_freq) * 1000000l;
	w <<
	  "&pss_ref_clk {\n"
	  "   clock-frequency = <" << freq_Hz << ">;\n"
	  "   };\n";
      }
    
    if ( one.equals(get_xsa_parameter_value(x, "PSU__VIDEO_REF_CLK__ENABLE")))
      {
	const char* freq = get_xsa_parameter_value(x, "PSU__VIDEO_REF_CLK__FREQMHZ");

	if(freq)
	  {
	    long freq_Hz = atol(freq) * 1000000l;
	    w <<
	      "&video_clk {\n"
	      "   clock-frequency = <" << freq_Hz << ">;\n"
	      "   };\n";
	  }
      }


    //
    // No known source for these in the input XSA / HWH file.
    //
    w <<
      "&ams_ps {\n"
      "   status = \"okay\";\n"
      "   };\n"
      "&ams_pl {\n"
      "   status = \"okay\";\n"
      "   };\n"
      "&dcc {\n"
      "   status = \"okay\";\n"
      "   };\n"
      "\n";


}
