
/*
 * Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., http://www.fsf.org/about/contact/
 *
 */


#include "readstream.hh"
#include <errno.h>
#include <unistd.h>

readstream::readstream(FILE* fpin)
{
  int i;

  fp = fpin;

  for(i=0; i<MAXFILENAMELEN-1; i++)
    {
      int c = filename[i] = ("preopened file")[i];
      if(!c)
	break;
    }

  readindex = 0;
  writeindex = 0;
  num_in_buffer = 0;
  eof = 0;
  linenumber = 1;
  fileposition = 0;
}

readstream::readstream(const char* name)
{
  int i;

  fp = fopen(name, "r");
  if(!fp)
    {
      printf("Can't open input stream \"%s\".\n", name);
      exit(10);
    }

  for(i=0; i<MAXFILENAMELEN-1; i++)
    {
      int c = filename[i] = name[i];
      if(!c)
	break;
    }

  readindex = 0;
  writeindex = 0;
  num_in_buffer = 0;
  eof = 0;
  linenumber = 1;
  fileposition = 0;
}

readstream::~readstream()
{
}

//
// Only reading 1/4 at a time allows plenty of room for ungetchar().
//
int readstream::getchar()
{
  int v;
  int num;

  while(num_in_buffer==0 && !eof)
    {
      num = READSTREAM_BUFFERSIZE/4;
      if(writeindex+num>READSTREAM_BUFFERSIZE)
	num = READSTREAM_BUFFERSIZE-writeindex;
      
      v = fread(&buffer[writeindex], 1, num, fp);
      
      if(v<=0)
	{
	  v=0;

	  if(feof(fp))
	    {
	      eof = 1;
	    }
	  else if (errno==EAGAIN)
	    {
	      usleep(50000);
	    }
	  else
	    {
	      printf("Error reading file in readstream::getchars()!\n");
	      exit(10);
	    }
	}
      
      num_in_buffer += v;
      writeindex = (writeindex+v)%READSTREAM_BUFFERSIZE;
    }
  
  if(eof && num_in_buffer==0)
    {
      eof++;  // Tell ungetchar how far past EOF we are.
      return EOF;
    }

  v = (int)buffer[readindex];
  readindex = (readindex+1)%READSTREAM_BUFFERSIZE;
  fileposition++;
  num_in_buffer--;

  if(v=='\n')
    linenumber++;

  return v;
}

int readstream::getchars(char* buff, int num)
{
  int i;
  int c;

  for(i=0; i<num; i++)
    {
      c = getchar();

      if(c==EOF)
	break;

      buff[i] = (char)c;
    }

  return i;
}

void readstream::ungetchars(int num)
{
  int i;
  
  for(i=0; i<num; i++)
    ungetchar();
}

void readstream::ungetchar()
{
  //
  // Reduce length after eof if we have passed eof.
  //
  if(eof>=2)
    {
      eof--;
      return;
    }

  fileposition--;
  num_in_buffer++;
  if(num_in_buffer>READSTREAM_BUFFERSIZE || fileposition<0)
    {
      printf("Capacity exceeded for readstream::ungetchar()!\n");
      exit(10);
    }

  readindex = (readindex+READSTREAM_BUFFERSIZE-1)%READSTREAM_BUFFERSIZE;

  if(buffer[readindex]=='\n')
    linenumber--;
}

int readstream::rewindToStartOfLine()
{
  int i=0;
  for(;;)
    {
      if(fileposition==0)
	break;
      ungetchar();
      i++;
      int c = getchar();
      if(c=='\n')
	break;
      ungetchar();
    }

  return i;
}
