
/*
 * Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., http://www.fsf.org/about/contact/
 *
 */


#include "tokens.hh"

//
// Returns 0 if no match, length of token if match.
//
int Token::match(char* s)
{
  int i;

  for(i=0; tokentext[i]; i++)
    {
      if(s[i]!=tokentext[i])
	return 0;
    }

  return i;
}

TokenList::TokenList(TokenListSetupFunction tl, const char* tchars)
{ 
  termchars = tchars;
  numtokens = 0; 
  maxlength = 0; 

  tl(*this);
}

void TokenList::addToken(Token* t) 
{ 
  if(t==0)
    {
      printf("Token was null in TokenList::addToken.\n");
      exit(10);
    }

  if(numtokens>=TOKENLIST_MAXTOKENS) 
    { 
      printf("Max Tokens Exceeded in TokenList::addToken.\n"); 
      exit(10);
    } 

  int length = strlen(t->text());
  if(length>maxlength)
    maxlength=length;

  tokens[numtokens++] = t; 
}

//
// Matches a token.  Makes sure that character after token is
// from terminating list, if list is nonzero.  Positions file 
// on terminating character.  Returns Token matched if there is
// a match, zero otherwise.
//
Token*
TokenList::matchToken(readstream& r)
{
  int i;
  char buffer[100];

  if(maxlength>100-5)
    {
      printf("Token string too long in TokenList::matchToken!\n");
      exit(10);
    }

  int num = r.getchars(buffer, maxlength+1);

  //
  // If we couldn't get enough characters for the longest possible
  // match, we could still match a shorter string.  Since you can't
  // match zeros, fill first unread position with zero and continue.
  //
  if(num<maxlength+1)
    buffer[num] = 0;

  for(i=0; i<numtokens; i++)
    {
      int m = tokens[i]->match(buffer);
      if(m)
	{
	  if( (!termchars) || isInString(buffer[m], termchars))
	    {
	      r.ungetchars(num-m);
	      return tokens[i];
	    }
	}
    }

  r.ungetchars(num);
  return 0;
}

void
TokenList::printExpecting()
{
  int i;

  printf("\nFailure to find one of these tokens:\n\n");

  for(i=0; i<numtokens; i++)
    {
      printf("%s\n", tokens[i]->text());
    }

  printf("\n");
}

