#!/bin/bash

VERSION=xlnx_rel_v2024.2


depend_exe()
{
    tool="$1"
    if [ "`which $tool`" = "" ]; then
	echo ""
	echo "Tool \"$tool\" must be built to run this script.  Run exebuild.sh there."
	echo ""
	exit 20
    fi
}

depends()
{
    tool="$1"
    if [ "`which $tool`" = "" ]; then
	echo ""
	echo "Xilinx's tool $tool must be installed to run this script."
	echo ""
	exit 20
    fi
}

IN_XSA="$1"

if [ ! -f "$IN_XSA" ]; then
    echo "Usage: $0 XSA_FILE.xsa"
    exit 20
fi

OUTDIR=out
TMPDIR=tmp

DESIGN=`basename "$IN_XSA" | sed 's/[.].*//'`
DTS="$OUTDIR/${DESIGN}.dts"
DTB="$OUTDIR/${DESIGN}.dtb"
XSA="${DESIGN}.xsa"

if [ ! -f "$DTS" ]; then
    echo "No Device Tree \"$DTS\" exists; building it."
elif [ ! -f "$DTB" ]; then
    echo "No Device Tree \"$DTB\" exists; building it."
else
    if [ "$IN_XSA" -nt "$DTS" ]; then
	echo "Rebuilding \"$DTS\"; the XSA file \"$XSA\" is newer."
    elif [ "$IN_XSA" -nt "$DTB" ]; then
	echo "Rebuilding \"$DTB\"; the XSA file \"$XSA\" is newer."
    else
	echo "Skipping build of \"$DTS\" and \"$DTB\"; the XSA file \"$XSA\" is older."
	exit 0
    fi
fi

depend_exe ../u-boot-xlnx/exe/dtc
depends xsct

rm -f "$DTS" "$DTB"

rm -rf "$TMPDIR"
mkdir -p "$TMPDIR"
mkdir -p "$OUTDIR"


cp "$IN_XSA" "$TMPDIR/$XSA"

rm -rf device-tree-xlnx

if [ ! -f device-tree-xlnx-$VERSION.tgz ]; then
    echo ""
    echo ""
    echo "  Have no local copy of device-tree-xlnx-$VERSION.tgz.  Fetching and creating local copy."
    echo ""
    echo ""
    git clone https://github.com/Xilinx/device-tree-xlnx
    (cd device-tree-xlnx; git checkout $VERSION)
    tar czf device-tree-xlnx-$VERSION.tgz device-tree-xlnx
else
    tar xzf device-tree-xlnx-$VERSION.tgz
fi


rm -rf gen-machine-conf

if [ ! -f gen-machine-conf.tgz ]; then
    echo ""
    echo ""
    echo "  Have no local copy of gen-machine-conf.tgz.  Fetching and creating local copy."
    echo ""
    echo ""
    git clone https://github.com/Xilinx/gen-machine-conf.git
    #(cd gen-machine-conf; git checkout $VERSION)
    tar czf gen-machine-conf.tgz gen-machine-conf
else
    tar xzf gen-machine-conf.tgz
fi


#
# Note:
#
#  Information is at these web addresses:
#
#   https://xilinx-wiki.atlassian.net/wiki/spaces/A/pages/3164274708/Running+gen-machine-conf
#   https://xilinx-wiki.atlassian.net/wiki/spaces/A/pages/3156475927/Overview+of+System+Device+Tree+Generator+SDTGen
#   https://xilinx-wiki.atlassian.net/wiki/spaces/A/pages/3172532226/Understanding+the+Relationship+Between+SDTGen+and+XSCT
#   https://xilinx-wiki.atlassian.net/wiki/spaces/A/pages/2743468485/Porting+embeddedsw+components+to+system+device+tree+SDT+based+flow#Known-Gaps-(2024.2)
#   https://github.com/Xilinx/system-device-tree-xlnx/blob/master/README.md#steps-to-use-sdtgen
#
#   https://xilinx-wiki.atlassian.net/wiki/spaces/A/pages/3164209180/Overview+of+gen-machine-conf
#   
#   
# Example run of gen-machine-conf:
#
#   gen-machine-conf parse-sdt --hw-description <path_to_sdtgen_output_directory> -c <conf-directory> -l <path-to-build-directory>/build/conf/local.conf --machine-name <soc-family>-<board-name>-sdt-<design-name> -g full
#

rm -rf device-tree-xlnx
#mv device-tree-xlnx "$TMPDIR/"
mv gen-machine-conf "$TMPDIR/"
#
# Enable overlay to exclude PL items from the device tree.
#

#cat >"$TMPDIR"/convert.tcl <<EOF
#createdts -local-repo device-tree-xlnx -hw $XSA -platform-name mydevice -out .
#EOF

ls -l $TMPDIR

cat >"$TMPDIR"/convert.tcl <<EOF
sdtgen set_dt_param -xsa $XSA -dir xsa
sdtgen generate_sdt
EOF

echo ""
echo "Running SDTGen"
echo ""

(cd "$TMPDIR"; xsct convert.tcl)

PCW=`find "$TMPDIR" -name pcw.dtsi`

if [ ! -f "$PCW" ]; then
    echo "Generation of device tree failed in XSCT"
    exit 20
fi

echo ""
echo "Running gen-machine-conf"
echo ""

PCW_DIR=`dirname "$PCW"`

touch "$TMPDIR"/local.conf

(cd "$TMPDIR"; gen-machine-conf/gen-machineconf --soc-family zynqmp --hw-description "$XSA" --xsct-tool /tools/Xilinx/Vitis/2024.2/bin )


exit 20

#
# createdts doesn't properly generate the psgtr clocks, for some reason.
# Figure out what they are from the XST and add definitions here.
#

# First, extract the XSA
(cd "$TMPDIR"; mkdir xsa_dir; cd xsa_dir; unzip ../$XSA)

HWH=`ls "$TMPDIR"/xsa_dir/*.hwh`

if [ ! -f "$HWH" ]; then
    echo "Failed to properly extract XSA file and get its .hwh"
    exit 20
fi


# Next, extract the frequency of each of the four refclk's.
refclk_uses=""
clocks=""
clock_names=""
clock_separator=""

for refclk in 0 1 2 3; do
    refclk_name="Ref Clk$refclk"  # This is the name given to the refclk in the XSA hwh file
    refclk_use=`grep REF_CLK_SEL "$HWH" | grep "$refclk_name" | head -1 | sed 's/_SEL.*//; s/.*NAME=.//'`
    refclk_freq_MHz=`grep REF_CLK_FREQ "$HWH" | grep "$refclk_use" | head -1 | sed 's/.*VALUE=.//; s/[^0-9.]*//g'`

    if [ "$refclk_use" = "" ]; then
	refclk_use=unused;
	refclk_freq_MHz=0;
    fi
    
    refclk_freq_Hz=`echo "1000000 * $refclk_freq_MHz" | bc -l | sed 's/[.].*//'` 
    echo "Reference clock $refclk is used for $refclk_use at frequency ${refclk_freq_MHz}MHz (${refclk_freq_Hz}Hz)"

    if [ "$refclk_use" != "unused" ]; then
	echo >>"$PCW" ""
	echo >>"$PCW" "/ {"
	echo >>"$PCW" "    ref_clk_$refclk:$refclk_use {"
	echo >>"$PCW" "            compatible = \"fixed-clock\";"
	echo >>"$PCW" "            #clock-cells = <0x00>;"
	echo >>"$PCW" "            clock-frequency = <$refclk_freq_Hz>;"
	echo >>"$PCW" "    };"
	echo >>"$PCW" "};"
	echo >>"$PCW" ""
    
	refclk_uses="$refclk_uses$refclk_use  "
	clocks="${clocks}${clock_separator}<&ref_clk_$refclk> "
	clock_names="${clock_names}${clock_separator}\"ref$refclk\""
	clock_separator=", "
    fi
done

echo >>"$PCW" ""
echo >>"$PCW" ""
echo >>"$PCW" "&psgtr {"
echo >>"$PCW" "      status = \"okay\";"
echo >>"$PCW" "      //  $refclk_uses"
echo >>"$PCW" "      clocks = $clocks ;"
echo >>"$PCW" "      clock-names = $clock_names ;"
#echo >>"$PCW" "      clocks = <&ref_clk_0>, <&ref_clk_1>, <&ref_clk_2>, <&ref_clk_3> ;"
#echo >>"$PCW" "      clock-names = \"ref0\", \"ref1\", \"ref2\", \"ref3\";"
echo >>"$PCW" "};"
echo >>"$PCW" ""
echo >>"$PCW" ""


DTS_DIR=`dirname "$PCW"`

#
# xsct is targeted at older kernels that have different device tree requirements, especially for the
# DisplayPort settings.  Remove these and replace with ones shipped with the kernel.
#
#mkdir "$TMPDIR"/old
#mv "$TMPDIR"/include "$TMPDIR"/*.dtsi "$TMPDIR"/old
#mv "$TMPDIR"/include "$TMPDIR"/old
#cp ../kernel/linux-xlnx/arch/arm64/boot/dts/xilinx/zynqmp*.dtsi "$TMPDIR"/
#mkdir "$TMPDIR"/include
#cp -r ../kernel/linux-xlnx/include/dt-bindings "$TMPDIR"/include

#
# Make fixes to match old stuff from xsct with new kernel stuff.
#
# Note that in &psgtr 1 6 0 0, the translation is LANE, TYPE, INSTANCE, REFCLK
#
# LANE is the physical serdes number.  TYPE refers to how it is used, 6 indicates DisplayPort
# INSTANCE is .  REFCLK is the number of the reference clock.  Note that the reference clock
# is dependent on the board layout, external chips on the board, and their configuration.
#
# The REFCLK 1 appears to match the number of the clock in the list in psgtr.  The number in "ref0"
# of 0 matches the hardware clock input 0.
#
#sed 's/xlnx_dpdma/zynqmp_dpdma/; s/serdes/psgtr/; s/[<][&]lane1 6 0 0 27000000[>]/<\&psgtr 1 6 0 1>/; s/[<][&]lane0 6 1 0 27000000[>]/<\&psgtr 0 6 1 1>/;  s/[<][&]lane1 6 0 1 27000000[>]/<\&psgtr 1 6 0 1>/; s/[<][&]lane0 6 1 1 27000000[>]/<\&psgtr 0 6 1 1>/' <"$TMPDIR"/old/pcw.dtsi >"$TMPDIR"/pcw.dtsi

#cat >>"$TMPDIR"/zynqmp.dtsi <<EOF
#/ {
#     dummy {
#       // These are only here because other things reference them.  Unused.
#       zynqmp_dp_snd_codec0: zynqmp_dp_snd_codec0 { };
#       zynqmp_dp_snd_pcm0: zynqmp_dp_snd_pcm0 { };
#       zynqmp_dp_snd_pcm1: zynqmp_dp_snd_pcm1 { };
#       zynqmp_dp_snd_card0: zynqmp_dp_snd_card { };
#     };
#};
#EOF

#cat >>"$TMPDIR"/pcw.dtsi <<EOF
#
#/ {
#       dp_clk:psgtr_dp_clock {
#                compatible = "fixed-clock";
#                #clock-cells = <0x00>;
#                clock-frequency = <27000000>;
#        };
#
#	usb_clk:psgtr_usb_clock {
#               compatible = "fixed-clock";
#		#clock-cells = <0x00>;
#                clock-frequency = <26000000>;
#	};
#
#//	dpcon {
#//		compatible = "dp-connector";
#//		label = "P11";
#//		type = "full-size";
#//
#//		port {
#//			dpcon_in: endpoint {
#//				remote-endpoint = <&dpsub_dp_out>;
#//			};
#//		};
#//	};
#
#
#};
#
#&psgtr {
 #     status = "okay";
 #     clocks = <&usb_clk>, <&dp_clk>;
 #     clock-names = "ref0", "ref1";
#};
#
#
#
#//&zynqmp_dpsub {
#//	status = "okay";
#//	phy-names = "dp-phy0", "dp-phy1";
#//	phys = <&psgtr 1 6 0 1>,
#//	       <&psgtr 0 6 1 1>;
#//
#//	ports {
#//		port@5 {
#//			dpsub_dp_out: endpoint {
#//				remote-endpoint = <&dpcon_in>;
#//			};
#//		};
#//	};
#//};
#
#
#EOF

cpp -nostdinc -I "$DTS_DIR"/include -I arch  -undef -x assembler-with-cpp  "$DTS_DIR"/system-top.dts -o - | sed 's@.*bootargs.*@  bootargs = "earlycon console=ttyPS0,115200 clk_ignore_unused rootwait root=/dev/mmcblk0p2 rw earlyprintk net.ifnames=0";@' >$DTS

../u-boot-xlnx/exe/dtc -@ -o "$DTB" "$DTS"

if [ -f "$DTS" -a -f "$DTB" ]; then
    echo ""
    echo "SUCCESS!  Device tree created in \"${DTS}\"."
    echo "          Compiled version in \"${DTB}\"."
    echo ""
    #rm -rf "$TMPDIR" device-tree-xlnx
else
    echo ""
    echo "ERROR!  Couldn't create device tree $DTS of compiled version $DTB."
    echo "Leaving intermediate files for debugging."
    echo ""
fi
