#!/bin/sh -e

USER="$1"
PASSWD="$2"
HOSTNAME="$3"
IP="$4"
GATEWAY_IP="$5"
NAMESERVER_IP="$6"

usage()
{
    echo ""
    echo "USAGE:  $0 username password hostname [dhcp | IP gateway_IP [nameserver_IP]]"
    echo ""
    echo "  This program creates a Debian base filesystem with an ssh server"
    echo "  enabled.  The IP can be set to \"dhcp\" to get an IP over DHCP, or"
    echo "  it can be set to any IPv4 address on your local network."
    echo ""
    echo "  If not using DHCP, a gateway IP and nameserver IP is usually needed"
    echo "  so that apt can get additional packages."
    echo ""
    echo "  If the gateway IP is omitted, no gateway will be configured."
    echo "  If the nameserver IP is omitted, the gateway will be used"
    echo "  for the nameserver also."
    echo ""
    exit 20
}

#
# Must be run as root
#
if [ `id -u` != "0" ]; then
    echo ""
    echo "This script must be run as root.  Use sudo."
    echo ""
    exit 20
fi

MODDIR=../kernel/out/lib/modules

if [ ! -d  "$MODDIR" ]; then

    echo ""
    echo "Missing directory \"$MODDIR\".  Build the kernel first,"
    echo "Since modules need to be copied out of its directory"
    echo "and into the filesystem."
    echo ""
    exit 20
fi


if [ "$USER" = "" -o "$PASSWD" = "" -o "$HOSTNAME" = "" ]; then
    usage
fi

if [ "$IP" = dhcp -o "$IP" = DHCP ]; then

    IP=DHCP

else
    
      IPCHECK=`echo "$IP"            | sed 's/[1-9][0-9]*[.][0-9][0-9]*[.][0-9][0-9]*[.][1-9][0-9]*//'`
    GWIPCHECK=`echo "$GATEWAY_IP"    | sed 's/[1-9][0-9]*[.][0-9][0-9]*[.][0-9][0-9]*[.][1-9][0-9]*//'`
    NSIPCHECK=`echo "$NAMESERVER_IP" | sed 's/[1-9][0-9]*[.][0-9][0-9]*[.][0-9][0-9]*[.][1-9][0-9]*//'`

    if [ "$IPCHECK" != "" ]; then

	echo ""
	echo "IP \"$IP\" is invalid."
	echo ""
	usage
    fi
    if [ "$GWIPCHECK" != "" ]; then

	echo ""
	echo "Gateway IP \"$GATEWAY_IP\" is invalid."
	echo ""
	usage
    fi
    if [ "$NSIPCHECK" != "" ]; then

	echo ""
	echo "Nameserver IP \"$NAMESERVER_IP\" is invalid."
	echo ""
	usage
    fi
    
fi

if [ "$NAMESERVER_IP" = "" ]; then
    NAMESERVER_IP="$GATEWAY_IP"
fi


echo "Creating image with user \"$USER\", password \"$PASSWD\", hostname \"$HOSTNAME\","

if [ "$IP" = DHCP ]; then
    echo "Configured for network configuration via DHCP."
else
    echo "IP \"$IP\", GATEWAY \"$GATEWAY_IP\", NAMESERVER \"$NAMESERVER_IP\"."

fi

ROOT_NOPATH=STYNQ_filesystem
ROOT="$PWD/$ROOT_NOPATH"

echo "##########################################################################################"
echo "##"

if [ -d "$ROOT" ]; then
    echo "##  This will delete the old STYNQ filesytem in $ROOT"
    echo "##"
fi

echo "##  A new STYNQ filesystem will be made at $ROOT from Debian sources."
echo "##"
echo "##  ctrl-d to continue, ctrl-c to abort."
echo "##"
echo "##########################################################################################"
cat

echo ""
echo ""
echo "######################################################################"
echo "##"
echo "##  Installing local packages needed to create the STYNQ filesystem."
echo "##"
echo "######################################################################"
echo ""

apt --yes install debootstrap debian-archive-keyring qemu-user-static

echo ""
echo "######################################################################"
echo "##"
echo "##  Creating a base Debian filesystem."
echo "##"
echo "######################################################################"
echo ""

#
# These umounts are in case the script is re-run after a failure.
# in that case, these may be left mounted.  They should not be mounted
# as we start again, so umount them just in case.  Failures in the
# umount are likely.
#
umount "$ROOT"/sys  >/dev/null 2>&1 || true
umount "$ROOT"/dev  >/dev/null 2>&1 || true
umount "$ROOT"/proc >/dev/null 2>&1 || true

umount "$ROOT"/sys  >/dev/null 2>&1 || true
umount "$ROOT"/dev  >/dev/null 2>&1 || true
umount "$ROOT"/proc >/dev/null 2>&1 || true


rm -rf "$ROOT"
mkdir "$ROOT"

if [ -f "${ROOT}_unconfigured_bootstrap.tgz" ]; then

    echo "Extracting bootstrap files from cached archive."
    echo "This avoids downloading most packages again from the internet."

    tar xpzf "${ROOT}_unconfigured_bootstrap.tgz"

else

    echo "Getting bootstrap files from Debian server."

    /usr/sbin/debootstrap --arch arm64 bookworm "$ROOT" http://deb.debian.org/debian 

    #
    # Enable running arm executables in the chroot on an x86 machine
    #
    cp /usr/bin/qemu-aarch64-static "$ROOT"/usr/bin/

    
    echo ""
    echo "######################################################################"
    echo "##"
    echo "##  Adding additional Debian packages for STYNQ."
    echo "##"
    echo "######################################################################"
    echo ""

    sudo mount /sys  "$ROOT"/sys  -o bind
    sudo mount /dev  "$ROOT"/dev  -o bind
    sudo mount /proc "$ROOT"/proc -o bind
    
    #
    # Install needed packages
    #    
    chroot "$ROOT" apt update
    chroot "$ROOT" apt --yes upgrade
    
    #chroot "$ROOT" bash -c "apt list | tee apt_packages.txt"
    
    chroot "$ROOT" apt --yes install locales
    echo "en_US.UTF-8 UTF-8" >> "$ROOT"/etc/locale.gen
    chroot "$ROOT" locale-gen
    
    chroot "$ROOT" apt --yes install fonts-freefont-ttf
    chroot "$ROOT" apt --yes install alsa-tools
    chroot "$ROOT" apt --yes install alsa-utils
    chroot "$ROOT" apt --yes install libfreetype6
    chroot "$ROOT" apt --yes install librsvg2-2
    chroot "$ROOT" apt --yes install libsysfs2
    chroot "$ROOT" apt --yes install i2c-tools
#    chroot "$ROOT" apt --yes install ifupdown
    chroot "$ROOT" apt --yes install psmisc
    chroot "$ROOT" apt --yes install libdrm-tests
    chroot "$ROOT" apt --yes install usbutils
    chroot "$ROOT" apt --yes install libwebsockets17
    chroot "$ROOT" apt --yes install openssh-server
    chroot "$ROOT" apt --yes install sudo
    chroot "$ROOT" apt --yes install network-manager
    chroot "$ROOT" apt --yes install net-tools
    chroot "$ROOT" apt --yes install netcat-traditional
    chroot "$ROOT" apt --yes install systemd-resolved
#    chroot "$ROOT" apt --yes install 
#    chroot "$ROOT" apt --yes install 
#    chroot "$ROOT" apt --yes install 

    chroot "$ROOT" apt clean

    sudo umount "$ROOT"/sys
    sudo umount "$ROOT"/dev
    sudo umount "$ROOT"/proc

    echo "Creating cache so that a different base system with"
    echo "different options doesn't load as many packages from the"
    echo "Internet again"

    tar czf "${ROOT}_unconfigured_bootstrap.tgz" $ROOT_NOPATH
fi

#
# Configure the system and make modifications based on the command-line options.
#

mv "$ROOT"/etc/resolv.conf "$ROOT"/etc/resolv.conf-bak
echo "nameserver 8.8.8.8" >"$ROOT"/etc/resolv.conf

sudo mount /sys  "$ROOT"/sys  -o bind
sudo mount /dev  "$ROOT"/dev  -o bind
sudo mount /proc "$ROOT"/proc -o bind


echo ""
echo "######################################################################"
echo "##"
echo "##  Ensuring that all packages are up-to-date."
echo "##"
echo "######################################################################"
echo ""

#
# Install needed packages
#
chroot "$ROOT" apt update
chroot "$ROOT" apt --yes upgrade


#
# No longer needed?
#
#cat > "$ROOT"/etc/fstab <<EOF
#proc             /proc         proc    defaults                 0    0
#sys              /sys          sysfs   defaults                 0    0
#EOF

echo ""
echo ""
echo "###########################################################################"
echo "##"
echo "##  Copying in kernel modules."
echo "##"
echo "###########################################################################"
echo ""

cp -a "$MODDIR" $ROOT/lib/


echo ""
echo ""
echo "###########################################################################"
echo "##"
echo "##  Setting the hostname to \"$HOSTNAME\""
echo "##"
echo "###########################################################################"
echo ""

echo $HOSTNAME >"$ROOT"/etc/hostname


echo ""
echo ""
echo "###########################################################################"
echo "##"
echo "##  Adding the user $USER"
echo "##"
echo "###########################################################################"
echo ""

chroot "$ROOT" bash -c "(echo $PASSWD; echo $PASSWD) | adduser --allow-bad-names --gecos \"STYNQy One\" $USER"

echo >>"$ROOT"/etc/sudoers "$USER ALL=(ALL:ALL) ALL"


echo ""
echo ""
echo "###########################################################################"
echo "##"
echo "##  Creating a script \"/opt/STYNQ/startup.sh\" that runs at boot as root."
echo "##"
echo "###########################################################################"
echo ""

mkdir -p "$ROOT"/opt/STYNQ/startup_scripts

cat >"$ROOT"/opt/STYNQ/startup.sh <<EOF
#!/bin/bash

export HOME=/opt/STYNQ
cd /opt/STYNQ

# Wait 1 second before redirecting output to the serial port,
# since in some cases getty will block it if it executes after
# this script.

sleep 1

exec >/dev/ttyPS0 2>&1

echo "#################################################################################################"
echo "##"
echo "##  Script /opt/STYNQ/startup.sh is running as root."
echo "##"
echo "#################################################################################################"
echo ""

SCRIPTS=\`ls startup_scripts\`

if [ "\$SCRIPTS" = "" ]; then
   echo ""
   echo "Normally, the startup action is to execute all scripts/programs put into"
   echo "the directory /opt/STYNQ/startup_scripts.  However, there are currently no"
   echo "scripts there."
   echo ""
else
   for script in \$SCRIPTS; do
      echo ""
      echo "Executing script /opt/STYNQ/startup_scripts/\$script."
      echo ""
      /opt/STYNQ/startup_scripts/\$script
  done
fi

sleep 1

EOF
chmod ugo+x "$ROOT"/opt/STYNQ/startup.sh


cat >"$ROOT"/etc/systemd/system/STYNQ_startup.service <<EOF
[Unit]
Description=STYNQ Startup Script

[Service]
ExecStart=/opt/STYNQ/startup.sh

[Install]
WantedBy=multi-user.target
EOF

chroot "$ROOT" systemctl enable STYNQ_startup



MACADDR=$(echo $HOSTNAME|md5sum|sed 's/^\(..\)\(..\)\(..\)\(..\)\(..\).*$/02:\1:\2:\3:\4:\5/')
UUID=`uuidgen`

echo ""
echo ""
echo "######################################################################################"
echo "##"
echo "##  Configuring Networking as follows:"
echo "##"
echo "##    Ethernet Address:  $MACADDR  -- This is generated using hostname"
echo "##                                             \"$HOSTNAME\" as a random seed"
echo "##"

if [ "$IP" = DHCP ]; then
    echo "##    Automatic configuration from network's DHCP server"
else
    echo "##    Static     IP $IP"
    if [ "$GATEWAY_IP" != "" ]; then
	echo "##    Gateway    IP $GATEWAY_IP"
    else
	echo "##    Gateway    IP NONE"
    fi
    if [ "$NAMESERVER_IP" != "" ]; then
	echo "##    Nameserver IP $NAMESERVER_IP"
    else
	echo "##    Nameserver IP NONE"
    fi    
fi
echo "##"
echo "######################################################################################"
echo ""

NMCONFIG="$ROOT/etc/NetworkManager/system-connections/eth0.nmconnection"

echo    >"$NMCONFIG"  "[connection]"
echo   >>"$NMCONFIG"  "id=eth0"
echo   >>"$NMCONFIG"  "uuid=$UUID"
echo   >>"$NMCONFIG"  "type=ethernet"
echo   >>"$NMCONFIG"  "autoconnect-priority=-999"
echo   >>"$NMCONFIG"  "interface-name=eth0"
echo   >>"$NMCONFIG"  "timestamp=1747419628"
echo   >>"$NMCONFIG"  ""
echo   >>"$NMCONFIG"  "[ethernet]"
echo   >>"$NMCONFIG"  "cloned-mac-address=$MACADDR"
echo   >>"$NMCONFIG"  ""

if [ "$IP" = DHCP ]; then
    echo   >>"$NMCONFIG"  "[ipv4]"
    echo   >>"$NMCONFIG"  "method=auto"
    echo   >>"$NMCONFIG"  ""
    echo   >>"$NMCONFIG"  "[ipv6]"
    echo   >>"$NMCONFIG"  "addr-gen-mode=default"
    echo   >>"$NMCONFIG"  "method=auto"
else
    echo   >>"$NMCONFIG"  "[ipv6]"
    echo   >>"$NMCONFIG"  "method=ignore"
    echo   >>"$NMCONFIG"  ""
    echo   >>"$NMCONFIG"  "[ipv4]"
    echo   >>"$NMCONFIG"  "method=manual"
    echo   >>"$NMCONFIG"  "address1=$IP"
    if [ "$GATEWAY_IP" != "" ]; then
	echo   >>"$NMCONFIG"  "gateway=$GATEWAY_IP"
    fi
    if [ "$NAMESERVER_IP" != "" ]; then
	echo   >>"$NMCONFIG"  "dns=$NAMESERVER_IP"
    fi
fi

chmod go-rwx "$NMCONFIG"


echo ""
echo "###########################################################################"
echo "##"
echo "##  Setting up dynamic device tree configuration."
echo "##"
echo "###########################################################################"
echo ""

cat >"$ROOT/etc/modules-load.d/dynamic_device_tree.conf" <<EOF
# /etc/modules-load.d/dynamic_device_tree.conf
# Load dtbocfg module that allows dynamic modifications to the
# device tree through /sys/kernel/config/device-tree
# in the configfs filesystem mounted there.
dtbocfg
EOF

echo ""
echo "###########################################################################"
echo "##"
echo "##  Adding user programs/scripts."
echo "##"
echo "###########################################################################"
echo ""

cp -a home_extras/. "$ROOT"/home/$USER/


echo ""
echo ""
echo "###########################################################################"
echo "##"
echo "##  Creating a tar archive of the STYNQ filesystem."
echo "##"
echo "###########################################################################"
echo ""

mv "$ROOT"/etc/resolv.conf-bak "$ROOT"/etc/resolv.conf

sudo umount "$ROOT"/sys
sudo umount "$ROOT"/dev
sudo umount "$ROOT"/proc

#BACKUP="$ROOT_NOPATH"_`date +%Y_%m_%d_%H.%M`.tgz

BACKUP=$ROOT_NOPATH.tgz

tar czf $BACKUP $ROOT_NOPATH

#rm -rf "$ROOT"

echo ""
echo ""
echo "###########################################################################################"
echo "##"
echo "##  Finished successfully with creation of a basic STYNQ OS."
echo "##  The output is in the tar file $BACKUP"
echo "##"
echo "##  To use this, an SD card needs to be formatted with two partitions, typically by"
echo "##  using the program \"fdisk\"."
echo "##"
echo "##  The first partition needs to be formatted to a VFAT filesystem, typicallly bu using"
echo "##  the program \"mkfs.vfat\".  Then it needs to have the files BOOT.BIN,"
echo "##  BOOT.SCR, and image.ub placed on it."
echo "##"
echo "##  The second partition needs to be formatted to EXT4, typicaly by using \"mkfs.ext4\"."
echo "##  Then it is mounted and equipped with the files just generated, as root, like so:"
echo "##"
echo "##     cd DIRETORY_OF_MOUNTED_SECOND_PARTITION"
echo "##     tar xzpf PATH/$BACKUP"
echo "##     mv $ROOT_NOPATH/* ."
echo "##     rmdir $ROOT_NOPATH"
echo "##"
echo "##  Then sync the disks with \"sync\" and cleanly unmount the partitions.  Remove"
echo "##  the SD card, set the board to boot from it, insert it in the board, and the"
echo "##  board should boot."
echo "##"
echo "##  You can connect to the board using its serial debug interface, or by ssh to its IP."
echo "##"
echo "##  Note that this is a basic STYNQ system.  For it to do anything, like configure"
echo "##  the PL, configure ADC/DAC  clocks, talk to the PL, etc., additional packages"
echo "##  are required.  Debian packages can be installed by running \"apt install package\"."
echo "##  on the booted FPGA board."
echo "##"
echo "###########################################################################################"
echo ""

