#!/bin/bash

VERSION=6.15


check_package()
{
    if [ -x "`which $1`" ]; then
	return
    fi
    
    if [ -f "$1" ]; then
	return
    fi

    if [ "`apt list $2 2>/dev/null | grep installed`" = ""  ]; then
	echo ""
	echo "Package containing \"$1\" is missing."
	echo ""
	echo "Install it before running this script."
	echo "Installation is usually via:"
	echo ""
	echo "  sudo apt install $2"
	echo ""
	exit 100;
    fi
}

check_packages()
{
    check_package make                         make
    check_package gcc                          build-essential
    check_package g++                          build-essential
    check_package /usr/include/ncurses.h       libncurses-dev
    check_package bison                        bison
    check_package flex                         flex
    check_package /usr/include/openssl/core.h  libssl-dev
    check_package /usr/include/libelf.h        libelf-dev
    check_package aarch64-linux-gnu-gcc        gcc-aarch64-linux-gnu
}


check_packages


TARFILE=linux-${VERSION}.tar.xz
KERNEL_DIR=linux-${VERSION}

if [ ! -f "$TARFILE" ]; then
    echo "Local tar file \"$TARFILE\" doesn't exist.  Fetching it."

    wget https://cdn.kernel.org/pub/linux/kernel/v6.x/$TARFILE
fi

CPUs=`nproc`

MAXJOBS=$(( 3 * CPUs / 4 - 1 ))

if [ $MAXJOBS -lt 1 ]; then
   MAXJOBS=1
fi

echo "Will run parallel build with $MAXJOBS threads on $CPUs cores."


#
# Get kernel files
#
if [ ! -d "$KERNEL_DIR" ]; then
    echo "Kernel directory missing.  Unarchiving it."
    tar xf "$TARFILE"
fi

rm -rf out
mkdir out

CROSS=

LOCAL_ARCH=`arch`

if [ "$LOCAL_ARCH" != aarch64 ]; then

    CROSS=aarch64-linux-gnu-

fi

gcc_compiler=`which "${CROSS}gcc"`

if [ "$gcc_compiler" = "" ]; then

    echo "GCC Compiler ${CROSS}gcc is missing."
    exit 20
fi

pushd "$KERNEL_DIR"

#
# Set compile defaults.  Not sure what else this does.
#

#make ARCH=arm64 CROSS_COMPILE=aarch64-linux-gnu- -j$MAXJOBS defconfig


#
# Configure my own options.  I have them in STYNQ.config.  Originally created with
# make ARCH=arm64 menuconfig
#

#make ARCH=arm64 menuconfig

CONFFILE=inputs/STYNQ-${VERSION}.config

if [ ! -f "../$CONFFILE" ]; then
    echo ""
    echo "Missing config file \"$CONFFILE\"."
    echo "Copy another one, configure the kernel, and copy"
    echo "the .config file from the kernel to this"
    echo "name to be able to rerun this script."
    echo ""
    exit 20
fi

cp ../$CONFFILE .config

#
# Build the kernel.  
#

make ARCH=arm64 CROSS_COMPILE=aarch64-linux-gnu- -j$MAXJOBS all

#
# Copy the two files needed to kexec into it.
#

#cp arch/arm64/boot/dts/xilinx/zynqmp-zcu111-revA.dtb ..
#gzip <vmlinux >../vmlinux.gz

#
# Kernel can then be kexec'ed to with
#
#    sudo kexec --dtb=/home/ross/kernel/zynqmp-zcu111-revA.dtb /home/ross/kernel/vmlinux.gz
#
#

cp arch/arm64/boot/Image.gz ../out/

if [ "$CROSS" = "" ]; then
    echo ""
    echo "Installing modules"
    echo ""
    sudo make modules_install
else
    make ARCH=arm64 CROSS_COMPILE=aarch64-linux-gnu- INSTALL_MOD_PATH=../out -j$MAXJOBS modules_install
    echo ""
    echo "Cross-compiling; installed modules in out/modules."
    echo "Copy out/lib to /lib on target machine."
    echo ""
fi

popd

echo "********************************************************************************"
echo "**"
echo "**  Building extra modules not in the mainline kernel."
echo "**"
echo "********************************************************************************"

echo "  Building dtbocfg, to dynamically modify the device tree."

if [ -f dtbocfg/dtbocfg.ko ]; then

    echo "  dtbocfg already built"

else
    rm -rf dtbocfg

    if [ ! -f dtbocfg.tgz ]; then
    
	git clone https://github.com/ikwzm/dtbocfg.git
	tar czf dtbocfg.tgz dtbocfg
	
    else

	tar xzf dtbocfg.tgz

    fi

    (cd dtbocfg; make ARCH=arm64 CROSS_COMPILE=aarch64-linux-gnu- KERNEL_SRC=../"$KERNEL_DIR" INSTALL_MOD_PATH=../out -j$MAXJOBS )
    
fi

echo "  installing dtbocfg"
(cd dtbocfg; make ARCH=arm64 CROSS_COMPILE=aarch64-linux-gnu- KERNEL_SRC=../"$KERNEL_DIR" INSTALL_MOD_PATH=../out -j$MAXJOBS modules_install)


echo "********************************************************************************"
echo "**"
echo "**  All finished."
echo "**"
echo "**    Kernel is in out/Image.gz, and must be incorporated into the boot files."
echo "**    modules are in out/lib, and must be copied into /lib in the filesystem."
echo "**"
echo "********************************************************************************"
