
/*
 * Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., http://www.fsf.org/about/contact/
 *
 */


#ifndef EXPRESSION_HH
#define EXPRESSION_HH

#include "readstream.hh"
#include "writestream.hh"
#include "String.hh"

class expression;
struct expressionOperator;

#define AMBIGUOUS_BITS -1


typedef long bits;

struct Value
{
  bool valid;     // Not valid if an expression contains an undefined variable, for example
  int  numbits;   // Set to AMBIGUOUS_BITS if bits aren't defined in the constant.  Set to zero if an expression has internal bit inconsistencies
  long value;
  int  base;      // base of the original constant.  10=decimal, 16=hex, 8=octal, 2=binary

  void setbits(int numbits);

  Value(long value=0) { valid = true; numbits=AMBIGUOUS_BITS; base=10; this->value = value; }
};

typedef int variable_check_function(char* s, void* data, readstream& r, bool clock_ok);
typedef Value expression_value_function(expression* e, 
					expressionOperator* oe,
					expressionOperator* uoe);


//
// previous_expression is set only for binary (or higher) operators.
//
typedef expression* expression_parse_function(readstream&              r, 
					      variable_check_function* f, 
					      expressionOperator*      ops,
					      expressionOperator*      uops,
					      void*                    data,
					      expressionOperator*      this_operator,
					      expression*              previous_expression);

typedef String expression_to_string_function(expression* e, 
					     expressionOperator*      ops,
					     expressionOperator*      uops);



const int EXPRESSION_UNDEFINED                 = 0x00000000;
const int EXPRESSION_NULL                      = 0xAAAAAAAA;

//
// Standard UNARY expressionOperator types.  These can be extended.
//
const int EXPRESSION_CONSTANT                  = 0x80000001;
const int EXPRESSION_VARIABLE                  = 0x80000002;
const int EXPRESSION_OPERATOR_BIT_NOT          = 0x80000003;
const int EXPRESSION_OPERATOR_LOGICAL_NOT      = 0x80000004;
const int EXPRESSION_OPERATOR_UNARY_MINUS      = 0x80000005;
const int EXPRESSION_OPERATOR_UNARY_PLUS       = 0x80000006;
const int EXPRESSION_OPERATOR_BIT_REDUCE_OR    = 0x80000007;
const int EXPRESSION_OPERATOR_BIT_REDUCE_AND   = 0x80000008;
const int EXPRESSION_OPERATOR_BIT_REDUCE_XOR   = 0x80000009;
const int EXPRESSION_BIT_SELECTION             = 0x8000000A;
const int EXPRESSION_QUESTION_MARK             = 0x80000080;
const int EXPRESSION_CONCATENATION             = 0x80000081;
const int EXPRESSION_BEGINGROUP                = 0x80000100;

//
// Standard BINARY expressionOperator types.  These can be extended.
//
const int EXPRESSION_TERMINATOR                 = 0xFFFFFFFF;
const int EXPRESSION_OPERATOR_ADD               = 0x00000001;
const int EXPRESSION_OPERATOR_SUBTRACT          = 0x00000002;
const int EXPRESSION_OPERATOR_BIT_AND           = 0x00000003;
const int EXPRESSION_OPERATOR_BIT_OR            = 0x00000004;
const int EXPRESSION_OPERATOR_BIT_NAND          = 0x00000005;
const int EXPRESSION_OPERATOR_BIT_NOR           = 0x00000006;
const int EXPRESSION_OPERATOR_BIT_XOR           = 0x00000007;
const int EXPRESSION_OPERATOR_BIT_XNOR          = 0x00000008;
const int EXPRESSION_OPERATOR_LOGICAL_AND       = 0x00000009;
const int EXPRESSION_OPERATOR_LOGICAL_OR        = 0x0000000A;
const int EXPRESSION_OPERATOR_LOGICAL_NAND      = 0x0000000B;
const int EXPRESSION_OPERATOR_LOGICAL_NOR       = 0x0000000C;
const int EXPRESSION_OPERATOR_LOGICAL_XOR       = 0x0000000D;
const int EXPRESSION_OPERATOR_LOGICAL_XNOR      = 0x0000000E;
const int EXPRESSION_OPERATOR_MULTIPLY          = 0x0000000F;
const int EXPRESSION_OPERATOR_DIVIDE            = 0x00000010;
const int EXPRESSION_OPERATOR_MODULO            = 0x00000011;
const int EXPRESSION_OPERATOR_SHIFT_LEFT        = 0x00000012;
const int EXPRESSION_OPERATOR_SHIFT_RIGHT       = 0x00000013;
const int EXPRESSION_OPERATOR_LESS_THAN         = 0x00000014;
const int EXPRESSION_OPERATOR_LESS_OR_EQUAL     = 0x00000015;
const int EXPRESSION_OPERATOR_EQUAL             = 0x00000016;
const int EXPRESSION_OPERATOR_GREATER_THAN      = 0x00000017;
const int EXPRESSION_OPERATOR_GREATER_OR_EQUAL  = 0x00000018;
const int EXPRESSION_OPERATOR_NOT_EQUAL         = 0x00000019;
const int EXPRESSION_OPERATOR_ASSIGNMENT        = 0x0000001A;
const int EXPRESSION_REPLICATION                = 0x0000001B;
const int EXPRESSION_ENDGROUP                   = 0x00000101;

//
// This goes in an array of expression operator pointers
// used to parse or print or evaluate the expression.
//
struct expressionOperator
{
  int type;
  int priority;
  const char* token;
  expression_value_function*     function;
  expression_parse_function*     parse_function;
  expression_to_string_function* string_function;
};

//
// TEMP
//
extern expressionOperator binary_STAMP_operators[];
extern expressionOperator unary_STAMP_operators[];
void print_expression(expression* e, 
		      expressionOperator*      ops,
		      expressionOperator*      uops);
//
// ENDTEMP
//

class expression
{
private:
  void operator=(expression&); // Not actually defined.  Just here to disable.
  expression(expression& e);

public:

  int    type;

  Value  value;  // For Constant expressions.

  String name;   // For Variable expressions.
  void*  data;   // Holds the data pointer passed into parse_expression

  expression* exp1; // For Operator expressions.
  expression* exp2;
  expression* exp3; // exp3 only used for question-mark operators

  // for expressions that take many args.  Currently only concatenation
  int          exp_count;
  expression** exp_n;

  expression() { exp1=0; exp2=0; exp3=0; exp_n=0; type=0; }
  ~expression() { if (exp1) delete exp1; if (exp2) delete exp2; if (exp3) delete exp3; if (exp_n) delete exp_n; }
};

//
// The data value is an argument passed to the variable check function
// to help it in finding variable names.
//
expression* parse_expression(readstream&              r, 
			     variable_check_function* f, 
			     expressionOperator*      ops,
			     expressionOperator*      uops,
			     void*                    data,
			     bool                     clock_ok=false,
			     int                      priority=0);

expressionOperator* getOperator(int type, expressionOperator* oe);

Value getValue(expression* e,
	       expressionOperator* oe, 
	       expressionOperator* uoe);


String expressionToString(expression* e, 
			  expressionOperator*      ops,
			  expressionOperator*      uops);

void stuff_variables(expression* e, 
		     expressionOperator*      ops,
		     expressionOperator*      uops);

void evaluate_outer_leaf(expression* e, 
			 expressionOperator*      ops,
			 expressionOperator*      uops);


void write(writestream& w,  
	   expression* e,
	   expressionOperator*      ops,
	   expressionOperator*      uops);

void write_nospaces(writestream& w,  
		    expression* e,
		    expressionOperator*      ops,
		    expressionOperator*      uops);

void write(char* bigbuff,
	   int bigbuffsize,
	   expression* e,
	   expressionOperator*      ops,
	   expressionOperator*      uops);

void write_nospaces(char* bigbuff,
		    int bigbuffsize,
		    expression* e,
		    expressionOperator*      ops,
		    expressionOperator*      uops);

#endif
