#!/bin/bash -e

#
#  This script accepts "uncertainty" as an argument.  Why?
#  This is because FPGA placement is random.  It is repeatable, so that
#  if nothing changes the same random result is obtained.  Still, it is
#  random, so that if anything does change placement may be entirely
#  different.  One thing that can change is the clock uncertainty.
# 
#  By changing the clock uncertainty by what should be inconsequential 
#  amounts, placement starts with different random seeds.  This can produce
#  results that are hundreds of MHz different in Fmax.
#
#  Experimentally, this appears to be at least as good as changing
#  compile strategies or other things that can be done to obtain
#  timing closure.  And it is better, because it is entirely automated.
#
#  This script can be called multiple times with multiple values of
#  uncertainty, in an entirely automated way, to close timing.  So
#  you run a script overnight, which calls this script multiple times
#  each with different uncertainty, and one of those may meet timing,
#  without any extra work on an FPGA engineer's part.
#
#  This will of course not always work.  However, when it does it can
#  be an incredible time saver.  When it does not, the process
#  produces many runs with different timing results, and seeing what
#  are common bottlenecks across the multiple runs can help narrow
#  down what needs to be done to meet timing.
#
#  The input "DESIGN" just specifies the basename of the output files.
#

DESIGN=RFSoC4x2_BxBDemo1

uncertainty="$1"

if [ "$uncertainty" = "" ]; then
    uncertainty=0.000
fi

ucheck=`echo $uncertainty | sed 's/0[.][0-9]*//'`

if [ "$2" != "" -o "$ucheck" != "" ]; then
    echo ""
    echo "  Usage:  compile_design.sh [uncertainty]"
    echo ""
    echo "    Uncertainty is effectively a placement random seed."
    echo "    Uncertainty should be of the form \"0.001\", and"
    echo "    be less than 1.000.  If uncertainty is omitted, it"
    echo "    will be \"0.000\"."
    exit 20
fi

dir=${DESIGN}_$uncertainty


vivado=`which vivado`

if [ ! -f "$vivado" ]; then
    echo "Vivado not found.  Set up build environment."
    exit 20
fi


vivado_version=`vivado -version 2>/dev/null | head -1 | sed 's/[ ][^ ]*$//'`

if [ "$vivado_version" != "vivado v2024.2" ]; then
    echo "This script was tested with vivado v2024.2."
    echo "Instead found vivado version \"$vivado_version\"."
    echo "echo This may cause issues, or may not."
    sleep 3
fi


if [ -d "$dir" ]; then
    echo ""
    echo "Directory \"$dir\" already exists.  Aborting."
    echo ""
    exit 20
fi

mkdir -p "$dir"

exec 3>&1 >"$dir"/compile_log.txt 2>&1
set -x

cp -r inputs/* "$dir"/


cat >"$dir"/compile_design.tcl <<EOF

fconfigure stdout -buffering none
fconfigure stderr -buffering none

set origin_dir_loc "./BxB_Demo1"

source ./BxB_Demo1.tcl

cd BxB_Demo1

#open_project BxB_Demo1/BxB_Demo1.xpr

update_compile_order -fileset sources_1

generate_target all [get_files  design_1.bd]

reset_runs synth_1



write_hw_platform -fixed -force -file ../${DESIGN}.xsa

set_property STEPS.OPT_DESIGN.ARGS.DIRECTIVE NoBramPowerOpt [get_runs impl_1]

#
# pin_constraints.xdc needs to be last in the constraint list, since it
# defines clock groups to be asynchronous, and earlier XDC files create
# those clock groups.
#
set_property PROCESSING_ORDER LATE [get_files pin_constraints.xdc]
report_compile_order -constraints

#
# Change the uncertainty, just for placement.  Keep the original
# file modification time, or else Vivado thinks the file has changed
# when it is done, and thinks the design is out of date.
#
exec mv ../pin_constraints.xdc     ../pin_constraints.xdc-bak
exec cp ../pin_constraints.xdc-bak ../pin_constraints.xdc
set cfile [open ../pin_constraints.xdc a]
puts \$cfile "set_clock_uncertainty $uncertainty \\[get_clocks\\] -setup"
close \$cfile

launch_runs impl_1 -to_step place_design -jobs 10
wait_on_run impl_1

#
# Change the uncertainty back for routing and final
# timing.  This prevents the uncertainty from distorting
# final timing results.
#
exec mv ../pin_constraints.xdc-bak ../pin_constraints.xdc

launch_runs impl_1 -to_step write_bitstream -jobs 10
wait_on_run impl_1
open_run impl_1

write_verilog -force ../${DESIGN}_netlist.v

EOF


echo "Running Vivado to compile the project." 1>&3

( cd "$dir"; \time -f "%U" -o vivado_run_time_seconds.txt vivado -mode tcl <./compile_design.tcl ) 1>&3 2>&3


BITFILE=`find "$dir" -name '*.bit'`

if [ ! -f "$BITFILE" ]; then
    echo "" 1>&3
    echo "ERROR: Bitfile is missing.  Compilation error must have occured." 1>&3
    echo "" 1>&3
    exit 20
fi

cp "$BITFILE" "$dir"/"$DESIGN".bit

cat >>"$dir"/bootgen.bif <<EOF
bootgen:
{
  $DESIGN.bit
}
EOF

(cd "$dir"; bootgen -image bootgen.bif -w -arch zynqmp -process_bitstream bin)

mv "$dir"/${DESIGN}.bit.bin "$dir"/${DESIGN}.bin 

HWHFILE=`find "$dir" -name '*.hwh'`

if [ -f "$HWHFILE" ]; then
    cp "$HWHFILE" "$dir"/"$DESIGN.hwh"
fi


if [ ! -f "$dir"/"${DESIGN}.bin" ]; then
    echo "" 1>&3
    echo "ERROR:  bootgen didn't create binary file for loading from Linux." 1>&3
    echo "" 1>&3
    exit 20
fi

exec 1>&3

echo ""
echo "Project compilation in \"$dir\" directory complete, creating these files:"
echo ""
echo "    Software Architecture, for building maching software:  ${DESIGN}.xsa"
echo "    Hardware Handoff, hardware settings from the XSA:      ${DESIGN}.hwh"
echo "    netlist, the synthesized and optimized PL verilog:     ${DESIGN}_netlist.v"
echo "    FPGA PL bitfile, to program the FPGA:                  ${DESIGN}.bit"
echo "    FPGA PL binfile, to program the FPGA from Linux:       ${DESIGN}.bin"
echo ""

exit 0
