
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

`timescale 1ps / 1ps

`define w assign
`define raxis always@(posedge OUT_AXIS_aclk)
`define raxi  always@(posedge S_AXI_ACLK)

//
// AXIS_SAMPLES_PER_CLOCK must be 1, 2, 4, or 8.
//
module swept_spread_phase_generator
  #(
    // Capture parameters
    parameter AXIS_SAMPLES_PER_CLOCK       = 8,
    parameter AXIS_BITS_PER_SAMPLE         = 32,
    parameter AXIS_BUS_WIDTH               = AXIS_BITS_PER_SAMPLE * AXIS_SAMPLES_PER_CLOCK,
    
    // ARM interface parameters
    parameter AXI_BITS_PER_WORD            = 32,
    parameter AXI_BYTES_PER_WORD           = AXI_BITS_PER_WORD/8,
    parameter NUM_REGISTERS                = 16,  // Must be greater than 1
    parameter AXI_BYTE_ADDR_WIDTH          = $clog2( NUM_REGISTERS * AXI_BYTES_PER_WORD )
    )
   (
    // Data collection options, muxed to output

    (* X_INTERFACE_INFO = "xilinx.com:signal:clock:1.0 OUT_AXIS_aclk CLK" *)
    (* X_INTERFACE_PARAMETER = "ASSOCIATED_BUSIF OUT_AXIS, ASSOCIATED_RESET OUT_AXIS_aresetn" *)
    input                                OUT_AXIS_aclk,

    (* X_INTERFACE_INFO = "xilinx.com:signal:reset:1.0 OUT_AXIS_aresetn RST" *)
    input                                OUT_AXIS_aresetn,
    
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_AXIS TDATA" *)   output reg [AXIS_BUS_WIDTH-1:0]       OUT_AXIS_tdata,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_AXIS TREADY" *)  input                                 OUT_AXIS_tready,
    (* X_INTERFACE_INFO = "xilinx.com:interface:axis:1.0 OUT_AXIS TVALID" *)  output reg                            OUT_AXIS_tvalid,

    output [4:0]                         sine_shift_right_o,
    
    //
    // This is the interface to the ARM processor
    //
    input                                S_AXI_ACLK,
    input                                S_AXI_ARESETN,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_AWADDR,
    input [2 : 0]                        S_AXI_AWPROT,
    input                                S_AXI_AWVALID,
    output reg                           S_AXI_AWREADY,
    input [AXI_BITS_PER_WORD-1 : 0]      S_AXI_WDATA,
    input [AXI_BYTES_PER_WORD-1 : 0]     S_AXI_WSTRB,
    input                                S_AXI_WVALID,
    output reg                           S_AXI_WREADY,
    output reg [1 : 0]                   S_AXI_BRESP,
    output reg                           S_AXI_BVALID,
    input                                S_AXI_BREADY,
    input [AXI_BYTE_ADDR_WIDTH-1 : 0]    S_AXI_ARADDR,
    input [2 : 0]                        S_AXI_ARPROT,
    input                                S_AXI_ARVALID,
    output reg                           S_AXI_ARREADY,
    output reg [AXI_BITS_PER_WORD-1 : 0] S_AXI_RDATA,
    output reg [1 : 0]                   S_AXI_RRESP,
    output reg                           S_AXI_RVALID,
    input                                S_AXI_RREADY
);
   

   genvar                                i, j;

   initial OUT_AXIS_tvalid = 0;
   initial OUT_AXIS_tdata  = 0;
   
   //
   // Registers coming from other clock domain
   //
   // Two registers after a clock transition are needed to avoid metastability.  Truthfully it's not a
   // big issue since these will only change once in a blue moon, and metastability will *eventually*
   // die out.  Also, this isn't sufficient to make sure that all bits in the register update at the
   // same time.  Still, it's something in the right direction.
   //
   
   /**/                                                                          reg [31:0]  reg_freq_ramp_ar         = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  freq_ramp_sr             = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [31:0]  freq_ramp_srr            = 0;

   /**/                                                                          reg [31:0]  reg_freq_min_ar          = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  freq_min_sr              = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [31:0]  freq_min_srr             = 0;

   /**/                                                                          reg [31:0]  reg_freq_max_ar          = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  freq_max_sr              = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [31:0]  freq_max_srr             = 0;

   /**/                                                                          reg [31:0]  reg_spread_period_ar     = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  spread_period_sr         = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [31:0]  spread_period_srr        = 0;

   /**/                                                                          reg [31:0]  reg_spread_code_ar       = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  spread_code_sr           = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [31:0]  spread_code_srr          = 0;

   /**/                                                                          reg [31:0]  reg_sine_shift_right_ar  = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  sine_shift_right_sr      = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [31:0]  sine_shift_right_srr     = 0;

   /**/                                                                          reg [31:0]  reg_freq_fract_ar = 0;  // AXI clock
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)  reg [31:0]  freq_fract_sr     = 0;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                            reg [31:0]  freq_fract_srr    = 0;
   
   `raxis  freq_ramp_sr           <=  reg_freq_ramp_ar;
   `raxis  freq_ramp_srr          <=  freq_ramp_sr;

   `raxis  freq_min_sr            <=  reg_freq_min_ar;
   `raxis  freq_min_srr           <=  freq_min_sr;

   `raxis  freq_max_sr            <=  reg_freq_max_ar;
   `raxis  freq_max_srr           <=  freq_max_sr;

   `raxis  spread_period_sr       <=  reg_spread_period_ar;
   `raxis  spread_period_srr      <=  spread_period_sr;

   `raxis  spread_code_sr         <=  reg_spread_code_ar;
   `raxis  spread_code_srr        <=  spread_code_sr;

   `raxis  sine_shift_right_sr    <=  reg_sine_shift_right_ar;
   `raxis  sine_shift_right_srr   <=  sine_shift_right_sr;
   `w      sine_shift_right_o      =  sine_shift_right_srr[4:0];

   `raxis  freq_fract_sr          <=  reg_freq_fract_ar;
   `raxis  freq_fract_srr         <=  freq_fract_sr;


   //////////////////////////////////////////////////////////////////////////////////////////////////////
   //
   // Main code here, on AXIS clock.  Control registers after clock transition are 
   //
   //    freq_ramp_srr     freq_min_srr      freq_max_srr    spread_period_srr    spread_code_srr   sine_shift_right_srr   freq_fract_srr
   //


   //
   // PN Generator
   //
   wire                                  pn_out_w;
   reg                                   pn_enable_sr = 0;
   reg signed [31:0]                     spread_count_sr = 0;

   `raxis spread_count_sr <= (spread_count_sr==  0)  ?  spread_period_srr   :
                             (spread_count_sr== -1)  ?  spread_period_srr   :
                             /**/                       spread_count_sr-1   ;

   // Checking for 1 here allows us to turn off spreading by setting spread_period to 0.
   // Checking for -1 here allows us to turn spreading on to maximum by setting to -1
   `raxis pn_enable_sr <= (spread_count_sr==1) || (spread_count_sr==-1);
   
   pn_generator
     #(
       .PN_CODE            ( 0                 ),
       .DYNAMIC_PN_CODE    ( 1                 )
       )
   pn
     (
      .clk                 ( OUT_AXIS_aclk     ),
      .enable_i            ( pn_enable_sr      ),
      .pn_code_i           ( spread_code_srr   ),
      .out_o               ( pn_out_w          )
      );


   //
   // Frequency register
   //
   reg [31:0]                            freq_master_sr          = 0;
   reg                                   sweep_direction_up_sr   = 1;
   reg signed [31:0]                     current_freq_ramp_sr    = 0;
   reg [31:0]                            candidate_next_freq_sr  = 0;
   reg [31:0]                            candidate_next_freq_srr = 0;
   reg                                   force_sweep_up_sr       = 0;
   reg                                   force_sweep_down_sr     = 0;
   reg [9:0]                             update_count_sr         = 0;
   reg                                   update_time_sr          = 0;

   `raxis  sweep_direction_up_sr    <=  force_sweep_up_sr    ?  1                      :
                                        force_sweep_down_sr  ?  0                      :
                                        /**/                    sweep_direction_up_sr  ;
   
   `raxis  current_freq_ramp_sr     <=  sweep_direction_up_sr ? freq_ramp_srr : -freq_ramp_srr;

   `raxis  candidate_next_freq_sr   <=  current_freq_ramp_sr + freq_master_sr;
   `raxis  candidate_next_freq_srr  <=  candidate_next_freq_sr;

   `raxis  force_sweep_up_sr        <=  (candidate_next_freq_sr <= freq_min_srr);
   `raxis  force_sweep_down_sr      <=  (candidate_next_freq_sr >= freq_max_srr);
   
   `raxis  freq_master_sr           <=  (!update_time_sr)    ?   freq_master_sr          :
                                        force_sweep_up_sr    ?   freq_min_srr            :
                                        force_sweep_down_sr  ?   freq_max_srr            :
                                        /**/                     candidate_next_freq_srr ;

   `raxis  update_count_sr          <= update_count_sr + 1;

   `raxis  update_time_sr           <= (update_count_sr==0);
   

   //
   // Frequency multiples.  DONT_TOUCH here are to prevent optimizations that made these
   // use the same registers, which resulted in high fanouts.
   //
   wire [63:0] 				 freq_base_w;
   (* DONT_TOUCH = "YES" *) reg  [63:0]  freq_1_r;
   (* DONT_TOUCH = "YES" *) reg  [63:0]  freq_2_r;
   (* DONT_TOUCH = "YES" *) reg  [63:0]  freq_3_r;
   (* DONT_TOUCH = "YES" *) reg  [63:0]  freq_4_r;
   (* DONT_TOUCH = "YES" *) reg  [63:0]  freq_8_r;
   (* DONT_TOUCH = "YES" *) reg  [63:0]  freq_16_r;

   reg [0:AXIS_SAMPLES_PER_CLOCK] [63:0] freq_sr = 0;

   `w freq_base_w = { freq_master_sr, freq_fract_srr };
   
   `raxis freq_1_r  <= freq_base_w;
   `raxis freq_2_r  <= freq_base_w<<1;
   `raxis freq_3_r  <= freq_base_w + (freq_base_w<<1);
   `raxis freq_4_r  <= freq_base_w<<2;
   `raxis freq_8_r  <= freq_base_w<<3;
   `raxis freq_16_r <= freq_base_w<<4;

   generate   
      if(AXIS_SAMPLES_PER_CLOCK>=0)  `raxis freq_sr[0]  <= 0;
      if(AXIS_SAMPLES_PER_CLOCK>=1)  `raxis freq_sr[1]  <= freq_1_r;
      if(AXIS_SAMPLES_PER_CLOCK>=2)  `raxis freq_sr[2]  <= freq_2_r;
      if(AXIS_SAMPLES_PER_CLOCK>=3)  `raxis freq_sr[3]  <= freq_3_r;
      if(AXIS_SAMPLES_PER_CLOCK>=4)  `raxis freq_sr[4]  <= freq_4_r;
      if(AXIS_SAMPLES_PER_CLOCK>=5)  `raxis freq_sr[5]  <= freq_1_r + freq_4_r;
      if(AXIS_SAMPLES_PER_CLOCK>=6)  `raxis freq_sr[6]  <= freq_2_r + freq_4_r;
      if(AXIS_SAMPLES_PER_CLOCK>=7)  `raxis freq_sr[7]  <= freq_8_r - freq_1_r;
      if(AXIS_SAMPLES_PER_CLOCK>=8)  `raxis freq_sr[8]  <= freq_8_r;
      if(AXIS_SAMPLES_PER_CLOCK>=9)  `raxis freq_sr[9]  <= freq_8_r + freq_1_r;
      if(AXIS_SAMPLES_PER_CLOCK>=10) `raxis freq_sr[10] <= freq_8_r + freq_2_r;
      if(AXIS_SAMPLES_PER_CLOCK>=11) `raxis freq_sr[11] <= freq_8_r + freq_3_r;
      if(AXIS_SAMPLES_PER_CLOCK>=12) `raxis freq_sr[12] <= freq_8_r + freq_4_r;
      if(AXIS_SAMPLES_PER_CLOCK>=13) `raxis freq_sr[13] <= freq_16_r - freq_3_r;
      if(AXIS_SAMPLES_PER_CLOCK>=14) `raxis freq_sr[14] <= freq_16_r - freq_2_r;
      if(AXIS_SAMPLES_PER_CLOCK>=15) `raxis freq_sr[15] <= freq_16_r - freq_1_r;
      if(AXIS_SAMPLES_PER_CLOCK>=16) `raxis freq_sr[16] <= freq_16_r;
   endgenerate
   //
   // Phase accumulation
   //
   
   // phase register is 64-bit, but divided into an upper and a lower part.  The lower part (fract)
   // is summed separately and its carry registered before adding into the upper part.  This delays
   // its correction one clock cycle, which matters not at all.

   reg [31:0]                            phase_master_sr         = 0;
   reg [31:0] 				 phase_master_fract_sr   = 0;
   wire [32:0] 				 phase_master_fract_w;
   reg                                   phase_master_fract_overflow_r = 0;

   `w     phase_master_fract_w            =  phase_master_fract_sr + freq_sr[AXIS_SAMPLES_PER_CLOCK][31:0];
   `raxis phase_master_fract_sr          <=  phase_master_fract_w;
   `raxis phase_master_fract_overflow_r  <=  phase_master_fract_w[32];
   
   `raxis phase_master_sr <= phase_master_sr + freq_sr[AXIS_SAMPLES_PER_CLOCK][63:32] + phase_master_fract_overflow_r;

   //
   // Output
   //
   reg 					 zero_frequency_r = 0;

   `raxis zero_frequency_r <= (freq_base_w==0);

   generate
      for(i=0; i<AXIS_SAMPLES_PER_CLOCK; i=i+1)
	begin
	   reg [31:0] phase_sr;
	   (* DONT_TOUCH = "TRUE" *) reg [31:0] phase_master_srr;
	   reg [31:0] freq_srr;

	   `raxis phase_master_srr <= phase_master_sr;
	   `raxis freq_srr <= freq_sr[i][63:32] + { pn_out_w, 31'b0 };
	   `raxis phase_sr <= phase_master_srr + freq_srr; 
	   `raxis OUT_AXIS_tdata[ i * AXIS_BITS_PER_SAMPLE +: AXIS_BITS_PER_SAMPLE ]  <=  zero_frequency_r ?  32'h40000000 : phase_sr;
	end
   endgenerate
   
   `raxis OUT_AXIS_tvalid  <=  1'b1;


   //
   // End main code
   //
   //////////////////////////////////////////////////////////////////////////////////////////////////////


     
   //
   // Handle READs/WRITEs from the processor on the AXI bus.
   //    

   localparam                              WORD_ADDR_BITS         = $clog2(NUM_REGISTERS);   
   localparam                              BYTE_ADDR_BITS_IN_WORD = $clog2(AXI_BYTES_PER_WORD);
   
   reg [WORD_ADDR_BITS-1 : 0]              write_address     = 0;
   reg [AXI_BITS_PER_WORD-1:0]             write_data        = 0;
   wire                                    write_enable;
   wire                                    write_enable_w;
   reg                                     write_enable_r    = 0;
   wire                                    write_done;
   
   reg                                     read_enable       = 0;
   reg [WORD_ADDR_BITS-1 : 0]              read_address      = 0;
   wire [AXI_BITS_PER_WORD-1:0]            read_data;
   wire                                    read_done;

   //
   // The write interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_AWREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_AWREADY && S_AXI_AWVALID)   ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_AWREADY  ;

   `raxi S_AXI_WREADY  <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_WREADY && S_AXI_WVALID)     ?  1'b0           :
                          (S_AXI_BREADY  && S_AXI_BVALID)    ?  1'b1           :
                          /* default */                         S_AXI_WREADY   ;

   `raxi S_AXI_BVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          write_done                         ?  1'b1           :
                          (S_AXI_BREADY && S_AXI_BVALID)     ?  1'b0           :
                          /* default */                         S_AXI_BVALID   ;

   `raxi S_AXI_BRESP   <= 2'b0;

   //
   // The read interface could be faster, if pipelined.  This is slightly slow, but reliable.
   //
   `raxi S_AXI_ARREADY <= (!S_AXI_ARESETN)                   ?  1'b1           :
                          (S_AXI_ARREADY && S_AXI_ARVALID)   ?  1'b0           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b1           :
                          /* default */                         S_AXI_ARREADY  ;
   

   `raxi S_AXI_RVALID  <= (!S_AXI_ARESETN)                   ?  1'b0           :
                          read_done                          ?  1'b1           :
                          (S_AXI_RREADY  && S_AXI_RVALID)    ?  1'b0           :
                          /* default */                         S_AXI_RVALID   ;

   `raxi S_AXI_RRESP   <= 2'b0;
   
   `raxi S_AXI_RDATA   <= (!S_AXI_ARESETN)                   ?  0              : 
                          read_done                          ?  read_data      :
                          /* default */                         S_AXI_RDATA    ;


   //
   // Define write addresses, data, and enables
   //
   `w write_enable_w          = !S_AXI_WREADY && !S_AXI_AWREADY;
   `raxi write_enable_r       = (!S_AXI_ARESETN) ? 0 : write_enable_w;
   `w write_enable            = write_enable_w && !write_enable_r;   // Only pulse write enable for a single clock, regardless of BREADY
   
   `raxi write_address       <= (!S_AXI_ARESETN)                  ?  0                                                              :
                                (S_AXI_AWREADY && S_AXI_AWVALID)  ?  S_AXI_AWADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                        write_address                                                  ;
   
   `raxi write_data          <= (!S_AXI_ARESETN)                  ?  0                :
                                (S_AXI_WREADY && S_AXI_WVALID)    ?  S_AXI_WDATA      :
                                /* default */                        write_data       ;

   `w write_done              = write_enable;  // Write done immediately after address and data are both present
   

   //
   // Define read addresses, data, and enables
   //
   `raxi read_address        <= (!S_AXI_ARESETN)               ?  0                                                              : 
                                S_AXI_ARREADY & S_AXI_ARVALID  ?  S_AXI_ARADDR[WORD_ADDR_BITS-1+BYTE_ADDR_BITS_IN_WORD:BYTE_ADDR_BITS_IN_WORD]  :
                                /* default */                     read_address                                                   ;
   
   `raxi read_enable         <= S_AXI_ARREADY & S_AXI_ARVALID;

   `w read_done               = read_enable;  // Read data available immediately after address is present.
   
   // read_data  is assigned a value below



   //
   // Handle read data.  For this module we don't have any data to read on the AXIS clock, so we don't need to
   // handle any clock transitions here.
   //
      
   `w read_data = (read_address==0)  ?  reg_freq_ramp_ar         :
                  (read_address==1)  ?  reg_freq_min_ar          :
                  (read_address==2)  ?  reg_freq_max_ar          :
                  (read_address==3)  ?  reg_spread_period_ar     :
                  (read_address==4)  ?  reg_spread_code_ar       :
                  (read_address==5)  ?  reg_sine_shift_right_ar  :
                  /* default */         0                        ;

   
   //
   // Handle write data.  Registers write on the AXI clock.
   // None of the other registers are supposed to be written unless reg_trigger_arm_ar is 0.
   // Thus they won't be written at a critical time and thus metastability isn't an issue.
   //


`ifdef DEBUG
   reg [31:0]  init_reg_freq_ramp_ar          = 32'h0000FFFF;   // Register 0
   reg [31:0]  init_reg_freq_min_ar           = 32'h001FFFFF;   // Register 1
   reg [31:0]  init_reg_freq_max_ar           = 32'h003FFFFF;   // Register 2
   reg [31:0]  init_reg_spread_period_ar      = 32'h00000010;   // Register 3
   reg [31:0]  init_reg_spread_code_ar        = 32'h80002001;   // Register 4
   reg [31:0]  init_reg_sine_shift_right_ar   = 0;              // Reigster 5
   reg [31:0]  init_reg_freq_fract_ar         = 0;              // Register 6  Fractional part of frequency for when freq_min == freq_max
`else
   reg [31:0]  init_reg_freq_ramp_ar          = 0;   // Register 0
   reg [31:0]  init_reg_freq_min_ar           = 0;   // Register 1
   reg [31:0]  init_reg_freq_max_ar           = 0;   // Register 2
   reg [31:0]  init_reg_spread_period_ar      = 0;   // Register 3
   reg [31:0]  init_reg_spread_code_ar        = 0;   // Register 4
   reg [31:0]  init_reg_sine_shift_right_ar   = 0;   // Reigster 5
   reg [31:0]  init_reg_freq_fract_ar         = 0;              // Register 6  Fractional part of frequency for when freq_min == freq_max
`endif


   
   `raxi reg_freq_ramp_ar              <=  (!S_AXI_ARESETN)                       ?  init_reg_freq_ramp_ar          :
                                           write_enable && (write_address==0)     ?  write_data                     :
                                           /* default */                             reg_freq_ramp_ar               ;

   `raxi reg_freq_min_ar               <=  (!S_AXI_ARESETN)                       ?  init_reg_freq_min_ar           :
                                           write_enable && (write_address==1)     ?  write_data                     :
                                           /* default */                             reg_freq_min_ar                ;

   `raxi reg_freq_max_ar               <=  (!S_AXI_ARESETN)                       ?  init_reg_freq_max_ar           :
                                           write_enable && (write_address==2)     ?  write_data                     :
                                           /* default */                             reg_freq_max_ar                ;

   `raxi reg_spread_period_ar          <=  (!S_AXI_ARESETN)                       ?  init_reg_spread_period_ar      :
                                           write_enable && (write_address==3)     ?  write_data                     :
                                           /* default */                             reg_spread_period_ar           ;

   `raxi reg_spread_code_ar            <=  (!S_AXI_ARESETN)                       ?  init_reg_spread_code_ar        :
                                           write_enable && (write_address==4)     ?  write_data                     :
                                           /* default */                             reg_spread_code_ar             ;

   `raxi reg_sine_shift_right_ar       <=  (!S_AXI_ARESETN)                       ?  init_reg_sine_shift_right_ar   :
                                           write_enable && (write_address==5)     ?  write_data                     :
                                           /* default */                             reg_sine_shift_right_ar        ;
     
   `raxi reg_freq_fract_ar             <=  (!S_AXI_ARESETN)                       ?  init_reg_freq_fract_ar         :
                                           write_enable && (write_address==6)     ?  write_data                     :
                                           /* default */                             reg_freq_fract_ar              ;

endmodule
