
`define rp always @(posedge pl_clk)
`define rh always @(posedge high_speed_pl_clk)
`define rhn always @(negedge high_speed_pl_clk)
`define w  assign


//
// The point of this input circuit is to use selNeg to choose
// rising edge sampling or falling edge sampling to avoid metastability.
//

   
module pl_sysref( input wire  pl_clk,

		  input wire  high_speed_pl_clk,
 
                  (* X_INTERFACE_INFO = "xilinx.com:interface:diff_clock:1.0 pl_sysref CLK_P" *)
                  input wire  pl_sysref_p,
                  (* X_INTERFACE_INFO = "xilinx.com:interface:diff_clock:1.0 pl_sysref CLK_N" *)
                  input wire  pl_sysref_n,
    
                  input wire  selNeg,
                  output wire user_sysref_adc, 
                  output wire user_sysref_dac);       

   //
   // Convert the differential input to single-ended
   //
   wire pl_sysref_w;
   
   IBUFDS
     #(
       .DQS_BIAS  ( "FALSE"     )
      )
   IBUFDS_inst
      (
       .O          ( pl_sysref_w ),
       .I          ( pl_sysref_p ),
       .IB         ( pl_sysref_n )
      );


   //
   // Sample the input on both positive and negative edges of the clock.  This lets
   // us choose between them later.  At least one will avoid potential metastability
   // that could cause synchronization issues.
   //
   // Q1 is sampled on the rising edge and Q2 on the falling edge, but we want them both
   // presented on the rising edge to prevent timing problems in following circuitry.
   // Thus we choose SAME_EDGE.  SAME_EDGE_PIPELINED might be even better; it's hard
   // to know what IDDRE1 is doing since its documentation is so poor.  So we'll just
   // do the pipelining ourselves in the next step, removing all possible metastability.
   // This won't eliminate potential timing variation due to metastability though, so
   // it's still important to choose selNeg to avoid the problem.
   //
   // We only have a single clock, so invert it for the CB input.
   //
   wire pl_sysref_rising_0_w;
   wire pl_sysref_falling_0_w;

   IDDRE1
     #(
       .DDR_CLK_EDGE     ( "SAME_EDGE"           ),  // OPPOSITE_EDGE, SAME_EDGE, or SAME_EDGE_PIPELINED
       .IS_C_INVERTED    ( 1'b0                  ),
       .IS_CB_INVERTED   ( 1'b1                  )
      )
   IDDRE1_inst
      (
       .Q1               ( pl_sysref_rising_0_w  ),
       .Q2               ( pl_sysref_falling_0_w ),
       .C                ( pl_clk                ),
       .CB               ( pl_clk                ),
       .D                ( pl_sysref_w           ),
       .R                ( 1'b0                  )
      );


   //
   // Here we remove all potential metastability in pl_sysref_rising_0_w
   // and pl_sysref_falling_0_w by double-buffering each.
   //
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg  pl_sysref_rising_0_r;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg  pl_sysref_falling_0_r;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg  pl_sysref_rising_1_r;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) reg  pl_sysref_falling_1_r;

   `rp pl_sysref_rising_0_r  <= pl_sysref_rising_0_w;
   `rp pl_sysref_falling_0_r <= pl_sysref_falling_0_w;

   `rp pl_sysref_rising_1_r  <= pl_sysref_rising_0_r;
   `rp pl_sysref_falling_1_r <= pl_sysref_falling_0_r;


   //
   // selNeg comes from another clock domain.  Bring it into this clock domain safely.
   // FALSE_PATH_DEST is a user flag, that can be detected in the constraints file to
   // set it as a false path.
   // 
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *) (* FALSE_PATH_DEST = 1 *)    reg selNeg_r;
   (* ASYNC_REG = "TRUE" *) (* DONT_TOUCH = "TRUE" *)                              reg selNeg_rr;

   `rp selNeg_r            <= selNeg;
   `rp selNeg_rr           <= selNeg_r;
   

   //
   // Now we can select the appropriate signal, and send it out.  Give it
   // an extra register just to make sure it can meet timing on what may be
   // a long path.
   //
   (* DONT_TOUCH = "TRUE" *) reg user_sysref_r;
   (* DONT_TOUCH = "TRUE" *) reg user_sysref_adc_0_r;
   (* DONT_TOUCH = "TRUE" *) reg user_sysref_dac_0_r;
   (* DONT_TOUCH = "TRUE" *) reg user_sysref_adc_1_r;
   (* DONT_TOUCH = "TRUE" *) reg user_sysref_dac_1_r;
   (* DONT_TOUCH = "TRUE" *) reg user_sysref_adc_2_r;
   (* DONT_TOUCH = "TRUE" *) reg user_sysref_dac_2_r;

   `rp user_sysref_r       <= selNeg_rr  ?  pl_sysref_rising_1_r : pl_sysref_falling_1_r;

   `rp user_sysref_adc_0_r <= user_sysref_r;
   `rp user_sysref_dac_0_r <= user_sysref_r;

   `rhn user_sysref_adc_1_r <= user_sysref_adc_0_r;
   `rhn user_sysref_dac_1_r <= user_sysref_dac_0_r;

   `rh user_sysref_adc_2_r <= user_sysref_adc_1_r;
   `rh user_sysref_dac_2_r <= user_sysref_dac_1_r;

   `w  user_sysref_adc      = user_sysref_adc_2_r;
   `w  user_sysref_dac      = user_sysref_dac_2_r;
    
endmodule // pl_sysref
