// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include <signal.h>
#include <math.h>
#include <unistd.h>
#include "hwalloc.hh"
#include "graph_data_hw.hh"
#include "hotspot.hh"
#include "rfdc.hh"
#include "xrfdc.h"
#include "xrfdc_clk.h"
#include "displays.hh"
#include "draw_text.hh"
#include "graph.hh"
#include "all_svg.hh"
#include "svg_window.hh"
#include "menu.hh"
#include "menu_graph_select.hh"
#include "menu_action.hh"
#include "document_window.hh"
//#include "clock_config.hh"
#include "document_dir.hh"
#include "menu_trigger_select.hh"
#include "menu_adc_control.hh"
#include "menu_dac_control.hh"
#include "menu_config.hh"
#include "intro.hh"
#include "data_update.hh"
#include "bxbapp_config.hh"
#include "check_clocks.hh"
#include "setup_XRFdc_config.h"
#include "RFSoC4x2_BxBDemo1.hh"

bool ZCU111   = false;
bool RFSoC4x2 = false;

int     main_argc;
char**  main_argv;

static XRFdc RFdcInst;      /* RFdc driver instance */
XRFdc *RFdcInstPtr = &RFdcInst;

menu_adc_control* adc_controls[MAX_NUM_ADCS];
menu_dac_control* dac_controls[MAX_NUM_DACS];

//
// 0 = Gen 1
// 1 = Gen 2
// 2 = Gen 3  ( XRFDC_GEN3 is 2 )
//
int XRFdc_Version;

//
// From menu_dac_control.hh
//
// struct dac_info
// {
//   int                                     dac_number;
//   int                                     dac_tile;
//   int                                     dac_block;
//   char                                    name[128];
//   int                                     nyquist_zone;
//   uint32_t                                hw_address;
//   color                                   dac_color;
//   volatile swept_spread_phase_generator*  dac_control;
// };

//
// Tile and block and name and address are updated later
//
dac_info dacs[MAX_NUM_DACS] =
  {
#ifdef HAVE_Function_Gen0_swept_spread_phase_g_0
    { 0, 0, 0, "DAC 0", 1, HW_RFSoC4x2_BxBDemo1::Function_Gen0_swept_spread_phase_g_0.C_BASEADDR /*0xA0100000*/, RED,       0},
#endif

#ifdef HAVE_Function_Gen1_swept_spread_phase_g_0
    { 1, 0, 1, "DAC 1", 1, HW_RFSoC4x2_BxBDemo1::Function_Gen1_swept_spread_phase_g_0.C_BASEADDR /*0xA0110000*/, GREEN,     0},
#endif
    
#ifdef HAVE_Function_Gen2_swept_spread_phase_g_0
    { 2, 0, 2, "DAC 2", 1, HW_RFSoC4x2_BxBDemo1::Function_Gen2_swept_spread_phase_g_0.C_BASEADDR /*0xA0120000*/, BLUE,      0},
#endif
    
#ifdef HAVE_Function_Gen3_swept_spread_phase_g_0
    { 3, 0, 3, "DAC 3", 1, HW_RFSoC4x2_BxBDemo1::Function_Gen3_swept_spread_phase_g_0.C_BASEADDR /*0xA0130000*/, ORANGE,    0},
#endif
    
#ifdef HAVE_Function_Gen4_swept_spread_phase_g_0
    { 4, 1, 0, "DAC 4", 2, HW_RFSoC4x2_BxBDemo1::Function_Gen4_swept_spread_phase_g_0.C_BASEADDR /*0xA0140000*/, BLUEGREEN, 0},
#endif
    
#ifdef HAVE_Function_Gen5_swept_spread_phase_g_0
    { 5, 1, 1, "DAC 5", 2, HW_RFSoC4x2_BxBDemo1::Function_Gen5_swept_spread_phase_g_0.C_BASEADDR /*0xA0150000*/, PURPLE,    0},
#endif
    
#ifdef HAVE_Function_Gen6_swept_spread_phase_g_0
    { 6, 1, 2, "DAC 6", 1, HW_RFSoC4x2_BxBDemo1::Function_Gen6_swept_spread_phase_g_0.C_BASEADDR /*0xA0160000*/, YELLOW,    0},
#endif
    
#ifdef HAVE_Function_Gen7_swept_spread_phase_g_0
    { 7, 1, 3, "DAC 7", 1, HW_RFSoC4x2_BxBDemo1::Function_Gen7_swept_spread_phase_g_0.C_BASEADDR /*0xA0170000*/, GREY,      0},
#endif
};

//
// From menu_adc_control.hh
//
// struct adc_info
// {
//   int                                     adc_number_hw;
//   int                                     adc_tile;
//   int                                     adc_block;
//   int                                     nyquist_zone;
//   bool                                    dither_enabled;
//   u8                                      calibration_mode;
//   bool                                    calibration_freeze;
//   menu_adc_control*                       control;
// };

//
// This structure is set up during init.
//
adc_info adcs[MAX_NUM_ADCS];


#define GRAPH_PANE   0
#define MENU_PANE    1
#define SCRATCH_PANE 2
#define NUM_PANES    3

rfdc* rfdc_control = 0;
struct sigaction sa_old;

bool do_captures = true;


display*           primary_display;
hotspot<display>*  pause_hotspot;
hotspot<display>*  menu_enter_hotspot;




enum hw_trigger_mode : uint32_t
  {
   TRIGGER_BELOW_MIN_THEN_ABOVE_MAX = 0x00000000,
   TRIGGER_ABOVE_MAX_THEN_BELOW_MIN = 0x00010000,
   TRIGGER_BETWEEN_MIN_AND_MAX      = 0x00020000,
   TRIGGER_NOT_BETWEEN_MIN_AND_MAX  = 0x00030000,
  };

struct hw_trigger
{
  uint32_t         arm;
  uint32_t         mode;
  uint32_t         samples_before_trigger;
  int16_t          min_trigger;
  int16_t          dummy1;
  int16_t          max_trigger;
  int16_t          dummy2;
  uint32_t         samples_after_trigger;
  int32_t          sample;
  int32_t          min_trigger_readback;
  int32_t          max_trigger_readback;
};


bool at_least_one_graph = false;

void delete_rfdc(int value, void* ptr = 0)
{
  if(rfdc_control)
    delete rfdc_control;
}

void delete_rfdc_2(int value)
{
  if(rfdc_control)
    delete rfdc_control;
  sigaction(SIGINT, &sa_old, NULL);
  kill(0, SIGINT);
}


double sinc(double x)
{
  if(x==0.0)
    return 1.0;
  
  return sin(x)/x;
}


void fill_empty_display(display* d)
{
  //printf("entering fill_empty_display(); d->layout_dirty=%d\n", d->layout_dirty);
  primary_display->remove_no_delete(menu_enter_hotspot);
  primary_display->remove_no_delete(pause_hotspot);

  d->remove_all();
  intro* intro_window = new intro();
  primary_display->add(menu_enter_hotspot);
  primary_display->add(pause_hotspot);
  d->add(intro_window);
  //printf("leaving fill_empty_display();  d=%p, d->layout_dirty=%d\n", d, d->layout_dirty);
}



bool menu_enter_function(hotspot<display>* hs, my_event&me, void* data)
{
  display* primary_display = hs->get_display();

  if(me.type==EVENT_TOUCH)
    {
      primary_display->set_current_pane(MENU_PANE);
      primary_display->mark_layout_dirty();
      at_least_one_graph = true;
    }

  primary_display->touch_recognized_beep();

  return true;
}


bool menu_enter_from_document_function(hotspot<display>* hs, my_event&me, void* data)
{
  display* primary_display = hs->get_display();

  if(me.type==EVENT_TOUCH)
    {
      document_window* dw = (document_window*)data;

      delete dw;  // need to delete only after getting primary_display.
  
      primary_display->set_current_pane(MENU_PANE);
      primary_display->mark_layout_dirty();
      at_least_one_graph = true;
    }

  primary_display->touch_recognized_beep();

  return true;
}



void menu_document_action(menu_action* ma, bool selected, void* data)
{
  docfile* df = (docfile*) data;

  display* primary_display = ma->get_display();
  
  if(selected)
    {
      primary_display->set_current_pane(SCRATCH_PANE);

      //
      // Clean up scratch display
      //
      for(;;)
	{
	  ListElement<window>* le=primary_display->subwindows->first();

	  if(!le)
	    break;
	
	  primary_display->subwindows->remove(le);
	}

      //
      // In the future may want to split the document name into name and directory
      // with a slash separating.
      //
      document_window* dw = new document_window(df->base_dir, df->name);

      hotspot<display>* menu_enter_hotspot = new hotspot<display>(primary_display, menu_enter_from_document_function, dw);

      float designed_height = 1080.0;
      float designed_width  = 1920.0;
      float ratio_x = primary_display->width / designed_width;
      float ratio_y = primary_display->height / designed_height;
      float ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;
      
      menu_enter_hotspot->resize(0,             // xoffset
				 0,             // yoffset 
				 140*ratio,     // width
				 140*ratio);    // height
      
      dw->add(menu_enter_hotspot);
      primary_display->add(dw);
    }

  primary_display->mark_layout_dirty();
  primary_display->trash_events_until_release();
}


void menu_return_to_graph_action(menu_action* ma, bool selected, void* data)
{
  display* primary_display = ma->get_display();

  do_captures = true;

  menu*    graph_menu      = (menu*)data;
  if(selected)
    {
      // Remove all graphs from graph pane.  But not hotspots.

      primary_display->set_current_pane(GRAPH_PANE);

      ListElement<window>* le=primary_display->subwindows->first(); 
      while(le)
	{
	  window* w = le->data();
	  le=le->next();  // Get the next item before deleting this one, or our next() disappears in the delete.
	  if(w->window_subject_to_layout)
	    delete w;
	}
      //primary_display->remove_all(GRAPH_PANE);

      // First is the New Graph action item.  Skip it.
      at_least_one_graph = false;
      for(ListElement<window>* le=graph_menu->subwindows->first()->next(); le; le=le->next())
	{
	  menu_graph_select* mgs = (menu_graph_select*)le->data();

	  mgs->update_x_units_possible_sampling_rate_change();
	  
	  graph* g = hw_graph_create(mgs->gc);


	  // Determine that all signals are in the right nyquist zone
	  int zone = -1;
	  bool xfer = false;
	  int source_type = mgs->gc.type;

	  if(source_type==TYPE_FREQ_Z1)
	    {
	      zone = 1;
	      xfer = false;
	    }
	  else if(source_type==TYPE_Z1_TRANSFER_MAG || source_type==TYPE_Z1_TRANSFER_PHASE ||
	     source_type==TYPE_Z1_TRANSFER_GROUP_DELAY || source_type==TYPE_Z1_TRANSFER_PHASE_DEBUG)
	    {
	      zone = 1;
	      xfer = true;
	    }
	  else if(source_type==TYPE_FREQ_Z2)
	    {
	      zone = 2;
	      xfer = false;
	    }
	  else if(source_type==TYPE_Z2_TRANSFER_MAG || source_type==TYPE_Z2_TRANSFER_PHASE ||
	     source_type==TYPE_Z2_TRANSFER_GROUP_DELAY || source_type==TYPE_Z2_TRANSFER_PHASE_DEBUG)
	    {
	      zone = 2;
	      xfer = true;
	    }

	  if(zone==1 || zone==2)
	    {
	      // Determine that all ADCs/DACs graphed are also in this zone.

	      if(xfer)
		{
		  int sw_reference = mgs->gc.reference;

		  if(sw_source_is_adc(sw_reference))
		    {
		      // it is an ADC
		      if(adcs[sw_source_to_hw_adc(sw_reference)].nyquist_zone != zone)
			{
			  sprintf(g->error_text, " %c%s%c Nyquist Zone doesn't match graph Nyquist Zone \"Band\"",
				  graph::MAX_GRAPHS-1+10,
				  source_names_sw(sw_reference),
				  10+graph::MAX_GRAPHS);
			}
		    }
		  else if(sw_source_is_dac(sw_reference))
		    {
		      // it is a DAC
		      if(dacs[sw_source_to_hw_dac(sw_reference)].nyquist_zone != zone)
			{
			  sprintf(g->error_text, " %c%s%c Nyquist Zone doesn't match graph Nyquist Zone \"Band\"",
				  graph::MAX_GRAPHS-1+10,
				  source_names_sw(sw_reference),
				  10+graph::MAX_GRAPHS);
			}
		    }		  

		}

	      for(int i=0; i<g->num_graphs; i++)
		{
		  // Transfer graphs can have more graphs than sources.
		  if(i>=mgs->gc.num_hw_sources)
		    break;
		  
		  int hsn = mgs->gc.source_list_number[i];
		  hw_source* hs = &hw_source_list[hsn];
		  int sw_source = hs->source_number_sw;
		  if(sw_source_is_adc(sw_source))
		    {
		      //printf("ADC %d is in zone %d, expecting %d\n", i, adcs[adc_number].nyquist_zone, zone);
		      // it is an ADC
		      if(adcs[sw_source_to_hw_adc(sw_source)].nyquist_zone != zone)
			{
			  sprintf(g->error_text, " %c%s%c Nyquist Zone doesn't match graph Nyquist Zone \"Band\"",
				  i+10, source_names_sw(sw_source), 10+graph::MAX_GRAPHS);
			  break;
			}
		    }
		  else if(sw_source_is_dac(sw_source))
		    {
		      // it is a DAC
		      if(dacs[sw_source_to_hw_dac(sw_source)].nyquist_zone != zone)
			{
			  sprintf(g->error_text, " %c%s%c Nyquist Zone doesn't match graph Nyquist Zone \"Band\"",
				  i+10, source_names_sw(sw_source), 10+graph::MAX_GRAPHS);
			  break;
			}
		    }		  
		}
	    }

	  primary_display->add(g);
	  at_least_one_graph = true;
	}
      
      if(!at_least_one_graph)
	{
	  fill_empty_display(primary_display);
	}
      
    }
  primary_display->trash_events_until_release();
}



void new_graph_action(menu_action* ma, bool selected, void* data)
{
  //display* primary_display = ma->get_display();
  menu*    graph_menu      = (menu*)data;
  if(selected)
    {
      int num = graph_menu->subwindows[0].numInList();
      if(num==graph::MAX_GRAPHS)
	{
	  graph_menu->get_display()->entry_rejected_beep();
	  return;
	}
      char temp[200];
      graph_config gc = default_gc(TYPE_FREQ_Z1);
      sprintf(temp, "Graph %d", num-1);
      ma->trash_events_until_release();
      menu_graph_select* mgs = new menu_graph_select(ma->get_display(), temp, gc);
      graph_menu->add(mgs);
      graph_menu->select_a_child(num);
    }
}


bool pause_function(hotspot<display>* hs, my_event&me, void* data)
{
  static point oldmousepoint;
  static bool  doodle_mode = false;
  static bool  doodle_mode_holdoff = true;

  display* primary_display=hs->get_display();

  if(me.type == EVENT_TOUCH)
    {
      if(doodle_mode)
	{
	  int x = me.c[0].x;
	  int y = me.c[0].y;
	  window* p = hs;
	  while(p)
	    {
	      
	      x -= p->offset.x;
	      y -= p->offset.y;
	      p = p->parent;
	    }

	  if(me.num_touches>1)
	    {
	      primary_display->mark_layout_dirty();
	      primary_display->touch_recognized_beep();
	    }
	  
	  if(x>0 && x<hs->width && y>0 && y<hs->height)
	    {
	      doodle_mode = false;
	      do_captures = true;
	      primary_display->mark_layout_dirty();
	      primary_display->touch_recognized_beep();
	    }
	  
	  doodle_mode_holdoff = false;
	  oldmousepoint = point(me.c[0].x, me.c[0].y);
	}
      else
	{
	  do_captures = !do_captures;
	  primary_display->touch_recognized_beep();
	  hs->claim_events();
	}
      return true;
    }
  if(me.type == EVENT_RELEASE)
    {
      if(!doodle_mode)
	hs->release_events();
      return true;
    }
  else if(me.type == EVENT_HOLD && me.count==10)
    {
      do_captures = false;
      doodle_mode = true;
      doodle_mode_holdoff = true;
      oldmousepoint = point(me.c[0].x, me.c[0].y);
      primary_display->entry_accepted_beep();
      return true;
    }
  else if(me.type == EVENT_HOLD && me.count==20)
    {
      do_captures = true;
      doodle_mode = true;
      doodle_mode_holdoff = true;
      oldmousepoint = point(me.c[0].x, me.c[0].y);
      primary_display->entry_accepted_beep();
      return true;
    }
  else if(me.type == EVENT_MOVE)
    {
      if(doodle_mode && !doodle_mode_holdoff)
	{
	  point mousepoint = point(me.c[0].x, me.c[0].y);

	  primary_display->beginDrawToStatic();  
	  primary_display->draw_line(oldmousepoint, mousepoint, GREEN);
	  primary_display->endDrawToStatic();  

	  oldmousepoint = mousepoint;
	}
      return true;
    }
  else if(me.type == EVENT_KEY_PRESS)
    {
      // Let someone else handle keys
      return false;
    }
  else
    {
      // Any other event types are for mouse movement over us, so
      // claim them.
      return true;
    }
}

void clear_mem(void* loc, int size)
{
  char* x = (char*)loc;
  for(int i=0; i<size; i++)
    x[i] = 0;
}

void init_xrfdc()
{
  u16 RFdcDeviceId = 0;
  int Status;
  XRFdc_Config Config;
  XRFdc_Config *ConfigPtr = &Config;
  struct metal_device *deviceptr = NULL;
  
  struct metal_init_params init_param = METAL_INIT_DEFAULTS;
	
  if(metal_init(&init_param))
    {
      printf("ERROR: Failed to run metal initialization\n");
      exit(20);
    }

  /* Initialize the RFdc driver. */
  //ConfigPtr = XRFdc_LookupConfig(RFdcDeviceId);
  //if(!ConfigPtr)
  //  {
  //    printf("ERROR: Failed to find the XRFdc driver.\n");
  //    exit(20);
  //  }

  MySetupXRFdc_Config(ConfigPtr);
  

  Status = XRFdc_RegisterMetal(RFdcInstPtr, RFdcDeviceId, &deviceptr);
  if(Status != XRFDC_SUCCESS)
    {
      printf("ERROR: Failed to register the XRFdc driver.\n");
      exit(20);
    }

  /* Initializes the controller */
  Status = XRFdc_CfgInitialize(RFdcInstPtr, ConfigPtr);
  if (Status != XRFDC_SUCCESS)
    {
      printf("ERROR: Failed to initialize the XRFdc driver.\n");
      exit(20);
    }

  XRFdc_Version = RFdcInstPtr->RFdc_Config.IPType;

  //typedef struct {
  //	u32 IsEnabled; /* 1, if tile is enabled, 0 otherwise */
  //	u32 TileState; /* Indicates Tile Current State */
  //	u8 BlockStatusMask; /* Bit mask for block status, 1 indicates block enable */
  //	u32 PowerUpState;
  //	u32 PLLState;
  //} XRFdc_TileStatus;

  //typedef struct {
  //	XRFdc_TileStatus DACTileStatus[4];
  //	XRFdc_TileStatus ADCTileStatus[4];
  //	u32 State;
  //} XRFdc_IPStatus;

  XRFdc_IPStatus IPStatus;
  clear_mem(&IPStatus, sizeof(IPStatus));
  
  u32 IPStatusReturn = XRFdc_GetIPStatus(RFdcInstPtr, &IPStatus);
  printf("Tile information from XRFdc_GetIPStatus (Which has return value IPStatusReturn=%d):\n", IPStatusReturn);
  printf("   State is 0x%08X\n", (int)IPStatus.State);
  for(int i=0; i<4; i++)
    {
      printf("   DAC Tile %d status:\n", i);
      printf("      is_Enabled       = %d\n",     (int)IPStatus.DACTileStatus[i].IsEnabled);
      printf("      TileState        = %d\n",     (int)IPStatus.DACTileStatus[i].TileState);
      printf("      BlockStatusMask  = 0x%02X\n", (int)IPStatus.DACTileStatus[i].BlockStatusMask);
      printf("      PowerUpState     = %d\n",     (int)IPStatus.DACTileStatus[i].PowerUpState);
      printf("      PLLState         = %d\n",     (int)IPStatus.DACTileStatus[i].PLLState);
    }
  for(int i=0; i<4; i++)
    {
      printf("   ADC Tile %d status:\n", i);
      printf("      is_Enabled       = %d\n",     (int)IPStatus.ADCTileStatus[i].IsEnabled);
      printf("      TileState        = %d\n",     (int)IPStatus.ADCTileStatus[i].TileState);
      printf("      BlockStatusMask  = 0x%02X\n", (int)IPStatus.ADCTileStatus[i].BlockStatusMask);
      printf("      PowerUpState     = %d\n",     (int)IPStatus.ADCTileStatus[i].PowerUpState);
      printf("      PLLState         = %d\n",     (int)IPStatus.ADCTileStatus[i].PLLState);
    }



  NUM_ADCS = 0;
  for(int tile=0; tile<4; tile++)
    for(int block=0; block<4; block++)
      {
	u32 NyquistZone;
	u32 dither_mode;
	u8 calibration_mode;
	XRFdc_Cal_Freeze_Settings freeze;
	      
	Status =  XRFdc_GetNyquistZone(RFdcInstPtr,  XRFDC_ADC_TILE, tile, block, &NyquistZone);
	
	if(Status != XRFDC_SUCCESS)
	  {
	    printf("\n******** ADC tile %d, block %d is NOT OPERATING.\n\n", tile, block);
	    continue;
	  }

	if(NyquistZone!=1 && !ZCU111)
	  {
	    //printf("  Resetting ADC %d at tile %d, block %d to Zone 1.\n", i, tile, block);
	    XRFdc_SetNyquistZone(RFdcInstPtr, XRFDC_ADC_TILE, tile, block, 1);
	    XRFdc_GetNyquistZone(RFdcInstPtr, XRFDC_ADC_TILE, tile, block, &NyquistZone);
	  }
	
	Status = XRFdc_GetCalFreeze(RFdcInstPtr, tile, block, &freeze);
	if(Status != XRFDC_SUCCESS)
	  printf("ERROR: Can't get Calibration Freeze for %s at tile %d block %d\n", adc_names_hw(NUM_ADCS), tile, block);

	Status = XRFdc_GetCalibrationMode(RFdcInstPtr, tile, block, &calibration_mode);
	if(Status != XRFDC_SUCCESS)
	  printf("ERROR: Can't get Calibration Mode for %s at tile %d block %d\n", adc_names_hw(NUM_ADCS), tile, block);
	
	Status = XRFdc_GetDither(RFdcInstPtr, tile, block, &dither_mode);
	if(Status != XRFDC_SUCCESS)
	  printf("ERROR: Can't get Dither Mode for %s at tile %d block %d\n", adc_names_hw(NUM_ADCS), tile, block);
      
	adcs[NUM_ADCS].adc_number_hw       =  NUM_ADCS;
	adcs[NUM_ADCS].adc_tile            =  tile;
	adcs[NUM_ADCS].adc_block           =  block;      
	adcs[NUM_ADCS].nyquist_zone        =  NyquistZone;
	adcs[NUM_ADCS].dither_enabled      = (dither_mode==XRFDC_DITH_ENABLE);
	adcs[NUM_ADCS].calibration_mode    = calibration_mode;
	adcs[NUM_ADCS].calibration_freeze  = (freeze.CalFrozen!=0);
	adcs[NUM_ADCS].control             = 0;

	printf("\n******** ADC tile %d, block %d is labeled %s and is in Nyquist Zone %d.\n******** Dither is %s, calibration is %s in %s.\n\n",
	       tile, block, adc_names_hw(NUM_ADCS), NyquistZone, adcs[NUM_ADCS].dither_enabled?"enabled":"disabled",
	       adcs[NUM_ADCS].calibration_freeze ? "frozen" : "operating", number_to_calibration_mode(adcs[NUM_ADCS].calibration_mode));

	NUM_ADCS++;
      }
  
  printf("Found %d ADCs.\n", NUM_ADCS);


  NUM_DACS = 0;
  for(int tile=0; tile<4; tile++)
    for(int block=0; block<4; block++)
      {
	u32 NyquistZone;
	      
	Status =  XRFdc_GetNyquistZone(RFdcInstPtr,  XRFDC_DAC_TILE, tile, block, &NyquistZone);
	
	if(Status != XRFDC_SUCCESS)
	  {
	    printf("\n******** DAC tile %d, block %d is NOT OPERATING.\n\n", tile, block);
	    continue;
	  }

	if(NyquistZone!=1 && !ZCU111)
	  {
	    //printf("  Resetting %s at tile %d, block %d to Zone 1.\n", DAC_NAMES[i], tile, block);
	    XRFdc_SetNyquistZone(RFdcInstPtr, XRFDC_DAC_TILE, tile, block, 1);
	    XRFdc_GetNyquistZone(RFdcInstPtr, XRFDC_DAC_TILE, tile, block, &NyquistZone);
	  }
	
	dacs[NUM_DACS].dac_tile            =  tile;
	dacs[NUM_DACS].dac_block           =  block;
	dacs[NUM_DACS].nyquist_zone        =  NyquistZone;

	strcpy(dacs[NUM_DACS].name, dac_names_hw(NUM_DACS));

	printf("\n******** DAC tile %d, block %d is labeled %s and is in Nyquist Zone %d.\n\n",
	       tile, block, dac_names_hw(NUM_DACS), NyquistZone);
	
	NUM_DACS++;
      }
  
  printf("Found %d DACs.\n", NUM_DACS);
}

int main(int argc, char** argv)
{
  main_argc = argc;
  main_argv = argv;
  
  int config = 0;
  const char* config_file = "/opt/BxB/BxBApp/bxbapp.config";
  const char* boardname = "unknown";

  printf("BxBApp called as ");
  for(int i=0; i<argc; i++)
    printf("\"%s\"%c", argv[i], (i==argc-1) ? '.' : ' ');
  printf(".\n");
  
  for(int i=1; i<argc; i++)
    {
      if(!strcmp("-board", argv[i]))
	{
	  i++;
	  boardname=argv[i];
	}
      else if(!strcmp("-configfile", argv[i]))
	{
	  i++;
	  if(i>=argc)
	    {
	      printf("ERROR:  \"-configfile\" takes an argument.\n");
	      exit(20);
	    }
	  config_file = argv[i];
	  printf("Setting configuration file to \"%s\"\n", config_file);
	}
      else if(!strcmp("-config", argv[i]))
	{
	  i++;
	  if(i>=argc)
	    {
	      printf("ERROR:  \"-config\" takes an argument.\n");
	      exit(20);
	    }
	  config = atoi(argv[i]);
	  printf("Setting configuration to #%d\n", config);
	}
      else
	{
	  printf("ERROR:  Illegal argument \"%s\".\n", argv[i]);
	  exit(20);
	}
    }


  // Read all allowable configurations, and set the default configuration.
  allowable_configurations = new bxbapp_configs(config_file);
  mark_current_config(config);

  if(!strcmp(boardname, "unknown"))
    {
      printf("Using config file board name \"%s\".\n", (char*)config_board_name);
      boardname = config_board_name;
    }
  else
    {
      printf("Using command line board name \"%s\".\n", boardname);
    }


  if(!strcmp(boardname, "ZCU111"))
    ZCU111 = true;
  if(!strcmp(boardname, "RFSoC4x2"))
    RFSoC4x2 = true;
  

  printf("Setting up display.\n"); fflush(stdout);
  
  primary_display = new displays(NUM_PANES);

  float designed_height = 1080.0;
  float designed_width  = 1920.0;
  float ratio_x = primary_display->width / designed_width;
  float ratio_y = primary_display->height / designed_height;
  float ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;

  
  pause_hotspot = new hotspot<display>(primary_display, pause_function, 0);

  pause_hotspot->resize(0,                              // xoffset
			primary_display->height - 140 * ratio,  // yoffset 
			140 * ratio,                            // width
			140 * ratio);                           // height


  menu_enter_hotspot = new hotspot<display>(primary_display, menu_enter_function, primary_display);

  menu_enter_hotspot->resize(0,             // xoffset
			     0,             // yoffset 
			     140 * ratio,           // width
			     140 * ratio);          // height

  
  primary_display->target_aspect_x = 19;
  primary_display->target_aspect_y = 6;
  
  //auto bxb = new svg_window(BxB_svg, BxB_svg_length);
  //primary_display->add(bxb,                 GRAPH_PANE);

  primary_display->add(pause_hotspot,       GRAPH_PANE);
  primary_display->add(menu_enter_hotspot,  GRAPH_PANE);

  fill_empty_display(primary_display);

  usleep(300000);  // Wait for display initialization in another thread.

  printf("First display draw.\n");
  
  primary_display->draw();	                  // Draw static, dynamic, and flip the page

  
  activate_current_config();
  
  //
  // Create a second thread that continually checks the clock measurements.
  //
  new check_clocks;

  data_init();

  printf("Configuring ADC/DAC\n"); fflush(stdout);

  init_xrfdc();  // This function is defined earlier in this file.




  //primary_display->entry_accepted_beep();

  
  printf("Setting up application control of ADC/DAC\n"); fflush(stdout);
  
  //
  // Init RFDC
  //
  rfdc_control = new rfdc (true /* verbose */,
			   true /* shut down on delete */,
			   HW_RFSoC4x2_BxBDemo1::ADC_DAC_usp_rf_data_converter_1.C_BASEADDR /*0xA0000000*/ /* base address */);


  //usleep(250000);  // Allow half a second for RFDC clocks to stabilize after they were started above.


  printf("Application control of ADC/DAC established.\n"); fflush(stdout);

  if(!rfdc_control->all_running(true /* verbose */))  //  || !clocks_measure_ok)
    {
      printf("Failed to initialize the ADCs and DACs.\n");
      //printf("  -- Method to determine this is suspect during testing.  Ignoring this for now.\n");
      intro::failed_initializing_clocks = true;
      primary_display->draw();	                  // Draw static, dynamic, and flip the page
      printf("Waiting 10 seconds, then restarting.\n");
      sleep(10);
      exit(20);
    }  

  intro::clocks_initialized = true;
  primary_display->draw();	                  // Draw static, dynamic, and flip the page
  printf("ADCs and DACs initialized.\n\n");


  
  // Do not delete rfdc_control.  It is set to turn off the ADCs/DACs when it is deleted.
  on_exit(delete_rfdc, 0);
  struct sigaction sa;
  memset(&sa, 0, sizeof(sa));
  sa.sa_handler = delete_rfdc_2;
  sigaction(SIGINT, &sa, &sa_old);
  
  
  menu* main                    = new menu("Main menu");
  
  menu* graph_menu              = new menu("Graph Setup");
  menu* new_graph               = new menu_action("New Graph", new_graph_action, graph_menu);  graph_menu -> add(new_graph);
  
  menu_action* return_to_graph  = new menu_action("Go to Graphs", menu_return_to_graph_action, graph_menu);

  main->add(return_to_graph);
  main->add(graph_menu);

  menu_trigger_select* main_2   = new menu_trigger_select("Trigger Source");  main         -> add(main_2);

  menu* main_3                  = new menu("ADC Control");            main         -> add(main_3);

  for(int i=0; i<NUM_ADCS; i++)
    {
      if(i>=4 && RFSoC4x2)
	continue;

      int adc_number_hw = sw_adc_to_hw_adc(i);
      
      adc_controls[i] = new menu_adc_control(adcs[adc_number_hw]);
      main_3->add(adc_controls[i]);
    }

  menu* main_4                  = new menu("DAC Control");            main         -> add(main_4);
  
  for(int i=0; i<NUM_DACS; i++)
    {
      if(i>=2 && RFSoC4x2)
	continue;

      int dac_number_hw = sw_dac_to_hw_dac(i);

      dac_controls[i] = new menu_dac_control(dacs[dac_number_hw]);
      main_4->add(dac_controls[i]);
    }

  if(ZCU111 && sampling_rate_kHz == 3932160)
    main_4->selected_num = NUM_DACS-1;

  
  menu* main_5              = new menu_config("Configuration", graph_menu);       main         -> add(main_5);

  
  //menu* main_4                  = new menu("System Setup");                   main         -> add(main_4);
  //menu* main_4_sub_1            = new menu("Sampling Rate");                  main_4       -> add(main_4_sub_1);
  //menu* main_5_sub_1       = new menu("Tutorial");             main_5       -> add(main_5_sub_1);
  //menu* main_5_sub_2       = new menu("User's Guide");         main_5       -> add(main_5_sub_2);

  
  document_dir(main, "/opt/BxB", "Documents");
  primary_display->add(main, MENU_PANE);

  //printf("in main: primary_display is %p\n", primary_display);



  
  //draw_text dt;
  
  //int num_periods_in_dac_buffer = 170;
  //int dac_buffer_samples        = 65536;


  
  //volatile uint32_t&    dac_trigger   = *hwalloc<uint32_t>    (0xB2000000);
  volatile hw_trigger&  adc_trigger   = *hwalloc<hw_trigger>  ( HW_RFSoC4x2_BxBDemo1::trigger_simple_trigger_1.C_BASEADDR  /*0xA0060000*/);


  // Set up DAC buffer
  //volatile int16_t*  dac_buffer  =  hwalloc<int16_t>  (0xB1000000, dac_buffer_samples);
  //for(int i=0; i<dac_buffer_samples; i++)
  //  {
  //    float   radians = i * num_periods_in_dac_buffer * 2 * M_PI / dac_buffer_samples;
  //    int16_t sample  = int(floor(32767 * sin(radians) + 0.5));
  //
  //    dac_buffer[i] = sample;
  //  }
  //hwfree(dac_buffer, dac_buffer_samples);
  
  
  //dac_trigger   = 1;    // Tell DAC to play the buffer

  
  
  uint32_t*  trigger_hw_address      =  new uint32_t           [menu_trigger_select::num_sources];
  int*       trigger_num_pts         =  new int                [menu_trigger_select::num_sources];
  volatile int16_t**  trigger_data   =  new volatile int16_t*  [menu_trigger_select::num_sources];

  //
  // For RFSoC4x2, reverse sources
  //
  //int trigger_source_start = RFSoC4x2 ?  num_hw_sources-1  :   0              ;
  //int trigger_source_end   = RFSoC4x2 ?  -1                :   num_hw_sources ;
  //int trigger_source_step  = RFSoC4x2 ?  -1                :   1              ;
  //for(int i=trigger_source_start; i!=trigger_source_end; i+=trigger_source_step)
  
  int source_num = 0;
  for(int i=0; i<num_hw_sources; i++)
    {
      struct hw_source* s = &hw_source_list[i];
      if(s->type==TYPE_TIME)
	{
	  int       adc_number_hw = sw_source_to_hw_adc(s->source_number_sw);
	  //	  uint32_t  hw_address    = 0xB0000000 + adc_number_hw * 0x00020000;

	  uint32_t  hw_address    = (

#ifdef HAVE_real_sample_capture_01
				      (adc_number_hw == 1) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_01.C_BASEADDR :
#endif

#ifdef HAVE_real_sample_capture_02
				      (adc_number_hw == 2) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_02.C_BASEADDR :
#endif

#ifdef HAVE_real_sample_capture_03
				      (adc_number_hw == 3) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_03.C_BASEADDR :
#endif

#ifdef HAVE_real_sample_capture_04
				      (adc_number_hw == 4) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_04.C_BASEADDR :
#endif

#ifdef HAVE_real_sample_capture_05
				      (adc_number_hw == 5) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_05.C_BASEADDR :
#endif

#ifdef HAVE_real_sample_capture_06
				      (adc_number_hw == 6) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_06.C_BASEADDR :
#endif

#ifdef HAVE_real_sample_capture_07
				      (adc_number_hw == 7) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_07.C_BASEADDR :
#endif

				      
				      /**/                   HW_RFSoC4x2_BxBDemo1::real_sample_capture_00.C_BASEADDR );

				      
	  trigger_hw_address[source_num]  =  hw_address;
	  trigger_num_pts[source_num]     =  s->num_points; 
	  trigger_data[source_num]        =  hwalloc<int16_t>(trigger_hw_address[source_num], trigger_num_pts[source_num]);
	  source_num++;
	}
    }


  
  int       trigger_points_after          =  trigger_num_pts[0] * 3 / 4;
  int       trigger_points_before         =  trigger_num_pts[0] - trigger_points_after + 400;
  //int       trigger_max                   =  0;
  //int       trigger_min                   =  -200;
  int       approximate_trigger_position  =  trigger_num_pts[0] - trigger_points_after;


  adc_trigger.arm                         =  0;
  adc_trigger.mode                        =  TRIGGER_BELOW_MIN_THEN_ABOVE_MAX;
  adc_trigger.samples_before_trigger      =  trigger_points_before / adc_samples_per_clock;
  adc_trigger.min_trigger                 =  menu_trigger_select::trigger_level;
  adc_trigger.max_trigger                 =  menu_trigger_select::trigger_level - menu_trigger_select::trigger_hysteresis;
  adc_trigger.samples_after_trigger       =  trigger_points_after  / adc_samples_per_clock;
  

  int valid_per_group;
  int group_length;

  {
    int x = current_config.adc_points_per_clock;
    int y = next_higher_power_of_2(x);
    if(y==x)
      {
	valid_per_group = 1;
	group_length    = 1;
      }
    else
      {
	valid_per_group = x;
	group_length    = y;
      }
  }
    
  primary_display->entry_accepted_beep();
  //usleep(250000);
  //primary_display->entry_accepted_beep();


  intro::demo_is_ready = true;
  printf("Demo is Ready.\n\n");
  primary_display->draw();	                  // Draw static, dynamic, and flip the page
  

  for(long int count = 0;; count++)
    {
      //printf("count=%ld\n", count);
      
      // Trigger an ADC Capture
      if(do_captures && !primary_display->need_fast_display())
	{
	  bool free_run    = (menu_trigger_select::trigger_source == -1);
	  int  trig_source = menu_trigger_select::trigger_source;
	  if(free_run)
	    {
	      adc_trigger.mode         =  TRIGGER_BELOW_MIN_THEN_ABOVE_MAX;
	      adc_trigger.max_trigger  =  0x8000;  // Max negative falue
	      adc_trigger.min_trigger  =  0x7FFF;  // Max positive value
	      adc_trigger.arm          =  1;
	    }
	  else
	    {
	      // Reverse trigger sources for RFSoC4x2
	      adc_trigger.mode         =  TRIGGER_BELOW_MIN_THEN_ABOVE_MAX + (RFSoC4x2 ? 3-trig_source : trig_source);
	      adc_trigger.max_trigger  =  menu_trigger_select::trigger_level;
	      adc_trigger.min_trigger  =  menu_trigger_select::trigger_level - menu_trigger_select::trigger_hysteresis;
	      adc_trigger.arm          =  1;
	    }


	  usleep(4000);  // Wait for data to be captured	

	  uint32_t arm = adc_trigger.arm;
	  adc_trigger.arm = 0;

	  int collecting = (arm>>6)&1;
	  
	  graph_data_base::time_data_good = (!collecting);

	  float x_offset = approximate_trigger_position;

	  int trig = menu_trigger_select::trigger_level;
	  if(graph_data_base::time_data_good && !free_run)
	    {
	      int trigger_position = approximate_trigger_position;
	      int index;

	      for(;;)
		{
		  if(group_length==1)
		    {
		      index = trigger_position;
		    }
		  else
		    {
		      int group_index = trigger_position / valid_per_group;
		      int base_index  = group_index * group_length;
		      int base_offset = trigger_position % valid_per_group;
		      index = base_index + base_offset;
		    }

		  if(trigger_data[trig_source][index]>=trig || trigger_position>=trigger_num_pts[trig_source]-1)
		    break;

		  trigger_position++;
		}
	      
	      for(;;)
		{
		  if(group_length==1)
		    {
		      index = trigger_position;
		    }
		  else
		    {
		      int group_index = trigger_position / valid_per_group;
		      int base_index  = group_index * group_length;
		      int base_offset = trigger_position % valid_per_group;
		      index = base_index + base_offset;
		    }

		  if(trigger_data[trig_source][index]<=trig || trigger_position==0)
		    break;
		  
		  trigger_position--;
		}

	      int index_below = index;
	      
	      int group_index = (trigger_position+1) / valid_per_group;
	      int base_index  = group_index * group_length;
	      int base_offset = (trigger_position+1) % valid_per_group;
	      int index_above = base_index + base_offset;
	      
	      x_offset = trigger_position + (trig - trigger_data[trig_source][index_below]) / float(trigger_data[trig_source][index_above] - trigger_data[trig_source][index_below]);
	    }
	  
	  graph_data_base::time_x_offset = -x_offset;
	  //graph_data_base::time_x_offset = 0;

#ifdef NOTDEF
	  printf("trigger status 0x%08x, mode=%d, collect=%d final_delay_done=%d, triggered=%d, armed=%d, enabled=%d, initial_delay_done=%d, software_armed=%d sample=%d, min=%d (set %d), max=%d (set %d) x_offset=%f\n",
		 arm,
		 (arm>>30)&3,
		 (arm>>6)&1,
		 (arm>>5)&1,
		 (arm>>4)&1,
		 (arm>>3)&1,
		 (arm>>2)&1,
		 (arm>>1)&1,
		 arm&1,
		 adc_trigger.sample,
		 adc_trigger.min_trigger_readback,
		 adc_trigger.min_trigger,
		 adc_trigger.max_trigger_readback,
		 adc_trigger.max_trigger,
		 -x_offset);
#endif
	}


      // Copy the data to be plotted
      //for(int i=0; i<num_graph_points; i++)
      //	  gr.y_data[0][i] = adc_buffer[i];
      
      primary_display->get_and_process_events();  // Fetch, distribute, and handle mouse and key events


      ratio_x = primary_display->width / designed_width;
      ratio_y = primary_display->height / designed_height;
      ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;
      
      pause_hotspot->resize_quietly(0,                              // xoffset
				    primary_display->height - 140 * ratio,  // yoffset 
				    140 * ratio,                            // width
				    140 * ratio);                           // height
      
      menu_enter_hotspot->resize_quietly(0,             // xoffset
					 0,             // yoffset 
					 140 * ratio,           // width
					 140 * ratio);          // height


      if(do_captures && !primary_display->need_fast_display())
	primary_display->update_data();    // Update data that needs updates whether draws occur or not.

      primary_display->draw();	         // Draw static, dynamic, and flip the page


      if(do_captures && !primary_display->need_fast_display())
	data_start_capture();
      
      usleep(400);  // Frequency capture to start

    }

  //hwfree(&dac_trigger);
  hwfree(&adc_trigger);
  //  hwfree(adc_buffer, adc_buffer_samples);
}
