// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include "hwalloc.hh"
#include "RFSoC4x2_BxBDemo1.hh"



//
// This describes the registers in hardware.  The structure lives in AXI space in the hardware.
// Thus when accessing a member of it, you are accessing a register value from hardware.
// For this to happen, it of course needs to be allocated to the correct physical memory
// address.  It also must be declared volatile, because it could change without any
// change from the program.
//
struct clock_measurement
{
  uint32_t ID;                           // Addr 0
  uint32_t control;                      // Addr 1
  uint32_t num_clocks_measured;          // Addr 2
  uint32_t axi_measured_clock_freq_hz;   // Addr 3
  uint32_t axi_set_ref_clock_freq_hz;    // Addr 4
  uint32_t filler[16-5];                 // Addr 5-15

  uint32_t measured_freqs_hz[48];        // Addr 0x10 - 0x3f

  uint32_t expected_freqs_hz[48];        // Addr 0x10 - 0x3f
};




int main()
{
  volatile clock_measurement* clock_measures = hwalloc<clock_measurement>(HW_RFSoC4x2_BxBDemo1::clock_measurement_0.C_BASEADDR /*0xA00A0000*/, sizeof(clock_measurement));


  if(clock_measures->ID == 0xCCCCCCCC)
    {
	printf("Found clock measurement IP at address 0x%llX, with reference clock frequency %fMHz.\n",
	       (long long)HW_RFSoC4x2_BxBDemo1::clock_measurement_0.C_BASEADDR,
	       clock_measures->axi_measured_clock_freq_hz/1000000.0);
    }
  else
    {
      printf("Expecting clock measurement device at address 0x%llX, but didn't find it.  Expected 0xCCCCCCCC, got 0x%08x\n",
	     (long long)HW_RFSoC4x2_BxBDemo1::clock_measurement_0.C_BASEADDR,
	       clock_measures->ID);
      sleep(10);
      exit(20);
    }

  //clock_measures->control = 0;
  clock_measures->control = 1;

  for(;;)
    {
      usleep(500000);

      printf("\n");
      printf("Reference clock frequency %fMHz.\n", clock_measures->axi_measured_clock_freq_hz/1000000.0);
      
      for(unsigned int i=0; i<clock_measures->num_clocks_measured; i++)
	{
	  uint32_t expected = clock_measures->expected_freqs_hz[i];
	  uint32_t got      = clock_measures->measured_freqs_hz[i];
	  printf("Measured clock %d, expected %fMHz, got %fMHz\n", i, expected/1000000.0, got/1000000.0);
	}
	
    }
  
}
