// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//



#ifndef FREQ2_GRAPH_HH
#define FREQ2_GRAPH_HH

#include "base_data.hh"
#include "data_update.hh"
#include "bxbapp_config.hh"

//#include "graph.hh"

//
// This is a base class that holds data and allows it to be updated by means
// specified in derived classes.
//

template <typename T> class freq2_data : public base_data<T>
{
 public:
  uint32_t             hw_address;

  averaging_type       averaging;
  double               alpha;
  bool                 peak_hold;
  bool                 peak_track;

  int                  adc_number;

  graph_type           source_type;

  volatile int32_t*    hw_data;
  double*              avg_data;

  constexpr static int NUM_PER_PEAK_HOLD = 5;
  constexpr static int NUM_PEAK_HOLD     = 10;
  double*              peak_track_data;
  int                  current_peak_hold;
  int                  num_this_peak_hold;

  float                y_scale;

  volatile hw_pfb_spectrum* pfb_spectrum;

  int                  data_address;
  int                  data_length;

  int                  MAX_STREAMS;
  int                  MAX_SECTIONS;
  int                  MAX_SAMPLES_PER_SECTION;
  
  int                  OFFSET_BETWEEN_SECTIONS;
  int                  OFFSET_BETWEEN_SAMPLES;
  int                  OFFSET_BETWEEN_STREAMS;

  int                  samples_per_section;

  int                  reset_countdown;
  bool                 reset_countdown_done;

  bool                 got_new_data;

  //bool direct_access;
  //int  num_points;

  // May read data directly if direct_access is true, for efficiency.
  //T*   data;

  // Access read this way any freq; read only through this function if direct_access is false
  //virtual T get_data (int index);

  // Scale is set by the end user to say what is expected regarding the x and y of the data.
  // This may be used to create the dataset, if these are set top down.  Often these are
  // set bottom up, so that the top level knows the scale of the data already and this
  // function does nothing.
  //virtual void set_scale(double x_data_start, double x_data_step, double y_scale);

  virtual void update_data();       // Incremental updates, not tied to display  (e.g. averaging)
  virtual void reset_data();        // Reset any updates
  virtual void finalize_data();     // Update prior to display

  virtual void find_ymin_ymax (double& ymin, double& ymax);

  freq2_data(uint32_t hw_address, int num_points, averaging_type averaging, int adc_number, graph_type source_type, float y_scale);  // To use external storage
  virtual ~freq2_data();
};


template<typename T> freq2_data<T>::freq2_data(uint32_t hw_address, int num_points, averaging_type averaging, int adc_number, graph_type source_type, float y_scale)
  : base_data<T>(num_points)
{
  this->hw_address     =  hw_address;
  this->direct_access  =  true;
  this->averaging      =  averaging;
  this->adc_number     =  adc_number;
  this->source_type    =  source_type;
  this->y_scale        =  y_scale;

  
  pfb_spectrum   = hwalloc<hw_pfb_spectrum>  (hw_address);


#ifdef NOTDEF
  printf("\n"
	 "Creating freq2_data with hwaddress=0x%08lX.\n"
	 "\n"
	 "  pfb_spectrum registers:\n"
         "    collect              = %d\n"
	 "    late_delay           = %d\n"
	 "    spectrum_done        = %d\n"
	 "    capture_offset       = 0x%08X\n"
	 "    samples_per_section  = %d\n"
	 "    num_sections         = %d\n"
	 "    num_streams_early    = %d\n"
	 "    num_streams_late     = %d\n"
	 "\n",
	 (long)hw_address,
	 pfb_spectrum->collect,
	 pfb_spectrum->late_delay,
	 pfb_spectrum->spectrum_done,
	 pfb_spectrum->capture_offset,
	 pfb_spectrum->samples_per_section,
	 pfb_spectrum->num_sections,
	 pfb_spectrum->num_streams_early,
	 pfb_spectrum->num_streams_late);
  #endif
  
  MAX_STREAMS              = pfb_spectrum->num_streams_early + pfb_spectrum->num_streams_late;
  MAX_SECTIONS             = 2*(1<<clog2(pfb_spectrum->num_sections));  // MAX_SECTIONS is real sections.  Multiply complex sections by 2.
  MAX_SAMPLES_PER_SECTION  = (1<<clog2(pfb_spectrum->samples_per_section));

  // There are MAX_STREAMS*2 streams.  Odd streams are imaginary, even streams are real.
  OFFSET_BETWEEN_SAMPLES   = 1;
  OFFSET_BETWEEN_SECTIONS  = OFFSET_BETWEEN_SAMPLES * MAX_SAMPLES_PER_SECTION;
  OFFSET_BETWEEN_STREAMS   = OFFSET_BETWEEN_SECTIONS * MAX_SECTIONS;
  
  data_address = hw_address + pfb_spectrum->capture_offset;
  data_length  = OFFSET_BETWEEN_STREAMS * MAX_STREAMS;

  samples_per_section = pfb_spectrum->samples_per_section;
  
  hw_data = hwalloc<int32_t>(data_address, data_length);
  
  avg_data = new double[num_points];
  
  alpha = ( (averaging == AVERAGING_NONE)     ? 1.0   :
	    (averaging == AVERAGING_MINIMAL)  ? 0.9   :
	    (averaging == AVERAGING_MODERATE) ? 0.5   :
	    (averaging == AVERAGING_LARGE)    ? 0.1   :
	    (averaging == AVERAGING_HUGE)     ? 0.01  :
	    (averaging == AVERAGING_IMMENSE)  ? 0.001 :
	    /**/                                0.9  );

  peak_track = (averaging == AVERAGING_PEAK_TRACK);
  peak_hold  = (averaging == AVERAGING_PEAK_HOLD);

  if(peak_track)
    {
      peak_track_data = new double[num_points * NUM_PEAK_HOLD];
    }
  
  reset_data();


  //printf("alpha=%f, peak_hold=%d\n", alpha, peak_hold);
  
  //printf("samples_per_section=%d, OFFSET_BETWEEN_SAMPLES=%d, OFFSET_BETWEEN_SECTIONS=%d\n",
  //	 samples_per_section, OFFSET_BETWEEN_SAMPLES, OFFSET_BETWEEN_SECTIONS);
}


template<typename T> freq2_data<T>::~freq2_data()
{
  hwfree(pfb_spectrum);
  hwfree(hw_data, data_length);
  delete [] avg_data;
}



inline graph_data<float>* create_freq2_graph(graph_type       source_type,
					     int              num_hw_addresses,
					     const int*       adc_numbers,
					     int              num_points,
					     double           x_data_start,
					     double           x_data_step,
					     double           y_scale,
					     averaging_type   averaging)
{
  if(source_type!=TYPE_FREQ_Z1  && source_type!=TYPE_FREQ_Z2)
    {
      printf("ERROR: creating freq2 Z1/Z2 graph from graph_config of different type.\n");
      abort();
    }

  uint32_t hw_address = HW_RFSoC4x2_BxBDemo1::spect_spectral_capture_960_0.C_BASEADDR;    //0xA0400000;

		   
  graph_data<float>* gdata = new graph_data<float>(source_type,
						   num_hw_addresses,
						   num_points,
						   x_data_start,
						   x_data_step,
						   y_scale);
      
  gdata->y_data = new base_data<float>*[num_hw_addresses];
  for(int g=0; g<num_hw_addresses; g++)
    {
      gdata->y_data[g] = new freq2_data<float>(hw_address, num_points, averaging, adc_numbers[g], source_type, y_scale);
    }

  return gdata;
}


template<typename T> void freq2_data<T>::update_data()
{
  extern int  adc_captured_at_this_time_1;
  extern int  adc_captured_at_this_time_2;
  extern bool do_captures;
  extern void update_adc_captured(int adc);
  extern void update_adc_not_captured_but_desired(int adc);

  // Averaging constan t

  
  if(!do_captures)
    return;

  //usleep(100000);
  
  //if(freq2_got_data)
  // return;
  
  for(int wait_timer=0; wait_timer<200; wait_timer++)
    {
      if(pfb_spectrum->spectrum_done == 3)
	break;

      usleep(10);
      //printf("update_data with spectrum_done = %d.  Waiting for finish (3).\n", pfb_spectrum->spectrum_done);
    }

  //freq2_got_data = true;
  
  //for(int i=0; i<10; i++)
  //printf("i=%d done=%d point r[1]=%lld (0x%08x)\n", i, pfb_spectrum->spectrum_done, (long long) (hw_data[8] | 0xF8000000), hw_data[8]| 0xF8000000);

  //for(int i=0; i<10; i++)
  //printf("i=%d done=%d point i[1]=%lld (0x%08x)\n", i, pfb_spectrum->spectrum_done, (long long) (hw_data[9]| 0xF8000000), hw_data[9]| 0xF8000000);

  //printf("graph update for ADC %d, captured are %d %d\n", adc_number, adc_captured_at_this_time_1, adc_captured_at_this_time_2);
  
  bool correct_adc_1 = ( adc_captured_at_this_time_1 == adc_number );
  bool correct_adc_2 = ( adc_captured_at_this_time_2 == adc_number );

  if(correct_adc_1 || correct_adc_2)
    {
      update_adc_captured(adc_number);
    }
  else
    {
      update_adc_not_captured_but_desired(adc_number);
      return;
    }

  got_new_data = true;
  reset_countdown++;

  if(peak_track)
    {
      num_this_peak_hold++;
      
      if(num_this_peak_hold>=NUM_PER_PEAK_HOLD)
	{
	  current_peak_hold = (current_peak_hold+1)%NUM_PEAK_HOLD;
	  num_this_peak_hold=0;
	  for(int point=0; point<this->num_points; point++)
	    {
	      peak_track_data[current_peak_hold*this->num_points+point] = 1.0e-7;
	    }
	}
    }
  
  for(int point=0; point<this->num_points; point++)
    {
      int point_in = (source_type==TYPE_FREQ_Z1) ? point : this->num_points - point;

      if(point_in==0)
	point_in = 1;
      else if(point_in == this->num_points)
	point_in = this->num_points - 1;

      int section_real           =  2 * (point_in / samples_per_section);
      int section_imag           =  section_real + 1;
      int sample_within_section  =  point_in % samples_per_section;

      int index_real = (correct_adc_2?OFFSET_BETWEEN_STREAMS:0) + section_real * OFFSET_BETWEEN_SECTIONS + sample_within_section * OFFSET_BETWEEN_SAMPLES; 
      int index_imag = (correct_adc_2?OFFSET_BETWEEN_STREAMS:0) + section_imag * OFFSET_BETWEEN_SECTIONS + sample_within_section * OFFSET_BETWEEN_SAMPLES; 


      //printf("point=%d, index=%d\n", point_in, index);

      //double re0 = hw_data[index_real];
      //double im0 = hw_data[index_imag];

      //double re1 = hw_data[index_real];
      double re2 = hw_data[index_real];
      //double im1 = hw_data[index_imag];
      double im2 = hw_data[index_imag];

      //double re3 = hw_data[index_real];
      //double im3 = hw_data[index_imag];

      //if(re1!=re2 || re1!=re3 || im1!=im2 || im1!=im3)
      //{
	  //printf("Read differences.  read0=(%f,%f), read_1=(%f,%f), read_2=(%f,%f), read_3=(%f,%f)\n",
	  //	 re0, im0,
	  //	 re1, im1,
	  //	 re2, im2,
	  //	 re3, im3);

      //  printf("Read differences.  read_1=(%f,%f), read_2=(%f,%f), read_3=(%f,%f)\n",
      //	 re1, im1,
      //	 re2, im2,
      //	 re3, im3);
      //}
      
      double power = re2*re2 + im2*im2;

      if(peak_track)
	{
	  if(power>peak_track_data[current_peak_hold*this->num_points+point])
	    peak_track_data[current_peak_hold*this->num_points+point] = power;

	  double v = power;
	  for(int nn=0; nn<NUM_PEAK_HOLD; nn++)
	    {
	      if(v<peak_track_data[nn*this->num_points+point])
		v=peak_track_data[nn*this->num_points+point];
	    }

	  avg_data[point] = v;
	}
      else if(peak_hold)
	{
	  if(power > avg_data[point])
	    avg_data[point] = power;
	}
      else
	{
	  if(reset_countdown_done)
	    {
	      avg_data[point] += alpha * (power + 1.0e-7 - avg_data[point]);
	    }
	  else
	    {
	      //double averaging_alpha = ( (reset_countdown==0)   ?  1.0   :
	      //			 (reset_countdown<5)    ?  0.9   :
	      //			 (reset_countdown<20)   ?  0.5   :
	      //			 (reset_countdown<100)  ?  0.1   :
	      //			 (reset_countdown<200)  ?  0.01  :
	      //			 /**/                      0.001 );

	      //double averaging_alpha = pow(0.977, reset_countdown);
	      double averaging_alpha = pow(0.96, reset_countdown);
	      
	      if(averaging_alpha<=alpha)
		{
		  averaging_alpha = alpha;
		  reset_countdown_done = true;
		}
	      
	      avg_data[point] += averaging_alpha * (power + 1.0e-7 - avg_data[point]);
	    }

	  //double averaging_alpha = reset_countdown_done ? alpha : 1.0/reset_countdown;
	  //avg_data[point] += averaging_alpha * (power + 1.0e-7 - avg_data[point]);
	}
    }

  //fprintf(stderr, "Wrong place for this; temporary\n");
  // Wrong place for this; temporary
  //pfb_spectrum->collect = 0;
  //pfb_spectrum->collect = 1;

  //
  // From data_update.cc
  //
  extern bool update_data_done;
  update_data_done = true;
}



template<typename T> void freq2_data<T>::reset_data()
{
  extern int  adc_captured_at_this_time_1;
  extern int  adc_captured_at_this_time_2;

  got_new_data = false;
  for(int point=0; point<this->num_points; point++)
    {
      avg_data[point] = 1.0e-7;
    }
  reset_countdown = 0;
  reset_countdown_done = false;

  if(peak_track)
    {
      for(int point=0; point<this->num_points * NUM_PEAK_HOLD; point++)
	peak_track_data[point] = 1.0e-7;

      current_peak_hold = 0;
      num_this_peak_hold = 0;
    }

  
  adc_captured_at_this_time_1 = -1;  // Invalidate any currently collected data.  quick and dirty method
  adc_captured_at_this_time_2 = -1;
}


template<typename T> void freq2_data<T>::finalize_data()
{
  if(!got_new_data)
    return;

  got_new_data = false;

  for(int point=0; point<this->num_points; point++)
    {
      // Convert to dB
      // 131.1 is Imperically observed correction to dBFS.  y_scale should be before log, not after.  Fix that here
      this->data[point] = (10.0 * log10(avg_data[point] * y_scale) - 131.1)/y_scale + spectral_gain_adjust_dB;
    }
}


template<typename T> void freq2_data<T>::find_ymin_ymax(double& ymin, double& ymax)
{
  ymin = -110.0 / y_scale;
  ymax = -10.0  / y_scale;
}


#endif
