// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#ifndef GRAPH_DATA_HW_HH
#define GRAPH_DATA_HW_HH

#include <stdint.h>

#include "graph_data.hh"
#include "hwalloc.hh"
#include "base_data.hh"

template <typename T> class graph_data_hw : public graph_data<T>
{
public:
  uint32_t*     hw_addresses;

  // In base class
  // T**       y_data;
  // int       num_graphs;
  // int       num_points;  
  // double    x_data_start;
  // double    x_data_step;
  // double    y_scale;
  //
  // virtual void plot_data(class graph* g);

  graph_data_hw(graph_type       source_type,
		int              num_hw_addresses,
		const uint32_t*  hw_addresses,
		const int*       adc_numbers,
		int              num_points,
		int              interleave,
		double           x_data_start,
		double           x_data_step,
		double           y_scale,
		averaging_type   averaging);
		

  virtual ~graph_data_hw();
};


template <typename T> graph_data_hw<T>::graph_data_hw(graph_type       source_type,
						      int              num_hw_addresses,
						      const uint32_t*  hw_addresses,
						      const int*       adc_numbers,
						      int              num_points,
						      int              interleave,
						      double           x_data_start,
						      double           x_data_step,
						      double           y_scale,
						      averaging_type   averaging)
  : graph_data<T>(source_type,
		  num_hw_addresses,
		  adc_numbers,
		  num_points,
		  interleave,
		  x_data_start,
		  x_data_step,
		  y_scale)
{
  this->averaging = averaging;
  this->hw_addresses = new uint32_t[this->num_graphs];
  for(int i=0; i<this->num_graphs; i++)
    this->hw_addresses[i] = hw_addresses[i];

  this->y_data = new base_data<T>*[this->num_graphs];
  for(int g=0; g<this->num_graphs; g++)
    {
      int num_points_round_to_next_power_of_2 = (1<<clog2(this->num_points));
      this->y_data[g] = new base_data<T>((T*)hwalloc<T>(this->hw_addresses[g], num_points_round_to_next_power_of_2), num_points_round_to_next_power_of_2);
    }
}



template <typename T> graph_data_hw<T>::~graph_data_hw()
{
  for(int g=0; g<this->num_graphs; g++)
    {
      int num_points_round_to_next_power_of_2 = (1<<clog2(this->num_points));
      hwfree(this->y_data[g]->data, num_points_round_to_next_power_of_2);
    }
}



#endif
