// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#ifndef MENU_DAC_CONTROL_HH
#define MENU_DAC_CONTROL_HH

#include "menu.hh"
#include "select_one.hh"
#include "event_types.hh"
#include "number_box.hh"

#define MAX_NUM_DACS 8
extern int NUM_DACS;

struct swept_spread_phase_generator
{
  uint32_t freq_ramp;             // Register 0
  uint32_t freq_min;              // Register 1
  uint32_t freq_max;              // Register 2
  uint32_t spread_period;         // Register 3
  uint32_t spread_code;           // Register 4
  uint32_t sine_shift_right;      // Register 5
  uint32_t freq_fractional;       // Register 6
};

struct dac_info
{
  int                                     dac_number;
  int                                     dac_tile;
  int                                     dac_block;
  char                                    name[128];
  int                                     nyquist_zone;
  uint64_t                                hw_address;
  color                                   dac_color;
  volatile swept_spread_phase_generator*  dac_control;
};


extern dac_info dacs[MAX_NUM_DACS];

class menu_dac_control : public menu
{
  void add(menu* m); // This type of menu is terminal; can't add children.  Thus make this private.
  
public:

  dac_info*  dac;

  volatile struct white_gauss_control* wgc;

  double ramp_rate_MHz_per_second;
  double min_frequency_MHz;
  double max_frequency_MHz;
  double frequency_MHz;        // Center frequency when not in sweep mode
  double modulate_rate_Mbps;
  int    amplitude_shift;

  int nyquist_button_x_offset;
  int nyquist_button_y_offset;
  int nyquist_button_width;
  int nyquist_button_height;

  int enable_button_x_offset;
  int enable_button_y_offset;
  int enable_button_width;
  int enable_button_height;

  int sweep_button_x_offset;
  int sweep_button_y_offset;
  int sweep_button_width;
  int sweep_button_height;

  int modulate_button_x_offset;
  int modulate_button_y_offset;
  int modulate_button_width;
  int modulate_button_height;

  int amplitude_button_x_offset;
  int amplitude_button_y_offset;
  int amplitude_button_width;
  int amplitude_button_height;

  int min_freq_sel_x_offset;
  int min_freq_sel_y_offset;
  int min_freq_sel_width;
  int min_freq_sel_height;

  int max_freq_sel_x_offset;
  int max_freq_sel_y_offset;
  int max_freq_sel_width;
  int max_freq_sel_height;

  int ramp_rate_sel_x_offset;
  int ramp_rate_sel_y_offset;
  int ramp_rate_sel_width;
  int ramp_rate_sel_height;

  int modulate_rate_sel_x_offset;
  int modulate_rate_sel_y_offset;
  int modulate_rate_sel_width;
  int modulate_rate_sel_height;

  int gauss_amplitude_button_x_offset;
  int gauss_amplitude_button_y_offset;
  int gauss_amplitude_button_width;
  int gauss_amplitude_button_height;

  select_one*  nyquist_button;
  multiselect* enable_button;
  select_one*  sweep_button;
  select_one*  modulate_button;
  select_one*  amplitude_button;
  select_one*  gauss_amplitude_button;

  number_box* freq_sel;
  number_box* min_freq_sel;
  number_box* max_freq_sel;
  number_box* ramp_rate_sel;
  number_box* modulate_rate_sel;
  
  connection<select_one, selection_changed_event, menu_dac_control, int, int>* select_trigger;
  connection<number_box, double, menu_dac_control, int, int>*                  number_trigger;

  const static color REGIONCOLOR        = GREY3;

  virtual bool handle_event(my_event& me);
  virtual void draw_dirty();
  virtual void draw_dynamic();
  virtual void layout();

  virtual void this_selected();

  void apply_settings();
  
  menu_dac_control(dac_info&, color bg=BLACK);
  virtual ~menu_dac_control();
};



#endif
