
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//



#ifndef TIME_GRAPH_HH
#define TIME_GRAPH_HH

#include "base_data.hh"
#include "bxbapp_config.hh"
#include "data_sources.hh"
#include "RFSoC4x2_BxBDemo1.hh"

//#include "graph.hh"

//
// This is a base class that holds data and allows it to be updated by means
// specified in derived classes.
//

template <typename T> class time_data : public base_data<T>
{
 public:
  uint32_t hw_address;
  
  //bool direct_access;
  //int  num_points;

  // May read data directly if direct_access is true, for efficiency.
  //T*   data;

  // Access read this way any time; read only through this function if direct_access is false
  //virtual T get_data (int index);

  // Scale is set by the end user to say what is expected regarding the x and y of the data.
  // This may be used to create the dataset, if these are set top down.  Often these are
  // set bottom up, so that the top level knows the scale of the data already and this
  // function does nothing.
  //virtual void set_scale(double x_data_start, double x_data_step, double y_scale);
  
  time_data(uint32_t hw_address, int num_points);  // To use external storage
  virtual ~time_data();
};


//
// hwalloc num_points*2 because the max ratio of group_length/valid_per_group
// is 2.  So the length won't exceed that.  This is because when points_per_clock
// isn't a power of 2 the time points are packed into address space inefficiently.
// So we need more address space than is required for that number of points
// if they had been packed efficiently.
//
template<typename T> time_data<T>::time_data(uint32_t hw_address, int num_points)
  : base_data<T>((T*)hwalloc<T>(hw_address, num_points * 2), num_points)
{
  this->hw_address     =  hw_address;
  this->direct_access  =  true;

  int x = current_config.adc_points_per_clock;
  int y = next_higher_power_of_2(x);
  if(y==x)
    {
      this->valid_per_group = 1;
      this->group_length    = 1;
    }
  else
    {
      this->valid_per_group = x;
      this->group_length    = y;
    }
}


template<typename T> time_data<T>::~time_data()
{
  hwfree(this->data, this->num_points);
}



inline graph_data<int16_t>* create_time_graph(graph_type       source_type,
					      int              num_hw_addresses,
					      const int*       adc_numbers_sw,
					      int              num_points,
					      double           x_data_start,
					      double           x_data_step,
					      double           y_scale)
{
  if(source_type!=TYPE_TIME)
    {
      printf("ERROR: creating time graph from graph_config of different type.\n");
      abort();
    }

		   
  graph_data<int16_t>* gdata = new graph_data<int16_t>(source_type,
						       num_hw_addresses,
						       num_points,
						       x_data_start,
						       x_data_step,
						       y_scale);
      
  gdata->y_data = new base_data<int16_t>*[num_hw_addresses];
  for(int g=0; g<num_hw_addresses; g++)
    {
      int       adc_number_hw = sw_source_to_hw_adc(adc_numbers_sw[g]);
      //uint32_t  hw_address    = 0xB0000000 + adc_number_hw * 0x00020000;


      uint32_t  hw_address    = (

#ifdef HAVE_real_sample_capture_01
				 (adc_number_hw == 1) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_01.C_BASEADDR :
#endif
				 
#ifdef HAVE_real_sample_capture_02
				 (adc_number_hw == 2) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_02.C_BASEADDR :
#endif
				 
#ifdef HAVE_real_sample_capture_03
				 (adc_number_hw == 3) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_03.C_BASEADDR :
#endif
				 
#ifdef HAVE_real_sample_capture_04
				 (adc_number_hw == 4) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_04.C_BASEADDR :
#endif
				 
#ifdef HAVE_real_sample_capture_05
				 (adc_number_hw == 5) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_05.C_BASEADDR :
#endif
				 
#ifdef HAVE_real_sample_capture_06
				 (adc_number_hw == 6) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_06.C_BASEADDR :
#endif
				 
#ifdef HAVE_real_sample_capture_07
				 (adc_number_hw == 7) ? HW_RFSoC4x2_BxBDemo1::real_sample_capture_07.C_BASEADDR :
#endif

				      
				 /**/                   HW_RFSoC4x2_BxBDemo1::real_sample_capture_00.C_BASEADDR );

      
      gdata->y_data[g] = new time_data<int16_t>(hw_address, num_points);
    }

  return gdata;
}



//template<typename T> time_data<T>::update_data()
//{
//}

#endif
