
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//



#ifndef XFER_GRAPH_HH
#define XFER_GRAPH_HH

#include <stdio.h>

#include "base_data.hh"
#include "data_update.hh"

#include "graph.hh"
#include "hwalloc.hh"
#include "RFSoC4x2_BxBDemo1.hh"

//
// This class interfaces with the hardware.  One hardware interface will work for MAG, PHASE, and GROUP_DELAY plots
//
class xfer_hw
{
public:
  int                         use_count;

  constexpr static int        MAX_XFER_HW = 50;
  static class xfer_hw*       hw_xfer_list[MAX_XFER_HW];
  static int                  num_hw_xfer;

  int                         adc_from;
  int                         adc_to;
  bool                        first_nyquist;

  volatile int32_t*           hw_data;
  double*                     avg_power_from;
  double*                     avg_corr_real;
  double*                     avg_corr_imag;

  double                      alpha;
  int                         reset_countdown;
  bool                        reset_countdown_done;

  unsigned int                seq;

  uint32_t                    hw_address;
  int                         num_points;
  
  volatile hw_pfb_spectrum*   pfb_spectrum;

  int                         data_address;
  int                         data_length;

  int                         MAX_STREAMS;
  int                         MAX_SECTIONS;
  int                         MAX_SAMPLES_PER_SECTION;
  
  int                         OFFSET_BETWEEN_SECTIONS;
  int                         OFFSET_BETWEEN_SAMPLES;
  int                         OFFSET_BETWEEN_STREAMS;

  int                         samples_per_section;

  double                      graph_xmin;  // For a hack to keep all axes in sync that use a common xfer_hw
  double                      graph_xmax;
  
  void reset(unsigned int& seq);

  // Seq is also kept by all clients.  They call with their seq, and get current seq back in return.  This allows
  // multiple clients to call and not update on every call.  Only update when new seq is less than current seq.
  void update_data(unsigned int seq);
  bool got_new_data(unsigned int& seq);
  
  xfer_hw(int adc_from, int adc_to, bool first_nyquist, uint32_t hw_address, int num_points, double alpha);
  ~xfer_hw();
};

inline xfer_hw::xfer_hw(int adc_from, int adc_to, bool first_nyquist, uint32_t hw_address, int num_points, double alpha)
{
  use_count             = 1;

  this->adc_from        = adc_from;
  this->adc_to          = adc_to;
  this->first_nyquist   = first_nyquist;

  reset_countdown       = 0;
  reset_countdown_done  = false;

  this->num_points      = num_points;
  this->hw_address      = hw_address;

  this->alpha           = alpha;
  
  pfb_spectrum          = hwalloc<hw_pfb_spectrum>(hw_address);

#ifdef NOTDEF
  printf("\n"
	 "Creating xfer_data.\n"
	 "\n"
	 "  pfb_spectrum registers:\n"
         "    collect              = %d\n"
	 "    late_delay           = %d\n"
	 "    spectrum_done        = %d\n"
	 "    capture_offset       = 0x%08X\n"
	 "    samples_per_section  = %d\n"
	 "    num_sections         = %d\n"
	 "    num_streams_early    = %d\n"
	 "    num_streams_late     = %d\n"
	 "\n",
	 pfb_spectrum->collect,
	 pfb_spectrum->late_delay,
	 pfb_spectrum->spectrum_done,
	 pfb_spectrum->capture_offset,
	 pfb_spectrum->samples_per_section,
	 pfb_spectrum->num_sections,
	 pfb_spectrum->num_streams_early,
	 pfb_spectrum->num_streams_late);
#endif
  
  MAX_STREAMS              = pfb_spectrum->num_streams_early + pfb_spectrum->num_streams_late;
  MAX_SECTIONS             = 2*(1<<clog2(pfb_spectrum->num_sections));  // MAX_SECTIONS is real sections.  Multiply complex sections by 2.
  MAX_SAMPLES_PER_SECTION  = (1<<clog2(pfb_spectrum->samples_per_section));

  // There are MAX_STREAMS*2 streams.  Odd streams are imaginary, even streams are real.
  OFFSET_BETWEEN_SAMPLES   = 1;
  OFFSET_BETWEEN_SECTIONS  = OFFSET_BETWEEN_SAMPLES * MAX_SAMPLES_PER_SECTION;
  OFFSET_BETWEEN_STREAMS   = OFFSET_BETWEEN_SECTIONS * MAX_SECTIONS;
  
  data_address = hw_address + pfb_spectrum->capture_offset;
  data_length  = OFFSET_BETWEEN_STREAMS * MAX_STREAMS;

  samples_per_section = pfb_spectrum->samples_per_section;
  
  hw_data = hwalloc<int32_t>(data_address, data_length);
  
  avg_power_from          = new double   [num_points];
  avg_corr_real           = new double   [num_points];
  avg_corr_imag           = new double   [num_points];

  graph_xmin = 0.0;
  graph_xmax = 0.0;

  reset(seq);
}


inline void xfer_hw::reset(unsigned int& seq)
{
  seq = this->seq = 0;
  
  extern int  adc_captured_at_this_time_1;
  extern int  adc_captured_at_this_time_2;

  for(int point=0; point<this->num_points; point++)
    {
      avg_power_from[point] = 1.0e-30;
      avg_corr_real[point] = 0.0;
      avg_corr_imag[point] = 0.0;
    }

  adc_captured_at_this_time_1 = -1;  // Invalidate any currently collected data.  quick and dirty method
  adc_captured_at_this_time_2 = -1;

  pfb_spectrum->late_delay = 0;
  
  reset_countdown = 0;
  reset_countdown_done = false;
}


inline xfer_hw::~xfer_hw()
{
  for(int i=0,j=0; j<num_hw_xfer; i++,j++)
    {
      if(hw_xfer_list[i]==this)
	{
	  i++;
	  num_hw_xfer--;
	}
      hw_xfer_list[j] = hw_xfer_list[i];
    }

  hwfree(pfb_spectrum);
  hwfree(hw_data, data_length);
  delete [] avg_power_from;
  delete [] avg_corr_real;
  delete [] avg_corr_imag;
}

inline xfer_hw* new_xfer_hw(int adc_from, int adc_to, bool first_nyquist, uint32_t hw_address, int num_points, double alpha)
{
  for(int i=0; i<xfer_hw::num_hw_xfer; i++)
    {
      auto hwxf = xfer_hw::hw_xfer_list[i];
      if(hwxf->adc_from==adc_from && hwxf->adc_to==adc_to && hwxf->first_nyquist==first_nyquist && hwxf->alpha == alpha)
	{
	  hwxf->use_count++;
	  return hwxf;
	}
    }

  auto hwxf = new xfer_hw(adc_from, adc_to, first_nyquist, hw_address, num_points, alpha);
  xfer_hw::hw_xfer_list[xfer_hw::num_hw_xfer++] = hwxf;

  return hwxf;
}

inline void delete_xfer_hw(xfer_hw* hwxf)
{
  hwxf->use_count--;
  if(hwxf->use_count==0)
    delete hwxf;
}


inline bool xfer_hw::got_new_data(unsigned int& seq)
{
  bool isnew = (seq!=this->seq);
  seq = this->seq;
  return isnew;
}

inline void xfer_hw::update_data(unsigned int seq)
{
  extern int  adc_captured_at_this_time_1;
  extern int  adc_captured_at_this_time_2;
  extern bool do_captures;
  extern void update_adc_pair_captured(int adc1, int adc2);
  extern void update_adc_pair_not_captured_but_desired(int adc1, int adc2);

  //printf("update_data called with this->seq=%d, seq=%d\n", this->seq, seq);
  
  if(seq!=this->seq)
    return;

  if(!do_captures)
    return;

  bool correct_adc_1 = ( adc_captured_at_this_time_1 == adc_from );
  bool correct_adc_2 = ( adc_captured_at_this_time_2 == adc_to );

  if(correct_adc_1 && correct_adc_2)
    {
      update_adc_pair_captured(adc_from, adc_to);
    }
  else
    {
      update_adc_pair_not_captured_but_desired(adc_from, adc_to);
      return;
    }
  
  reset_countdown++;
  this->seq++;
      
  double averaging_alpha = reset_countdown_done ? alpha : pow(0.9, reset_countdown);
  
  if(!reset_countdown_done)
    {
      if(averaging_alpha<=alpha)
	{
	  averaging_alpha = alpha;
	  reset_countdown_done = true;
	}
    }
  
  for(int point=0; point<this->num_points; point++)
    {
      int point_in = first_nyquist ? point : this->num_points - point;
      
      if(point_in==0)
	point_in = 1;
      else if(point_in == this->num_points)
	point_in = this->num_points - 1;
      
      int section_real           =  2 * (point_in / samples_per_section);
      int section_imag           =  section_real + 1;
      int sample_within_section  =  point_in % samples_per_section;
      
      int index_from_real = section_real * OFFSET_BETWEEN_SECTIONS + sample_within_section * OFFSET_BETWEEN_SAMPLES; 
      int index_from_imag = section_imag * OFFSET_BETWEEN_SECTIONS + sample_within_section * OFFSET_BETWEEN_SAMPLES; 
      
      int index_to_real = OFFSET_BETWEEN_STREAMS + section_real * OFFSET_BETWEEN_SECTIONS + sample_within_section * OFFSET_BETWEEN_SAMPLES; 
      int index_to_imag = OFFSET_BETWEEN_STREAMS + section_imag * OFFSET_BETWEEN_SECTIONS + sample_within_section * OFFSET_BETWEEN_SAMPLES; 

      // For testing
      //index_from_real = index_to_real;
      //index_from_imag = index_to_imag;
	  
      //printf("point=%d, index=%d\n", point_in, index);
      
      double re_from = hw_data[index_from_real];
      double im_from = hw_data[index_from_imag];
      
      double re_to = hw_data[index_to_real];
      double im_to = hw_data[index_to_imag];
      
      
      double power_from = re_from*re_from + im_from*im_from;
      double corr_real  = re_from*re_to   + im_from*im_to;  // to * conj(from)
      double corr_imag  = re_from*im_to   - re_to*im_from;
      
      avg_power_from[point] += averaging_alpha * (power_from + 1.0e-30 - avg_power_from[point]);
      avg_corr_real[point]  += averaging_alpha * (corr_real  - avg_corr_real[point]);
      avg_corr_imag[point]  += averaging_alpha * (corr_imag  - avg_corr_imag[point]);
    }

  extern bool update_data_done;
  update_data_done = true;
}


template <typename T> class xfer_data_interp : public base_data<T>
{
 public:
  virtual void find_ymin_ymax (double& ymin, double& ymax) {}     // derive this class simply so that find_ymin_ymax doesn't do anything.
  xfer_data_interp(int num_points) : base_data<T>(num_points) {}
  virtual ~xfer_data_interp() {}
};


//
// This is a base class that holds data and allows it to be updated by means
// specified in derived classes.
//

template <typename T> class xfer_data : public base_data<T>
{
 public:
  averaging_type              averaging;
  double                      alpha;

  char                        label[200];
  
  int                         adc_from;
  int                         adc_to;

  graph_type                  source_type;
  
  float                       y_scale;

  double                      slope_average;
  double                      corrected_slope_average;
  constexpr static double     slope_average_alpha = 0.5;

  double*                     phase_cycles;
  double*                     delta_phase_cycles;
  double*                     avg_phase_error_cycles;
  bool*                       excluded;

  int                         display_count;

  double                      min_group_delay;
  double                      max_group_delay;
  double                      last_min_group_delay;
  double                      last_max_group_delay;
  constexpr static double     default_min_group_delay  =  12.5;
  constexpr static double     default_max_group_delay  = 137.5;

  double                      min_phase;
  double                      max_phase;
  double                      last_min_phase;
  double                      last_max_phase;
  constexpr static double     default_min_phase  = -1.0;
  constexpr static double     default_max_phase  =  1.0;

  bool                        first_nyquist;

  constexpr static int        lower_cutoff = 500;

  xfer_hw*                    xfhw;
  unsigned int                seq;

  double                      graph_xmin;  // For a hack to keep all axes in sync that use a common xfer_hw
  double                      graph_xmax;

  base_data<T>*               interpolated_curve;
  
  //bool direct_access;
  //int  num_points;

  // May read data directly if direct_access is true, for efficiency.
  //T*   data;

  // Access read this way any freq; read only through this function if direct_access is false
  //virtual T get_data (int index);

  // Scale is set by the end user to say what is expected regarding the x and y of the data.
  // This may be used to create the dataset, if these are set top down.  Often these are
  // set bottom up, so that the top level knows the scale of the data already and this
  // function does nothing.
  //virtual void set_scale(double x_data_start, double x_data_step, double y_scale);

  virtual void update_data();       // Incremental updates, not tied to display  (e.g. averaging)
  virtual void reset_data();        // Reset any updates
  virtual void finalize_data();     // Update prior to display

  virtual void find_ymin_ymax (double& ymin, double& ymax);

  xfer_data(uint32_t hw_address, int num_points, averaging_type averaging, int adc_from, int adc_to, graph_type source_type, float y_scale, base_data<T>* interpolated_curve);
  virtual ~xfer_data();
};


template<typename T> xfer_data<T>::xfer_data(uint32_t hw_address, int num_points, averaging_type averaging, int adc_from, int adc_to, graph_type source_type, float y_scale, base_data<T>* interpolated_curve)
  : base_data<T>(num_points)
{
  this->direct_access       =  true;
  this->averaging           =  averaging;
  this->adc_from            =  adc_from;
  this->adc_to              =  adc_to;
  this->source_type         =  source_type;
  this->y_scale             =  y_scale;
  this->interpolated_curve  =  interpolated_curve;

  first_nyquist = (source_type==TYPE_Z1_TRANSFER_MAG || source_type==TYPE_Z1_TRANSFER_PHASE || source_type==TYPE_Z1_TRANSFER_PHASE_DEBUG || source_type==TYPE_Z1_TRANSFER_GROUP_DELAY);
  
  seq = 0;

  phase_cycles            = new double   [num_points];
  delta_phase_cycles      = new double   [num_points];
  avg_phase_error_cycles  = new double   [num_points];
  excluded                = new bool     [num_points];
  
  alpha = ( (averaging == AVERAGING_NONE)     ? 1.0   :
	    (averaging == AVERAGING_MINIMAL)  ? 0.9   :
	    (averaging == AVERAGING_MODERATE) ? 0.5   :
	    (averaging == AVERAGING_LARGE)    ? 0.1   :
	    (averaging == AVERAGING_HUGE)     ? 0.01  :
	    (averaging == AVERAGING_IMMENSE)  ? 0.001 :
	    /**/                                0.9  );

  xfhw = new_xfer_hw(adc_from, adc_to, first_nyquist, hw_address, num_points, alpha);

  reset_data();

  graph_xmin = 0.0;
  graph_xmax = 0.0;

  //printf("samples_per_section=%d, OFFSET_BETWEEN_SAMPLES=%d, OFFSET_BETWEEN_SECTIONS=%d\n",
  //	 samples_per_section, OFFSET_BETWEEN_SAMPLES, OFFSET_BETWEEN_SECTIONS);
}


template<typename T> xfer_data<T>::~xfer_data()
{
  delete_xfer_hw(xfhw);

  delete [] avg_phase_error_cycles;
  delete [] phase_cycles;
  delete [] delta_phase_cycles;
  delete [] excluded;
}



inline graph_data<float>* create_xfer_graph(graph_type        source_type,
					    int               num_hw_addresses,
					    const int*        adc_numbers,
					    int               reference,
					    int               num_points,
					    double            x_data_start,
					    double            x_data_step,
					    double            y_scale,
					    averaging_type    averaging)
 {
  if(source_type!=TYPE_Z1_TRANSFER_MAG  && source_type!=TYPE_Z1_TRANSFER_PHASE_DEBUG && source_type!=TYPE_Z1_TRANSFER_PHASE && source_type!=TYPE_Z2_TRANSFER_MAG  && source_type!=TYPE_Z2_TRANSFER_PHASE_DEBUG && source_type!=TYPE_Z2_TRANSFER_PHASE && source_type!=TYPE_Z1_TRANSFER_GROUP_DELAY && source_type!=TYPE_Z2_TRANSFER_GROUP_DELAY )
    {
      printf("ERROR: creating transfer mag/phase graph from graph_config of different type.\n");
      abort();
    }

  uint32_t hw_address = HW_RFSoC4x2_BxBDemo1::spect_spectral_capture_960_0.C_BASEADDR;    //0xA0400000;


  bool has_interpolated_curve = false;
  
  if( source_type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
      source_type==TYPE_Z1_TRANSFER_PHASE        ||
      source_type==TYPE_Z2_TRANSFER_PHASE_DEBUG  ||
      source_type==TYPE_Z2_TRANSFER_PHASE        ||
      source_type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
      source_type==TYPE_Z2_TRANSFER_GROUP_DELAY  )
    {
      if(num_hw_addresses!=1)
	{
	  printf("ERROR: creating transfer phase or group delay graph with more than one curve.\n");
	  abort();
	}

      // These graph types add a second curve that is a curve fit.
      has_interpolated_curve  = true;
      num_hw_addresses = 2;
    }

  //printf("xfer_graph.hh::create_xfer_graph from %d to \n", reference);
  //for(int i=0; i<num_hw_addresses; i++)
  //printf("%d%s", adc_numbers[i], (i==num_hw_addresses-1)?"\n":", ");
  		   
  graph_data<float>* gdata = new graph_data<float>(source_type,
						   num_hw_addresses,
						   num_points,
						   x_data_start,
						   x_data_step,
						   y_scale);
      
  int adc_from = reference;
  gdata->y_data = new base_data<float>*[num_hw_addresses];
  for(int i=0; i<num_hw_addresses; i++)
    {
      int adc_to = has_interpolated_curve ? adc_numbers[i-1] : adc_numbers[i];
      bool interpolated_curve = has_interpolated_curve && (i==0);

      if(interpolated_curve)
	{
	  gdata->y_data[i] = new xfer_data_interp<float>(num_points);
	}
      else if(has_interpolated_curve)
	{
	  gdata->y_data[i] = new xfer_data<float>(hw_address, num_points, averaging, adc_from, adc_to, source_type, y_scale, gdata->y_data[i-1]);
	}
      else
	{
	  gdata->y_data[i] = new xfer_data<float>(hw_address, num_points, averaging, adc_from, adc_to, source_type, y_scale, 0);
	}
    }
  
  return gdata;
}


template<typename T> void xfer_data<T>::update_data()
{
  xfhw->update_data(seq);
}




template<typename T> void xfer_data<T>::reset_data()
{
  xfhw->reset(seq);
  
  for(int point=0; point<this->num_points; point++)
    {
      this->data[point] = 0.0;
      avg_phase_error_cycles[point] = 0.0;
      phase_cycles[point] = 0.0;
      delta_phase_cycles[point] = 0.0;
      excluded[point] = false;
    }

  slope_average = 0.0;
  corrected_slope_average = 0.0;
  
  display_count = 0;
  
  min_group_delay = default_min_group_delay;
  max_group_delay = default_max_group_delay;

  min_phase = default_min_phase;
  max_phase = default_max_phase;

  last_min_group_delay = default_min_group_delay;
  last_max_group_delay = default_max_group_delay;

  last_min_phase = default_min_phase;
  last_max_phase = default_max_phase;
}


template<typename T> void xfer_data<T>::finalize_data()
{
  int displayed_start_point = 0;
  int displayed_end_point = this->num_points-1;
  
  if(this->g)
    {
      displayed_start_point = (this->g->xmin - this->g->gd->x_data_start) / this->g->gd->x_data_step;
      displayed_end_point   = (this->g->xmax - this->g->gd->x_data_start) / this->g->gd->x_data_step;

      // If graph has changed from our last copy, update this and other graphs.
      // But if we still match the graph and we don't match the xfhw, then someone
      // else got an update and we need to match ourselves and our graph.
      if(this->g->xmin != graph_xmin || this->g->xmax != graph_xmax)
	{
	  graph_xmin = this->g->xmin;
	  graph_xmax = this->g->xmax;
	  xfhw->graph_xmin = graph_xmin/this->g->gd->x_data_step;
	  xfhw->graph_xmax = graph_xmax/this->g->gd->x_data_step;

	  //printf("displayed_start_point=%d, displayed_end_point=%d\n", displayed_start_point, displayed_end_point);
	}
      else if( float(xfhw->graph_xmin*this->g->gd->x_data_step) != float(graph_xmin) ||
	       float(xfhw->graph_xmax*this->g->gd->x_data_step) != float(graph_xmax) )
	{
	  this->g->xmin = graph_xmin = xfhw->graph_xmin * this->g->gd->x_data_step;
	  this->g->xmax = graph_xmax = xfhw->graph_xmax * this->g->gd->x_data_step;
	  this->g->mark_layout_dirty();
	  if(this->g->external_xmin)
	    *this->g->external_xmin = this->g->xmin;
	  if(this->g->external_xmax)
	    *this->g->external_xmax = this->g->xmax;
	}
    }

  int percent_in  = (displayed_end_point - displayed_start_point) * 15 / 100;
  int start_plus  = displayed_start_point + percent_in;
  int end_minus   = displayed_end_point   - percent_in;

  if(start_plus<0)
    start_plus=0;
  if(end_minus<0)
    end_minus=0;
  if(start_plus>=this->num_points)
    start_plus=this->num_points-1;
  if(end_minus>=this->num_points)
    end_minus=this->num_points-1;

  
  if(!xfhw->got_new_data(seq))
    return;

  display_count++;
  
  if(source_type==TYPE_Z1_TRANSFER_MAG || source_type==TYPE_Z2_TRANSFER_MAG)
    {
      for(int point=0; point<this->num_points; point++)
	{
	  double xfer_power = (hypot(xfhw->avg_corr_real[point], xfhw->avg_corr_imag[point]) + 1.0e-30) / xfhw->avg_power_from[point];
	  double dB = (20.0 * log10(xfer_power * y_scale))/y_scale;
	  if(dB<-100.0)
	    dB = -100.0;
	  if(dB>100.0)
	    dB = 100.0;
	  if(point<lower_cutoff)
	    this->data[point] = NAN;
	  else
	    this->data[point] = dB;
	}
    }
  else //   source_type == TYPE_Z1_TRANSFER_PHASE or similar
    {
      //
      // Get phase
      //
      for(int point=0; point<this->num_points; point++)
	{
	  double xr       = xfhw->avg_corr_real[point];
	  double xi       = xfhw->avg_corr_imag[point];

	  double cycles = atan2(xi, xr) / (2.0*M_PI);

	  phase_cycles[point] = first_nyquist ? cycles : -cycles;
	}

      // Get delta phase averaged over a number of points
      delta_phase_cycles[0] = 0.0;
      for(int point=1; point<this->num_points; point++)
	{
	  double last_delta = 100.0;
	  int p;
	  for(p=0; p<20; p++)
	    {
	      if(point+p>=this->num_points || point-1-p<0)
		break;
	      
	      double delta = phase_cycles[point+p] - phase_cycles[point-1-p];
	      
	      if(delta>0.5)
		delta -= 1.0;
	      if(delta<-0.5)
		delta += 1.0;
	      
	      if(abs(delta)>0.1)
		{
		  if(p==0)
		    p = 1;
		  break;
		}
	      last_delta = delta;
	    }
	  delta_phase_cycles[point] = last_delta / (2*p-1);

	  //if(delta_phase_cycles[point]>0.5 || delta_phase_cycles[point]<-0.5)
	  //  {
	  //    printf("A delta_phase_cycles[%d]=%f\n", point, delta_phase_cycles[point]);
	  //  }
	}

      
      // Find the average delta from point to point

      double self_benefit                      = 0.1;
      double force_inclusion_threshold_cycles  = 0.2;  // If less than this, include
      double force_exclusion_threshold_cycles  = 0.5;  // If more than this, exclude
      
      excluded[0] = true;
      excluded[1] = true;
      excluded[this->num_points-1] = true;
      excluded[this->num_points-2] = true;

      avg_phase_error_cycles[0] = 1.0;
      avg_phase_error_cycles[1] = 1.0;
      avg_phase_error_cycles[this->num_points-1] = 1.0;
      avg_phase_error_cycles[this->num_points-2] = 1.0;
      
      double last_delay_time_ns = 0.0;
      extern double freq_bin_separation_MHz;

      for(int point=2; point<this->num_points-2; point++)
	{
	  double delta_between = delta_phase_cycles[point];
	  
	  double delay_time_ns = -delta_between * (1000.0 / freq_bin_separation_MHz);
	  bool bad_point = (point<lower_cutoff) || (abs(delay_time_ns-last_delay_time_ns)>100.0) || (abs(delta_between)>0.5);
	  last_delay_time_ns = delay_time_ns;
	    
	  
	  double center_benefit = excluded[point] ? self_benefit : -self_benefit;
	  
	  double delta = bad_point ? 1.0 : 0.0;

	  double phe_delta = 0.5;

	  avg_phase_error_cycles[point] = avg_phase_error_cycles[point] + phe_delta * (delta - avg_phase_error_cycles[point]);
	  avg_phase_error_cycles[point] = 0.4 * avg_phase_error_cycles[point] + 0.3 * ( avg_phase_error_cycles[point-1] + avg_phase_error_cycles[point+1] );
	  	  
	  double thr = avg_phase_error_cycles[point] + center_benefit;

	  if(bad_point)
	    {
	      excluded[point] = true;
	      excluded[point-1] = true;
	    }
	  else if(thr < force_inclusion_threshold_cycles)
	    {
	      excluded[point] = false;
	    }
	  else if(thr>force_exclusion_threshold_cycles)
	    {
	      excluded[point] = true;
	    }	  
	}


      //
      // New code to find corrected_slope_average and fix the averaging region.
      //
      int interp_start = start_plus;
      int interp_end = start_plus;

      int i = start_plus;
      for(;;)
	{
	  while(excluded[i] && i<end_minus)
	    i++;
	  if(i==end_minus)
	    break;
	  
	  // Got a region start
	  int rstart = i;
	  while(!excluded[i] && i<end_minus)
	    {
	      double phdelta = phase_cycles[i]-phase_cycles[i-1];
	      phdelta -= floor(phdelta+0.5);
	      phdelta = abs(phdelta);
	      if(phdelta<0.125)
		i++;
	    }

	  // Got a region end
	  int rend = i-1;
	  
	  // If region is larger than previous region, use it.
	  if(rend-rstart>interp_end-interp_start)
	    {
	      interp_start = rstart;
	      interp_end   = rend;
	    }
	}

      //
      // Now that we have the region, add the phase and then divide by the length.
      //
      double phase_offset        = 0.0;
      
      if(interp_end!=interp_start)
	{
	  double ph = 0.0;
	  for(int i=interp_start+1; i<=interp_end; i++)
	    {
	      double phdelta = phase_cycles[i]-phase_cycles[i-1];
	      phdelta -= floor(phdelta+0.5);
	      ph += phdelta;
	    }
	  corrected_slope_average = ph / (interp_end-interp_start);
	  phase_offset = phase_cycles[interp_start];

	  if(this->g)
	    {
	      sprintf(label, "%cAverage delay %.3fns", 10+graph::MAX_GRAPHS, -corrected_slope_average * (1000.0 / freq_bin_separation_MHz));
	      this->g->labels[0] = label;
	    }
	}
      else
	{
	  corrected_slope_average = 0.0;
	  phase_offset = 0.0;

	  if(this->g)
	    {
	      sprintf(label, "%cPut good phases in center to measure delay", 10+graph::MAX_GRAPHS);
	      this->g->labels[0] = label;
	    }
	}
      
      
	  //printf("average phase clope is %f cycles/point, averaged to %f\n", avg_phase_slope_cycles, slope_average);

      if(source_type==TYPE_Z1_TRANSFER_GROUP_DELAY || source_type==TYPE_Z2_TRANSFER_GROUP_DELAY)
	{
	  min_group_delay = -100.0;
	  max_group_delay =  100.0;
	  bool first = true;
	  
	  for(int point=0; point<this->num_points; point++)
	    {
	      extern double freq_bin_separation_MHz;
	      
	      if(excluded[point])
		{
		  this->data[point] = NAN;
		}
	      else
		{
		  double sum = 0.0;
		  int    cnt = 0;
		  for(int p=-20; p<=20; p++)
		    {
		      int q = point+p;
		      if(q>0 && q<this->num_points && !excluded[q])
			{
			  double d = delta_phase_cycles[q];
			  if(isfinite(d))
			    {
			      sum += d;
			      cnt++;
			    }
			}
		    }

		  if(cnt)
		    {
		      double delay_time_nseconds = -sum / cnt * (1000.0 / freq_bin_separation_MHz);

		      this->data[point] = delay_time_nseconds;
		      
		      if(first)
			{
			  min_group_delay = delay_time_nseconds;
			  max_group_delay = delay_time_nseconds;
			  first = false;
			}
		      else
			{
			  if(delay_time_nseconds<min_group_delay) min_group_delay = delay_time_nseconds;
			  if(delay_time_nseconds>max_group_delay) max_group_delay = delay_time_nseconds;
			}
		    }
		  else
		    {
		      this->data[point] = NAN;
		    }
		}
	    }

#ifdef NOTDEF
	  if(( (display_count%20)==0) && this->g && ( (min_group_delay<last_min_group_delay- 10)     ||
						      (min_group_delay>last_min_group_delay+ 50)     ||
						      (max_group_delay>last_max_group_delay+ 10)     ||
						      (max_group_delay<last_max_group_delay- 50)     ))
	    {
	      //printf("display_count is %d.  resetting group delay to (%.30f,%.30f)\n", display_count, min_group_delay, max_group_delay);
	      if(min_group_delay==max_group_delay)
		this->g->set_ymin_ymax(min_group_delay-100, max_group_delay+100);
	      else
		this->g->set_ymin_ymax(min_group_delay-20, max_group_delay+20);
	      this->g->mark_layout_dirty();
	      last_min_group_delay = min_group_delay;
	      last_max_group_delay = max_group_delay;
	    }
#endif
	}
      else
	{
	  
	  // Get the phase, subtracting off the average delta
	  double last_phase = 0.0;
	  double avg_phase = phase_offset - corrected_slope_average * interp_start;
	  avg_phase -= floor(avg_phase);
	  for(int point=0; point<this->num_points; point++)
	    {
	      double d= phase_cycles[point] - avg_phase + floor(last_phase);

	      while(d-last_phase<-0.5)
		d = d + 1.0;
	      while(d-last_phase>0.5)
		d = d - 1.0;

	      double c = floor(d+0.5);
	      if(point==interp_start && c!=0.0)
		{
		  // At wrong level by a factor of 1.0.  Correct.
		  //printf("Correcting unwrap by value of %f\n", c);
		  for(int i=0; i<interp_start; i++)
		    this->data[i] -= c;
		  d -= c;
		}
	      
	      if(!excluded[point])
		last_phase = d;

	      if(point<lower_cutoff)
		this->data[point] = NAN;
	      else
		this->data[point] = d;

	      if(point==displayed_start_point)
		{
		  min_phase = d;
		  max_phase = d;
		}
	      else if(point>displayed_start_point&&point<displayed_end_point)
		{
		  if(d<min_phase) min_phase = d;
		  if(d>max_phase) max_phase = d;
		}

	      avg_phase += corrected_slope_average - floor(avg_phase);  // floor is to keep it from getting too large
	    }

#ifdef NOTDEF
	  if(( (display_count%20)==0) && this->g && ( (abs(min_phase-last_min_phase)>2) || (abs(max_phase-last_max_phase)>2)) )
	    {
	      //printf("display_count is %d.  resetting group delay\n", display_count);
	      if(min_phase==max_phase)
		this->g->set_ymin_ymax(min_phase-1, max_phase+1);
	      else
		this->g->set_ymin_ymax(min_phase, max_phase);
	      this->g->mark_layout_dirty();
	      last_min_phase = min_phase;
	      last_max_phase = max_phase;
	    }
#endif
	}

      
      if(interpolated_curve)
	{
	  for(int i=0; i<=interp_start; i++)
	    interpolated_curve->data[i] = NAN;

	  if(source_type==TYPE_Z1_TRANSFER_PHASE       || source_type==TYPE_Z2_TRANSFER_PHASE       ||
	     source_type==TYPE_Z1_TRANSFER_PHASE_DEBUG || source_type==TYPE_Z2_TRANSFER_PHASE_DEBUG )
	    {
	      for(int i=interp_start+1; i<interp_end; i++)
		interpolated_curve->data[i] = 0.0;
	    }
	  else
	    {
	      T value = -corrected_slope_average  * (1000.0 / freq_bin_separation_MHz);
	      for(int i=interp_start+1; i<interp_end; i++)
		interpolated_curve->data[i] = value;
	      //printf("corrected_slope_average=%f (%fns)\n", corrected_slope_average, corrected_slope_average  * (1000.0 / freq_bin_separation_MHz));
	    }
	  
	  for(int i=interp_end; i<this->num_points; i++)
	    interpolated_curve->data[i] = NAN;
	}
    
      if(source_type==TYPE_Z1_TRANSFER_PHASE_DEBUG || source_type==TYPE_Z2_TRANSFER_PHASE_DEBUG)
	{
#ifdef NOTDEF
	  for(int point=0; point<this->num_points; point++)
	    {
	      this->data[point] = excluded[point] ? -avg_phase_error_cycles[point] : avg_phase_error_cycles[point];
	    }
#endif


	  for(int point=0; point<this->num_points; point++)
	    {
	      extern double freq_bin_separation_MHz;

	      if(excluded[point])
		{
		  this->data[point] = NAN;
		}
	      else if(true)
		{
		  double delay_time_nseconds = -delta_phase_cycles[point] * (1000.0 / freq_bin_separation_MHz);
		  
		  this->data[point] = delay_time_nseconds;
		}
	    }

	  
#ifdef NOTDEF
	  for(int point=0; point<this->num_points; point++)
	    {
	      extern double freq_bin_separation_MHz;
	      if(excluded[point])
		{
		  this->data[point] = NAN;
		}
	      else if(true)
		{
		  double delta = phase_cycles[point] - phase_cycles[point-1];
		  while(delta>0.5)
		    delta -= 1.0;
		  while(delta<-0.5)
		    delta += 1.0;
		  double delay_time_nseconds = -delta * (1000.0 / freq_bin_separation_MHz);
		  
		  this->data[point] = delay_time_nseconds;
		}
	      else
		{
		  double last_delta = 0.0;
		  int p;
		  for(p=0; p<20; p++)
		    {
		      if(point+p>=this->num_points || point-1-p<0)
			break;
		      
		      double delta = phase_cycles[point+p] - phase_cycles[point-1-p];
		      
		      while(delta>0.5)
			delta -= 1.0;
		      while(delta<-0.5)
			delta += 1.0;
		      
		      if(abs(delta)>0.3)
			{
			  if(p<2)
			    {
			      last_delta = NAN;
			      p = 1;
			    }
			  break;
			}
		      last_delta = delta;
		    }
		  double delta = last_delta / (2*p-1);
		  
		  double delay_time_nseconds = -delta * (1000.0 / freq_bin_separation_MHz);
		  
		  this->data[point] = delay_time_nseconds;
		}
	    }
#endif
	  
#ifdef NOTDEF
	  for(int point=0; point<this->num_points; point++)
	    {
	      this->data[point] = excluded[point] ? 0 : phase_cycles[point];
	    }
#endif


	}
      
#ifdef NOTDEF
      if(reset_countdown==reset_countdown_end-1)
	{
	  extern double sampling_rate_MHz;
	  extern double freq_bin_separation_MHz;
	  extern int    adc_samples_per_clock;
	  
	  // Last step before locking changes to be slow changes
	  // Delay time is the cycles divided by the frequency offset
	  double delay_time_useconds = -slope_average / freq_bin_separation_MHz;
	  double samples_delay = delay_time_useconds * sampling_rate_MHz;
	  int clocks_delay = floor(samples_delay / adc_samples_per_clock);



	  printf("late delay was %d, slope_average was %f\n", pfb_spectrum->late_delay, slope_average);

	  slope_average += clocks_delay * adc_samples_per_clock / sampling_rate_MHz * freq_bin_separation_MHz;
	  pfb_spectrum->late_delay = clocks_delay;

	  printf("late delay now %d, slope_average now %f\n", pfb_spectrum->late_delay, slope_average);
	}

      //pfb_spectrum->late_delay = 15;
#endif
    }
}


template<typename T> void xfer_data<T>::find_ymin_ymax(double& ymin, double& ymax)
{
  if(source_type==TYPE_Z1_TRANSFER_MAG || source_type==TYPE_Z2_TRANSFER_MAG)
    {
      ymin = -110.0 / y_scale;
      ymax = -10.0  / y_scale;
    }
  else if(source_type==TYPE_Z1_TRANSFER_GROUP_DELAY || source_type==TYPE_Z2_TRANSFER_GROUP_DELAY)
    {
      ymin = default_min_group_delay;
      ymax = default_max_group_delay;
    }
  else
    {
      ymin = default_min_phase;
      ymax = default_max_phase;
    }

  //printf("xfer_data<T>::find_ymin_ymax() returns %f, %f.\n", ymin, ymax);
}


#endif
